# coding=utf-8

from django.utils import timezone
from django.db import models
from django.contrib.auth.models import User
from api.models import Person
from api.tool.image_utils import get_full_path
# Enumeration is deprecated, reference: http://git.gengmei.cc/backend/gm-types
# from rpc.tool.enumeration import Enumeration
from gm_types.gaia import (
    FEEDBACK_RELATED_OBJECT_TYPE, FEEDBACK_MESSAGE_TYPE,
    ORDER_REASON_FIRST_CATEGORIES, ORDER_REASON_SECOND_CATEGORIES, )
from gm_upload import IMG_TYPE, ImgUrlField


# FEEDBACK_RELATED_OBJECT_TYPE = Enumeration([
#    ('0', 'ORDER', u'订单相关'),
# ])
#
# FEEDBACK_MESSAGE_TYPE = Enumeration([
#    ('0', 'TXT', u'文字'),
#    ('1', 'IMAGE', u'图片'),
# ])
#
# 目前数据库字段类型是 tinyint (-128 - 127)
# 由于前端页面一二级类别是一个选择框，请确保一二级类别的值不要重复，否则前端区分不出来
# ORDER_REASON_FIRST_CATEGORIES = Enumeration([
#    (127, 'R100', u'退款/返现慢'),
#    (126, 'R200', u'到院多收费'),
#    (125, 'R300', u'对美购不认可'),
#    (124, 'R400', u'接待问题'),
#    (123, 'R500', u'医院环境差'),
#    (122, 'R600', u'药品问题'),
#    (121, 'R700', u'负面案例'),
# ])
#
# ORDER_REASON_SECOND_CATEGORIES = Enumeration([
#    (1, 'R101', u'退款慢'),
#    (2, 'R102', u'返现慢'),
#    (3, 'R201', u'收注册费'),
#    (4, 'R202', u'临时加价'),
#    (5, 'R301', u'价格不认可'),
#    (6, 'R302', u'活动到期'),
#    (7, 'R303', u'一直没货'),
#    (8, 'R304', u'没有这个活动'),
#    (9, 'R401', u'只能在特定时间接待用户'),
#    (10, 'R402', u'一直联系不上医生/医院'),
#    (11, 'R403', u'临时更换别的医生/护士'),
#    (12, 'R404', u'接待态度不好'),
#    (13, 'R405', u'过度开发，引起反感'),
#    (14, 'R406', u'医生爽约'),
#    (15, 'R501', u'太远、太小、破旧'),
#    (16, 'R601', u'不是正品'),
#    (17, 'R701', u'效果不好'),
#])


class FeedbackCategory(models.Model):
    class Meta:
        verbose_name = u'反馈类型'
        verbose_name_plural = u'反馈类型'
        app_label = 'api'

    title = models.CharField(max_length=64, verbose_name=u"反馈类型标题")
    description = models.CharField(max_length=300, default=u'', verbose_name=u"反馈类型描述")
    position = models.IntegerField(default=1, verbose_name=u"展示顺序", help_text=u"小的排在前，大的排在后")
    enabled = models.BooleanField(verbose_name=u'是否展示此反馈类型', default=False, blank=True)
    related_type = models.CharField(max_length=1, default=None, choices=FEEDBACK_RELATED_OBJECT_TYPE)
    icon = ImgUrlField(img_type=IMG_TYPE.FEEDBACKCATEGORY, max_length=120, verbose_name='图标', blank=True, default='')


class Feedback(models.Model):
    class Meta:
        verbose_name = u'用户反馈'
        verbose_name_plural = u'用户反馈'
        app_label = 'api'

    user = models.ForeignKey(User, related_name="feedback_user", verbose_name=u"反馈用户")
    content = models.CharField(max_length=300, default=u'', verbose_name=u"内容")
    phone = models.CharField(max_length=20, default=u'', blank=True, null=True, verbose_name=u"电话号码")
    qq = models.CharField(max_length=20, default=u'', blank=True, null=True, verbose_name=u"QQ号码")
    is_handled = models.BooleanField(default=False, verbose_name=u'是否已经处理')
    has_more_asks = models.BooleanField(default=False, verbose_name=u'是否有追问')
    comments = models.CharField(max_length=200, null=True, blank=True, default='', verbose_name=u'备注')
    created_time = models.DateTimeField(verbose_name=u'提交时间', default=timezone.now)
    related_id = models.CharField(max_length=32, null=True, verbose_name='相关联的object id', default=None)
    who = models.ForeignKey(User, verbose_name=u"谁最后更新了这条反馈", default=None, null=True)
    last_modified = models.DateTimeField(auto_now=True, verbose_name='最后更新时间')
    feedbackcategory = models.ForeignKey(FeedbackCategory, default=None, null=True)
    order_reason_first_category = models.SmallIntegerField(verbose_name=u'订单反馈原因一级分类',
                                                           choices=ORDER_REASON_FIRST_CATEGORIES)
    order_reason_second_category = models.SmallIntegerField(verbose_name=u'订单反馈原因二级分类',
                                                            choices=ORDER_REASON_SECOND_CATEGORIES)


class FeedbackImage(models.Model):
    class Meta:
        verbose_name = u'用户反馈上传的图片'
        verbose_name_plural = u'用户反馈上传的图片'
        app_label = 'api'

    feedback = models.ForeignKey(Feedback, related_name='images')
    image_url = ImgUrlField(img_type=IMG_TYPE.FEEDBACKIMAGE, max_length=300, verbose_name=u'图片地址')

    def __unicode__(self):
        return u'<%s><FeedbackImage:%s><Image_url: %s>' % (self.id, self.feedback.id, self.image_url)

    @property
    def image_link(self):
        return get_full_path(self.image_url)


class FeedbackMessage(models.Model):
    class Meta:
        verbose_name = u'用户反馈消息'
        verbose_name_plural = u'用户反馈消息'
        app_label = 'api'

    feedback = models.ForeignKey(Feedback, verbose_name=u'反馈')
    person = models.ForeignKey(Person, verbose_name=u'用户', null=True, default=None)
    servant = models.ForeignKey(
        Person, verbose_name=u'客服',
        null=True, default=None,
        related_name='feedback_message_replied'
    )
    message = models.TextField(null=False, blank=False, verbose_name=u'消息内容')
    message_type = models.CharField(
        max_length=1,
        default=FEEDBACK_MESSAGE_TYPE.TXT,
        choices=FEEDBACK_MESSAGE_TYPE
    )
    created_time = models.DateTimeField(verbose_name=u'反馈时间', auto_now=True)
