# -*- coding: UTF-8 -*-
from __future__ import absolute_import

from django.db.models import Q, Count, When, Case, IntegerField
from gm_dataquery.dataquery import DataBuilder, DataSQLQuery
from gm_dataquery.db import DB

from django.contrib.auth.models import User
from api.models import Person
from social.models import UserFollow
from talos.models.topic.column import Columnist
from services.talos_service import (topic_count, diary_count)
from gm_types.gaia import MEMBERSHIP_LEVEL
from rpc.tool.log_tool import user_logger, logging_exception


class UserDB(DataBuilder):
    def getval_topic_num(self, obj):
        return topic_count(obj.id),

    def getval_diary_num(self, obj):
        return diary_count(obj.id)

    def getval_order_num(self, obj):
        return obj.order_set.count()

    def getval_cellphone(self, obj):
        return obj.person.phone

    def getval_blacklist(self, obj):
        if obj.userblacklist_set.count() == 0:
            return False
        else:
            return True

    def getval_membership_level(self, obj):
        '''
            返回用户等级
        '''
        if obj.userextra.membership_level == '1':
            return u'是'
        else:
            return u'否'

    def getval_bind(self, obj, default=''):
        if obj.person.phone is None or '':  # 判断是否能解绑
            return '绑定'
        else:
            return '解绑'

    def getval_sum_fans(self, obj):
        return UserFollow.objects.filter(follow_id=obj.id, bond=1).count()

    def getval_basic_fans(self, obj):
        return UserFollow.objects.filter(follow_id=obj.id, is_virtual_fan=False, bond=1).count()

    def getval_virtual_fans(self, obj):
        return UserFollow.objects.filter(follow_id=obj.id, is_virtual_fan=True, bond=1).count()

    def getval_add_fans(self, obj):
        return ''


@DB
class UserDQ(DataSQLQuery):
    model = User
    data_model = UserDB
    distinct = False


    def filter_membership_level(self, srch_key, srch_val, regex=False):
        if srch_val == '':
            return Q()
        q = Q(userextra__membership_level='1')
        if srch_val == '1':
            return q
        else:
            return ~q


    def filter_blacklist(self, srch_key, srch_val, regex=False):
        if srch_val == '':
            return Q()
        q = Q(userblacklist__isnull=False)
        if srch_val == '1':
            return q
        else:
            return ~q

    def filter_cellphone(self, srch_key, srch_val, regex=False):
        return Q(person__phone=srch_val)

    def order_sum_fans(self, qs, field):
        qs1 = qs.filter(userfollow__isnull=True).annotate(num=Count(u'userfollow'))
        qs2 = qs.annotate(num=Count(Case(
                                When(userfollow__bond=1, then=1),
                                output_field=IntegerField())))
        qs = qs1 | qs2
        return qs, u'num'

    def filter_columnist(self, srch_key, srch_val, regex=False):
        columnist_id = list(Columnist.objects.filter(is_online=True).values_list('user_id', flat=True))
        q = Q(id__in=columnist_id)
        return q if srch_val == '1' else ~q

    def filter_puppet_and_kol(self, srch_key, srch_val, regex=False):
        #所长账号 && 第一批kol账号

        kol_ids = [22, 29888501, 29853500, 29853624, 29853108,
                   29844692, 29841915, 29841721, 29824464, 21409100]
        puppet_ids = Person.objects.filter(is_puppet=True).values_list('user_id', flat=True)
        if srch_val == '1':
            kol_ids.extend(list(puppet_ids))
            return Q(id__in=kol_ids)
        return Q(id__in=list(puppet_ids))


@DB
class PersonDQ(DataSQLQuery):
    model = Person
    data_model = DataBuilder
