# -*- coding: utf-8 -*-
from django.db import models
from datetime import datetime
from gm_types.consultation import COUNSELLOR_TYPE, COUNSELLOR_STATUS, CONSULTANT_APPROVE_STATUS,\
        COUNSELLOR_GRADE, COUNSELLOR_SERVICE_TYPE, COUNSELLOR_SERVICE_MODE
import uuid
from django.utils import timezone
from django.conf import settings


class Counsellor(models.Model):
    class Meta:
        verbose_name = u'面诊师'
        db_table = 'consultation_counsellor'
        app_label = "diagnosis"

    objects = models.Manager().db_manager(using=settings.DIAGNOSIS_SLAVE_DB_NAME)
    id = models.UUIDField(primary_key=True, editable=False, default=uuid.uuid1, verbose_name=u'面诊师id')
    create_time = models.DateTimeField(verbose_name="创建时间", default=timezone.now)
    first_consultation_time = models.DateTimeField(verbose_name=u'首次启用面诊时间', null=True)
    is_banned = models.BooleanField(verbose_name='是否被禁止面诊', default=False)
    banned_time = models.DateTimeField(verbose_name=u'封禁时间', null=True)
    type = models.IntegerField(
        choices=COUNSELLOR_TYPE, default=COUNSELLOR_TYPE.DOCTOR, verbose_name=u'面诊师类别 1是医生，2是面诊师'
    )
    merchant_id = models.BigIntegerField(u'所属的商户Id')
    user_id = models.IntegerField(verbose_name=u'关联的用户Id，唯一', unique=True)
    doctor_id = models.CharField(max_length=100, null=True, verbose_name=u'关联的医生id，可能为null')
    consultant_id = models.CharField(max_length=100, null=True, verbose_name=u'关联的面诊师id，可能为null')
    status = models.IntegerField(
        choices=COUNSELLOR_STATUS, default=COUNSELLOR_STATUS.OFFLINE, verbose_name=u'面诊状态，0离线，1在线，2面诊中'
    )
    is_assistant = models.BooleanField(verbose_name='当前进行面诊的是否医生助理', default=False)
    good_at = models.TextField(verbose_name=u"当前进行面诊选择的擅长项目，JSON字符串，存的对应项目的id")
    record_id = models.UUIDField(verbose_name=u'当前进行面诊连线的id', null=True)
    facevideo_rec_rank = models.IntegerField(verbose_name=u'推荐位人工排序优先级', default=0)
    facevideo_rank = models.IntegerField(verbose_name=u'视频面诊人工排序优先级', default=0)


class Consultant(models.Model):
    class Meta:
        verbose_name = u'面诊师表'
        app_label = 'consultation'
        db_table = 'consultation_consultant'

    objects = models.Manager().db_manager(using=settings.DIAGNOSIS_SLAVE_DB_NAME)
    id = models.BigIntegerField(u'面诊师ID', primary_key=True)
    create_time = models.DateTimeField(verbose_name="创建时间", default=timezone.now)
    user_id = models.IntegerField(verbose_name=u'关联的用户Id')
    doctor_id = models.CharField(max_length=100, null=True, verbose_name=u'机构操作者医生id', db_index=True)
    hospital_id = models.CharField(max_length=100, null=True, verbose_name=u'操作机构医院id', db_index=True)
    business_partener_id = models.IntegerField(verbose_name=u'负责此面诊师的商务同事的UserId', null=True, db_index=True)
    name = models.CharField(max_length=100, null=True, verbose_name=u'注册面诊师身份证名字')
    nickname = models.CharField(max_length=100, null=True, verbose_name=u'注册面诊师昵称')
    country = models.CharField(max_length=100, blank=True, verbose_name=u'国家')
    area = models.CharField(max_length=100, blank=True, verbose_name=u'地区')
    employment_time = models.DateTimeField(verbose_name="从业开始时间", null=True)
    good_at = models.TextField(verbose_name=u"当前进行面诊选择的擅长项目，JSON字符串，存的对应项目的id, 例如'[0,1]'", null=True)
    introduction = models.TextField(verbose_name=u"面诊师简介", null=True)
    approve_status = models.IntegerField(verbose_name=u'是否认证，审核通过则为已认证', choices=CONSULTANT_APPROVE_STATUS,
                                         default=CONSULTANT_APPROVE_STATUS.UNAPPROVE)
    is_online = models.BooleanField(verbose_name='是否上线', default=True)


class AbstractBaseModel(models.Model):
    id = models.AutoField(auto_created=True, primary_key=True, serialize=False, verbose_name='ID')
    created_time = models.DateTimeField(auto_now_add=True, help_text='创建时间')
    update_time = models.DateTimeField(auto_now=True, help_text='更新时间')

    class Meta:
        abstract = True


class CounsellorSmartRank(AbstractBaseModel):
    '''
    # http://wiki.wanmeizhensuo.com/pages/viewpage.action?pageId=36545052
    '''
    class Meta:
        verbose_name = u'面诊师smart_rank表'
        app_label = 'consultation'
        db_table = 'consultation_counsellor_smart_rank'

    objects = models.Manager().db_manager(using=settings.DIAGNOSIS_SLAVE_DB_NAME)
    counsellor = models.ForeignKey(Counsellor, on_delete=models.CASCADE, verbose_name='面诊顾问')
    type = models.IntegerField(
        choices=COUNSELLOR_TYPE, default=COUNSELLOR_TYPE.DOCTOR,
        verbose_name=u'面诊顾问类别 1是医生，2是面诊师'
    )
    doctor_id = models.CharField(max_length=100, null=True,
                                 verbose_name=u'关联的医生id，可能为null', db_index=True)
    consultant_id = models.BigIntegerField(verbose_name="面诊师ID", null=True)

    recent_30_min_connected_rate = models.FloatField(default=0, verbose_name='最近30分钟的值, a0值')
    recent_1_hour_connected_rate = models.FloatField(default=0, verbose_name='最近1小时接通率, a1值')
    today_connected_rate = models.FloatField(default=0, verbose_name='当天接通率, a2值')
    avg_7_days_connected_rate = models.FloatField(default=0, verbose_name='7天平均接通率, a3值')
    avg_30_days_connected_rate = models.FloatField(default=0, verbose_name='30天平均接通率, a4值')

    avg_3_days_valid_rate = models.FloatField(default=0, verbose_name='3天平均有效率, b1值')
    avg_7_days_valid_rate = models.FloatField(default=0, verbose_name='7天平均有效率, b2值')
    avg_30_days_valid_rate = models.FloatField(default=0, verbose_name='30天平均有效率, b3值')
    avg_90_days_valid_rate = models.FloatField(default=0, verbose_name='90天平均有效率, b4值')
    consult_connects = models.IntegerField(default=0, verbose_name='当前面诊顾问总历史接通人次')
    consult_visits = models.IntegerField(default=0, verbose_name='当前面诊顾问有效历史接通人次')
    consult_records = models.IntegerField(default=0, verbose_name='当前面诊历史顾问总连线记录')
    recent_3_hour_connected_rate = models.FloatField(default=0, verbose_name='最近3小时接通率, a值')


class GradeSmartRank(AbstractBaseModel):
    '''
    获取全平台相同等级的统计数据
    接通率
    不分医生和面诊师，某一个等级的数据包含这个等级的医生个面诊师的全部数据
    每隔5分钟统计一次数据
    '''
    class Meta:
        verbose_name = u'面诊师、医生等级统计数据'
        app_label = 'consultation'
        db_table = 'consultation_grade_smart_rank'

    objects = models.Manager().db_manager(using=settings.DIAGNOSIS_SLAVE_DB_NAME)
    grade = models.IntegerField(default=0, verbose_name='等级')
    recent_30_min_connected_rate = models.FloatField(default=0, verbose_name='最近30分钟的值, a0值')
    recent_1_hour_connected_rate = models.FloatField(default=0, verbose_name='最近1小时接通率, a1值')
    recent_3_hour_connected_rate = models.FloatField(default=0, verbose_name='最近3小时接通率, a3值')


class ConsultationCtDtStrategyFactor(models.Model):
    '''
    # http://wiki.wanmeizhensuo.com/pages/viewpage.action?pageId=36545052
    '''
    class Meta:
        verbose_name = u'面诊师smart_rank数据统计表'
        app_label = 'consultation'
        db_table = 'sl_consultation_ct_dt_strategyfactor_d'

    objects = models.Manager().db_manager(using=settings.DIAGNOSIS_SLAVE_DB_NAME)
    id = models.IntegerField(verbose_name=u'策略统计id')
    counsellor_id = models.CharField(max_length=100, verbose_name='面诊顾问ID')
    counsellor_grade = models.IntegerField(verbose_name='面诊顾问等级')
    last_1v1_connect_rate_1d = models.FloatField(verbose_name='最近1日1V1面诊接通率')
    last_1v1_connect_rate_7d = models.FloatField(verbose_name='最近7日1V1面诊接通率')
    last_1v1_connect_rate_30d = models.FloatField(verbose_name='最近30日1V1面诊接通率')
    last_1v1_grant_connect_rate_1d = models.FloatField(verbose_name='最近1日同级别顾问1V1面诊接通率')
    last_1v1_grant_connect_rate_7d = models.FloatField(verbose_name='最近7日同级别顾问1V1面诊接通率')
    last_1v1_grant_connect_rate_30d = models.FloatField(verbose_name='最近30日同级别顾问1V1面诊接通率')
    last_1v1_all_connect_rate_30d = models.FloatField(verbose_name='最近30日所有顾问1V1面诊接通率')
    last_1v1_connect_EFFECTIVE_rate_1d = models.FloatField(verbose_name='最近1日1V1面诊有效接通率')
    last_1v1_connect_EFFECTIVE_rate_3d = models.FloatField(verbose_name='最近3日1V1面诊有效接通率')
    last_1v1_connect_EFFECTIVE_rate_7d = models.FloatField(verbose_name='最近7日1V1面诊有效接通率')
    last_1v1_connect_EFFECTIVE_rate_30d = models.FloatField(verbose_name='最近30日1V1面诊有效接通率')
    last_1v1_grant_connect_EFFECTIVE_rate_1d = models.FloatField(verbose_name='最近1日同级别顾问1V1面诊有效接通率')
    last_1v1_grant_connect_EFFECTIVE_rate_3d = models.FloatField(verbose_name='最近3日同级别顾问1V1面诊有效接通率')
    last_1v1_grant_connect_EFFECTIVE_rate_7d = models.FloatField(verbose_name='最近7日同级别顾问1V1面诊有效接通率')
    last_1v1_grant_connect_EFFECTIVE_rate_30d = models.FloatField(verbose_name='最近30日同级别顾问1V1面诊有效接通率')
    last_1v1_all_connect_EFFECTIVE_rate_30d = models.FloatField(verbose_name='最近30日所有顾问1V1面诊有效接通率')
    last_card_feed_ctr_1d = models.FloatField(verbose_name='最近1日feed流卡片点击率')
    last_card_feed_ctr_30d = models.FloatField(verbose_name='最近30日feed流卡片点击率')
    last_grade_card_feed_ctr_1d = models.FloatField(verbose_name='最近1日同级别顾问feed流卡片点击率')
    last_grade_card_feed_ctr_30d = models.FloatField(verbose_name='最近30日同级别顾问feed流卡片点击率')
    last_5_card_feed_ctr_30d = models.FloatField(verbose_name='最近30日feed流前5位卡片点击率均值')
    last_card_rate_1d = models.FloatField(verbose_name='最近1日1V1面诊发起呼叫率')
    last_card_rate_30d = models.FloatField(verbose_name='最近30日1V1面诊发起呼叫率')
    last_grade_card_rate_1d = models.FloatField(verbose_name='最近1日同级别顾问1V1面诊发起呼叫率')
    last_grade_card_rate_30d = models.FloatField(verbose_name='最近30日同级别顾问1V1面诊发起呼叫率')
    last_all_card_rate_30d = models.FloatField(verbose_name='最近30日所有顾问1V1面诊发起呼叫率')
    create_day = models.DateTimeField(verbose_name="创建时间", default=timezone.now      )


class CounsellorServiceSPU(models.Model):

    class Meta:
        db_table = "consultation_counsellor_spu"
        app_label = "consultation"
        verbose_name = "面诊顾问服务计费SPU"

    objects = models.Manager().db_manager(using=settings.DIAGNOSIS_SLAVE_DB_NAME)
    counsellor = models.ForeignKey(Counsellor, on_delete=models.CASCADE, db_constraint=False, null=True, verbose_name='面诊顾问')
    counsellor_type = models.IntegerField(
        choices=COUNSELLOR_TYPE, default=COUNSELLOR_TYPE.DOCTOR, verbose_name=u'面诊顾问类别 1是医生，2是面诊师'
    )
    counsellor_grade = models.IntegerField(
        choices=COUNSELLOR_GRADE, default=COUNSELLOR_GRADE.ONE, verbose_name=u'面诊顾问等级，1:一级，2:二级，3:三级，级别越高越厉害'
    )
    type = models.IntegerField(verbose_name="面诊顾问服务计费类型", choices=COUNSELLOR_SERVICE_TYPE, default=COUNSELLOR_SERVICE_TYPE.ONE_TO_ONE)
    mode = models.IntegerField(verbose_name="面诊顾问服务计费方式", choices=COUNSELLOR_SERVICE_MODE, default=COUNSELLOR_SERVICE_MODE.MIANDAN)

    unit_price = models.IntegerField(verbose_name="面诊服务单价(分)", default=0)
    created_time = models.DateTimeField(default=datetime.now, help_text='创建时间')

