# -*- coding: UTF-8 -*-

from datetime import datetime

from django.db.models import Q
from gm_types.gaia import (
    SLIDE_USER_TYPE, SLIDE_PAYMENT_TYPE,
    WORLD_AREA_TYPE,
    FEED_OPERATE_VERSION,
    TAB_TYPES_NEW,
    FEED_SERVICE_TYPE)
from api.tool.user_tool import get_user_and_payment_info
from api.tool.city_tool import is_oversea
from api.tool.geo_tool import get_city_by_id


def base_query():
    now = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    base_query = Q(is_online=True, start_time__lte=now, end_time__gte=now)
    return base_query


def user_info_query(user, platform, device_id):
    user_info = get_user_and_payment_info(user, platform, device_id)
    is_new_user, have_payment = user_info.get('is_new_user'), user_info.get('have_payment')
    user_query = Q()

    if is_new_user:
        user_query &= Q(user_type__in=[SLIDE_USER_TYPE.ALL_USER, SLIDE_USER_TYPE.NEW_USER])
    else:
        user_query &= Q(user_type__in=[SLIDE_USER_TYPE.ALL_USER, SLIDE_USER_TYPE.OLD_USER])

    if have_payment:
        user_query &= Q(payment_type__in=[SLIDE_PAYMENT_TYPE.ALL_PAYMENT, SLIDE_PAYMENT_TYPE.HAVE_PAYMENT])
    else:
        user_query &= Q(payment_type__in=[SLIDE_PAYMENT_TYPE.ALL_PAYMENT, SLIDE_PAYMENT_TYPE.NO_PAYMENT])

    return user_query


def city_query(city_id):
    """
    会把城市所属的省份下的其他城市数据带出
    :param city_id:
    :return:
    """
    city_query = Q(cities__isnull=True, regions__isnull=True)
    if not city_id or city_id in WORLD_AREA_TYPE or is_oversea([city_id])[0]:
        pass
    else:
        city_query |= Q(cities=city_id)
        region = None
        city = get_city_by_id(city_id)
        if city and city.province:
            try:
                region = city.province.region
            except:
                pass
        if region:
            city_query |= Q(regions=region)
    return city_query


def city_query_by_city_id(city_id):
    """
    只针对当前城市
    :param city_id:
    :return:
    """
    city_query = Q(cities__isnull=True, regions__isnull=True)
    if not city_id or city_id in WORLD_AREA_TYPE or is_oversea([city_id])[0]:
        pass
    else:
        city_query |= Q(cities=city_id)
    return city_query


def get_query_by_usertype_and_location(user, platform=None, device_id=None, city_id=None):
    """
    将以前的query逻辑代码重构，未改变任何逻辑
    :param user:
    :param platform:
    :param device_id:
    :param city_id:
    :return:
    """
    return base_query() & user_info_query(user, platform, device_id) & city_query(city_id)


def get_feed_query(user, platform=None, device_id=None, city_id=None, tab_id=None, location=None,
                   version=FEED_OPERATE_VERSION.V1, relevance_tag=None, plastic_template=None, plastic_tab_type=None):
    feed_query = base_query() & user_info_query(user, platform, device_id) & city_query_by_city_id(city_id) & Q(
        tab_id=tab_id) & Q(location=location)

    if location == TAB_TYPES_NEW.TAB_AI_PLASTIC:
        feed_query &= Q(relevance_tag=relevance_tag) & Q(plastic_template=plastic_template) & Q(
            tab_type=plastic_tab_type)
    # 兼容广告新老版本version状态1，2均展示广告service_type类型2
    feed_query = feed_query & (Q(version=version) | Q(version__in=[1, 2], service_type=FEED_SERVICE_TYPE.ADV))

    return feed_query
