# -*- coding: utf-8 -*-

import json
from rpc.cache import servicehome_cache, filter_cache
from api.models.slide import Slide, SLIDE_POSITION, ButtonGadget, GADGET_PAGE_TYPE


class ServiceHomeLayout(object):
    """美购首页模块配置类

    用redis cache保存配置数据，结构为
    {
        'special': bool[True], # 专题
        'group': bool[True], # 分类
        'activity': bool[True], # 活动
        'recommend': bool[True], # 商品推荐列表
    }
    redis key 为{prefix}:layout
    """

    rkey = 'layout'  # redis key
    default_layout = {
        'special': True,
        'group': True,
        'activity': True,
        'recommend': True,
    }

    @classmethod
    def get(cls):
        try:
            layout_json = servicehome_cache.get(cls.rkey)
            if not layout_json:
                layout = {
                    'special': True,  # 专题
                    'group': True,  # 分类
                    'activity': True,  # 活动
                    'recommend': True,  # 商品推荐列表
                }
            else:
                layout = json.loads(layout_json)
            # banner是否显示, 新方案
            layout['special'] = Slide.get_visible(position=SLIDE_POSITION.SERVICE_HOME)
        except:
            layout = cls.default_layout
        return layout

    @classmethod
    def set(cls, layout_info):
        layout = cls.get()
        if not layout:
            layout = cls.default_layout

        for k, v in layout_info.items():
            if k in layout:
                layout[k] = layout_info[k]

        servicehome_cache.set(cls.rkey, json.dumps(layout))


class BasicServiceHomeFilter(object):
    """美购首页筛选管理类

    用redis cache保存配置数据，结构为
    {
        'floor': bool[True], # 底价
        'cashback': bool[True], # 分享返现
        'popular': bool[True], # 好评最多
    }
    redis key 为{prefix}:filter
    """

    rkey = 'filter'  # redis key
    default_filter = {
        'basic': {
            'floor': {'visible': True, 'title': '底价'},
            'cashback': {'visible': True, 'title': '分享返现'},
            'popular': {'visible': True, 'title': '好评最多'},
            'famous_doctor' :{'visible': True, 'title': '大牌名医'}
        },
    }

    @classmethod
    def get(cls):
        try:
            filter_json = filter_cache.get(cls.rkey)
            filter_value = json.loads(filter_json)
        except:
            filter_value = cls.default_filter
        return filter_value

    @classmethod
    def set(cls, new_filter_value):
        filter_value = cls.get()
        if not filter_value:
            filter_value = cls.default_filter

        for k, v in new_filter_value['basic'].items():
            basic = filter_value['basic']
            new_basic = new_filter_value['basic']
            if k in cls.default_filter['basic']:
                basic[k] = new_basic[k]

        filter_cache.set(cls.rkey, json.dumps(filter_value))


class BaseServiceHomeDeploy(object):
    """
    美购首页redis配置
    """
    RKEY_PREFIX = None
    DEFAULT_DEPLOY = None

    @classmethod
    def _gen_key(cls):
        return '{}:{}'.format('servicehome', cls.RKEY_PREFIX)

    @classmethod
    def set(cls, deploy_info):
        rkey = cls._gen_key()
        servicehome_cache.set(rkey, json.dumps(deploy_info))

    @classmethod
    def get(cls):
        rkey = cls._gen_key()
        try:
            res = servicehome_cache.get(rkey)
            if not res:
                res = cls.DEFAULT_DEPLOY
            else:
                res = json.loads(res)
            return res
        except:
            return cls.DEFAULT_DEPLOY


class SeckillDeploy(BaseServiceHomeDeploy):
    """
    秒杀模块配置
    """
    RKEY_PREFIX = 'seckill'
    DEFAULT_DEPLOY = {
        'is_show': False,
        'bg_img': '',
        'sckill_alert': '',
    }


class SceneButton(BaseServiceHomeDeploy):
    """
    场景按钮配置
    """
    RKEY_PREFIX = 'scenebutton'
    DEFAULT_DEPLOY = {
        'is_show': False,
        'is_img': False,
        'icon_img': '',
        'title_color': '',
        'dec_color': '',
        'button_list': [
            {
                'title': '',
                'dec': '',
                'img': '',
                'need_login': False,
                'url': '',
            },
            {
                'title': '',
                'dec': '',
                'img': '',
                'need_login': False,
                'url': '',
            },
            {
                'title': '',
                'dec': '',
                'img': '',
                'need_login': False,
                'url': '',
            },
            {
                'title': '',
                'dec': '',
                'img': '',
                'need_login': False,
                'url': '',
            },
            {
                'title': '',
                'dec': '',
                'img': '',
                'need_login': False,
                'url': '',
            },
            {
                'title': '',
                'dec': '',
                'img': '',
                'need_login': False,
                'url': '',
            },
            {
                'title': '',
                'dec': '',
                'img': '',
                'need_login': False,
                'url': '',
            },
            {
                'title': '',
                'dec': '',
                'img': '',
                'need_login': False,
                'url': '',
            },
        ]
    }


class SideButton(BaseServiceHomeDeploy):
    """首页侧收按钮"""
    RKEY_PREFIX = 'side'
    DEFAULT_DEPLOY = {
        'old_is_show': False,
        'old_img': '',
        'old_url': '',
        'new_is_show': False,
        'new_img': '',
        'new_url': '',
    }
