# -*- coding: UTF-8 -*-

from django.conf import settings
from django.db.models import Q
from gm_dataquery.dataquery import DataBuilder, DataSQLQuery
from gm_dataquery.db import DB
from gm_types.gaia import USER_CLASSIFY

from api import Person
from talos.models.topic import TopicReply, ReplyHeadline
from statistic.models import ClassifyUser
from . import TalosDataSQLQuery


class TopicReplyDB(DataBuilder):
    def getval_user__last_name(self, obj):
        return obj.user.nickname

    def getval_problem_name(self, obj):
        name = obj.problem.title if obj.problem.title else obj.problem.ask
        if name:
            return name
        else:
            return u'查看详情'

    def getval_problem_info(self, obj):
        id = obj.problem.id
        name = obj.problem.title if obj.problem.title else obj.problem.ask or obj.problem.answer[:20]
        return {'id': id, 'text': name}

    def getval_diary_info(self, obj):
        id = obj.diary_id
        text = obj.diary.title
        return {'id': id, 'text': text}

    def getval_content(self, obj):
        return obj.content[:20] + '...' if len(obj.content) > 20 else obj.content

    def getval_doctor_info(self, obj):
        if getattr(obj, 'doctor', None):
            return {'id': obj.doctor.id, 'name': obj.doctor.name}
        else:
            return ''

    def getval_last_reply_time(self, obj):
        return str(obj.problem.topicreply_set.all().order_by("-id")[0].reply_date)

    def getval_is_headline(self, obj):
        return ReplyHeadline.objects.filter(reply=obj).exists()

    def getval_vote_num(self, obj):
        return obj.topicreplyvote_set.count() or 0

    def getval_replied_user_id(self, obj):
        try:
            tr = TopicReply.objects.get(id=obj.commented_reply_id)
            return tr.user_id
        except TopicReply.DoesNotExist:
            return ''


@DB
class TopicReplyDQ(TalosDataSQLQuery):
        model = TopicReply
        data_model = TopicReplyDB

        def filter_is_real(self, srch_key, srch_val, regex=False):
            if srch_val == USER_CLASSIFY.PUPPET:
                user_ids = Person.objects.using(settings.SLAVE_DB_NAME).filter(
                    is_puppet=True
                ).values_list('user_id', flat=True)
                return Q(user_id__in=list(user_ids))

            elif srch_val == USER_CLASSIFY.NORMAL:
                puppet_user_ids = list(Person.objects.using(settings.SLAVE_DB_NAME).filter(
                    is_puppet=True
                ).values_list('user_id', flat=True))
                mode_user_ids = list(ClassifyUser.objects.using(
                    settings.SLAVE_DB_NAME).filter(classify=USER_CLASSIFY.MODEL).values_list('user_id', flat=True))
                puppet_user_ids.extend(mode_user_ids)
                return ~Q(user_id__in=puppet_user_ids)

            elif srch_val in USER_CLASSIFY.MODEL:
                mode_user_ids = list(ClassifyUser.objects.using(
                    settings.SLAVE_DB_NAME).filter(classify=USER_CLASSIFY.MODEL).values_list('user_id', flat=True))
                return Q(user_id__in=mode_user_ids)

        def filter_problem_name(self, srch_key, srch_val, regex=False):
            return Q(problem__title__contains=srch_val) | Q(problem__ask__contains=srch_val)

        def filter_reply_date(self, srch_key, srch_val, regex=False):
            return self._qry_time_range(srch_key, srch_val, regex)
