#!/usr/bin/env python
# coding=utf-8
from __future__ import unicode_literals, absolute_import

from django.db import models
from gm_types.doctor import DOCTOR_MESSAGE_TYPE, DOCTOR_ORDER_MESSAGE_TYPE

from hippo.models import Doctor
from hippo.models import Merchant


MESSAGE_TYPE_CONTENT = {
    DOCTOR_MESSAGE_TYPE.BUDAN: "【补单通知】您本月还有{}元未补佣金，请于月底前补齐佣金",
    DOCTOR_MESSAGE_TYPE.STATEMENT_CONFIRM: "【对账单消息】您{}月的对账单已审核通过。请尽快确认",
    DOCTOR_MESSAGE_TYPE.STATEMENT_SETTLE: "【对账单消息】您的{}月对账单已结算完成。点击查看",
    DOCTOR_MESSAGE_TYPE.STATEMENT_FAIL: "【对账单消息】您{}月对账单打款失败，失败原因：{}。请修改财务信息",
    DOCTOR_MESSAGE_TYPE.SERVICE_ONLINE: "【美购消息】您提交的美购“{}”已审核上线。点击查看",
    DOCTOR_MESSAGE_TYPE.SERVICE_REFUSE: "【美购消息】您提交的美购“{}”审核被驳回，原因：{}。请重新修改",
    DOCTOR_MESSAGE_TYPE.SERVICE_OFFLINE: "【美购消息】您提交的美购“{}”已审核下线。点击查看",
    DOCTOR_MESSAGE_TYPE.SERVICE_OFFLINE_REFUSE: "【美购消息】您提交的美购“{}”下线审核被驳回，原因：{}。请重新修改",
    DOCTOR_MESSAGE_TYPE.SERVICE_EXPIRE: "【美购消息】您的美购“{}”将于3天后到期，点击查看",
    DOCTOR_MESSAGE_TYPE.ACTIVITY_SUCCESS: "【活动消息】您已成功报名“{}”活动，点击查看",
    DOCTOR_MESSAGE_TYPE.ACTIVITY_FAIL: "【活动消息】您报名的“{}”活动被驳回，原因：{}。点击查看",
    DOCTOR_MESSAGE_TYPE.DOCTOR_SUCCESS: "【审核反馈】您提交的个人资料已经审核通过。点击查看",
    DOCTOR_MESSAGE_TYPE.DOCTOR_REFUSE: "【审核反馈】您提交的个人资料审核被驳回，原因：{}。请重新修改",
    DOCTOR_MESSAGE_TYPE.FINANCE_SUCCESS: "【审核反馈】您提交的个人结算信息资料已经审核通过。点击查看",
    DOCTOR_MESSAGE_TYPE.FINANCE_REFUSE: "【审核反馈】您提交的个人财务资料审核被驳回，原因：{}。请重新修改",
    DOCTOR_MESSAGE_TYPE.AD_ARREARAGE: "【商桥消息】您的更美商桥已欠费，为不影响您的正常使用，请及时充值",
    DOCTOR_MESSAGE_TYPE.AD_RECHANGE: "【商桥消息】您的更美商桥账号已成功充值{}元，余额{}元。",
    DOCTOR_MESSAGE_TYPE.AD_RECHANGE_FAIL: "【商桥消息】您提交的的更美商桥账号充值被驳回，原因：{}。点击查看",
    DOCTOR_MESSAGE_TYPE.DOWNSTREAM_REMINDER: "【商桥消息】您的更美账户已欠费超过31天，系统自动对您进行降流处罚。请及时充值",
    DOCTOR_MESSAGE_TYPE.DOWNSTREAM_REMINDER_CANCEL: "【商桥消息】您的系统降流处罚已取消。",
    DOCTOR_MESSAGE_TYPE.DOCTOR_EXAMINE_REJECT: "【医生消息】你提交的{}的医生资料审核被驳回了，原因：{}，点击查看。",
    DOCTOR_MESSAGE_TYPE.DOCTOR_EXAMINE_PASS: "【医生消息】你提交的{}的医生资料已经审核通过，点击查看。",
    DOCTOR_MESSAGE_TYPE.DOCTOR_CASHBACK_EXPIRE: "【商桥消息】您的账户{}{}请及时关注并使用，如有任何问题，请联系您的商务经理。",
    DOCTOR_MESSAGE_TYPE.COUNSELLOR_INFO_AUDIT_PASS: "【医生消息】你提交的{}的面诊师资料已经审核通过，点击查看。",
    DOCTOR_MESSAGE_TYPE.COUNSELLOR_INFO_AUDIT_REJECT: "【医生消息】你提交的{}的面诊师资料已被审核驳回，点击查看。",
    DOCTOR_MESSAGE_TYPE.CPT_THREE_DAY_REMIND: "【商桥消息】您购买的${}{}$即将上线，排期为{}（共{}天），您可以在上线前优化推广内容，使投放效果加倍。",
    DOCTOR_MESSAGE_TYPE.CPT_ONLINE_REMIND: "【商桥消息】您购买的${}{}$已上线，排期为{}（共{}天）",
}


class DoctorMessage(models.Model):
    class Meta:
        verbose_name = u'消息中心'
        app_label = 'hippo'
        db_table = 'hippo_doctormessage'

    type = models.IntegerField(verbose_name=u'消息类型', choices=DOCTOR_MESSAGE_TYPE)
    created_time = models.DateTimeField(verbose_name=u'加入消息中心的时间', auto_now_add=True)
    info_id = models.CharField(verbose_name=u'对应的消息的ID，如美购ID，公告ID', max_length=100, default='')
    doctor = models.ForeignKey(Doctor, verbose_name=u'推送的医生', null=True)
    # 目前消息是按医生推送，但是为了之后拓展到商户，记录商户ID
    merchant = models.ForeignKey(Merchant, verbose_name=u'医生对应的商户', null=True)
    content = models.CharField(verbose_name=u'消息内容', null=True, max_length=200)
    is_read = models.BooleanField(verbose_name=u'是否已读', default=False)

    @classmethod
    def add_message(cls, info, *message_info):
        """
        添加一个消息类型
        >>> DoctorMessage.add_message(info, *message_info)
        :param info: {'type'：1， 'info_id': '', 'doctor_id': '', 'merchant_id':''}
        :param message_info: content的参数
        :return: message
        """
        message_type = info['type']
        if len(message_info):
            content = MESSAGE_TYPE_CONTENT[message_type].format(*message_info)
        else:
            content = MESSAGE_TYPE_CONTENT[message_type]
        info['content'] = content
        message = cls.objects.create(**info)
        doctor = message.doctor if message.doctor else message.merchant.doctor
        try:
            from services.notify import notify
            notify('system/add', doctor_id=doctor.id, data={})
        except Exception as e:
            pass
        return message


class OrderMessage(models.Model):
    class Meta:
        verbose_name = u'订单退款单消息'
        app_label = 'hippo'
        db_table = 'hippo_ordermessage'

    type = models.IntegerField(verbose_name=u'消息类型', choices=DOCTOR_ORDER_MESSAGE_TYPE)
    info_id = models.CharField(verbose_name=u'对应的order_id或refund_id', max_length=100)
    doctor = models.ForeignKey(Doctor, verbose_name=u'推送的医生', null=True)
    # 目前消息是按医生推送，但是为了之后拓展到商户，记录商户ID
    merchant = models.ForeignKey(Merchant, verbose_name=u'医生对应的商户', null=True)
    created_time = models.DateTimeField(verbose_name=u'加入消息中心的时间', auto_now_add=True)
    is_push = models.BooleanField(verbose_name=u'是否已推送', default=False)
