#!/usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import unicode_literals, absolute_import, print_function

import json
import datetime

from django.db.models import Q

from gm_types.error import ERROR
from gm_types.gaia import (
    SKIN_DATA_SOURCE,
    MARK_ACTIVITY_TYPE,
)

from mark.services import (
    MarkActivityService,
    UserMarkLogService,
)

from rpc.decorators import bind
from rpc.tool.error_code import gen


@bind("api/daily_check/write_log")
def update_user_daily_check_log(device_id, user_id, skin_result_id, daily_check_id, others):
    """
    更新用户打卡日志
    :param device_id:
    :param user_id:
    :param skin_result_id:
    :param daily_check_id:
    :param others: 其他字段
    :return:
    """
    # 更新LOG表
    log_obj = UserMarkLogService.create(**{
        "user_id": user_id,
        "device_id": device_id,
        "skin_result_id": skin_result_id,
        "skin_data_source": others.get("skin_data_source", SKIN_DATA_SOURCE.YIXI),
        "mark_activity_id": daily_check_id,
    })

    return {
        "id": log_obj.id
    }


@bind("api/daily_check/get_user_daily_check_count")
def get_user_daily_check_effective_count(user_id, daily_check_id):
    """
    获取用户某活动内已参与的次数
    :param user_id:
    :param daily_check_id:
    :return:
    """
    result = {
        "activity_status": False,
        "daily_check_counts": 0,
        "camera_toast": "",
    }

    daily_check_obj = MarkActivityService.get_mark_activity_obj_by_id(daily_check_id)

    if not daily_check_obj:
        return result

    daily_check_counts = UserMarkLogService.get_user_today_mark_counts(
        user_id=user_id,
        mark_activity_id=daily_check_id
    )

    result.update({
        "activity_status": True,
        "daily_check_counts": daily_check_counts,
        "camera_toast": daily_check_obj.remind_desc,
    })

    return result


def convert_luck_data(daily_check_obj, mark_fortune_obj):
    """
    获取运势数据
    :param daily_check_obj:
    :param mark_fortune_obj:
    :return:
    """
    rel_tag_ids = MarkActivityService.get_mark_activity_rel_tags(daily_check_obj.id)
    _notes = mark_fortune_obj.notes
    fitting_and_dread = mark_fortune_obj.fitting_and_dread
    _fitting_and_dread_dic = fitting_and_dread and json.loads(fitting_and_dread) or {}

    result = {
        "major_title_solid": daily_check_obj.solid_title,
        "major_title_hollow": daily_check_obj.hollow_title,
        "minor_title": mark_fortune_obj.title,
        "notes": _notes and json.loads(_notes) or [],
        "positive": _fitting_and_dread_dic.get("fitting", []),
        "negative": _fitting_and_dread_dic.get("dread", []),
        "tag_ids": rel_tag_ids,
    }
    return result


@bind("api/daily_check/get_luck_data")
def get_daily_check_luck_data(daily_check_id, skin_trait_id, constellation, others):
    """
    获取运势数据
    :param daily_check_id: 活动ID
    :param skin_trait_id: 关联id
    :param constellation: 星座
    :param others: 其他参数
    :return:
    """
    daily_check_obj = MarkActivityService.get_mark_activity_obj_by_id(daily_check_id)
    if not daily_check_obj:
        return gen(ERROR.DAILY_CHECK_ACTIVITY_NOT_FOUND)

    query = Q(mark_activity_id=daily_check_id, skin_trait_id=skin_trait_id, constellation=constellation)
    mark_fortune_obj = MarkActivityService.get_mark_fortune_config(query)
    if not mark_fortune_obj:
        return gen(ERROR.DAILY_CHECK_LUCK_DATA_NOT_FOUND)

    # log data update
    update_status = UserMarkLogService.update_user_mark_log_params(
        filters=dict(
            user_id=others.get("user_id", 0),
            mark_activity_id=daily_check_obj.id,
            skin_result_id=others.get("skin_result_id", 0),
            skin_data_source=others.get("skin_data_source", SKIN_DATA_SOURCE.YIXI)
        ),
        updates=dict(
            mark_fortune_id=mark_fortune_obj.id,
            update_time=datetime.datetime.now()
        )
    )

    return convert_luck_data(daily_check_obj, mark_fortune_obj)


@bind("api/daily_check/get_general_luck_data")
def get_daily_check_general_luck_data(daily_check_id, others):
    """
    获取运势数据
    :param daily_check_id: 活动ID
    :param others: 其他参数
    :return:
    """
    daily_check_obj = MarkActivityService.get_mark_activity_obj_by_id(daily_check_id)
    if not daily_check_obj:
        return gen(ERROR.DAILY_CHECK_ACTIVITY_NOT_FOUND)

    query = Q(mark_activity_id=daily_check_id)
    mark_fortune_obj = MarkActivityService.get_mark_fortune_config(query)
    if not mark_fortune_obj:
        return gen(ERROR.DAILY_CHECK_LUCK_DATA_NOT_FOUND)

    # log add
    log_obj = UserMarkLogService.create(**{
        "user_id": others.get("user_id", 0),
        "device_id": others.get("device_id", ""),
        "skin_result_id": others.get("skin_result_id", 0),
        "skin_data_source": others.get("skin_data_source", SKIN_DATA_SOURCE.ORDINARY),
        "mark_activity_id": daily_check_id,
        "mark_fortune_id": mark_fortune_obj.id
    })

    return convert_luck_data(daily_check_obj, mark_fortune_obj)
