# coding:utf-8

from __future__ import unicode_literals

import dateutil.parser as parser
import time
import datetime

from api.models import Merchant, Coupon, GiftToCoupon

from django.core.management import BaseCommand
from rpc.tool.error_code import CODES

from search.utils.conversation import search_conversation_from_es
from services.unread.stat import UserUnread

from api.models.types import MESSAGE_TYPE
from hippo.models import MerchantRelevance, Doctor
from gm_types.gaia import DOCTOR_TYPE, COUPON_TYPES, COUPON_DISTRIBUTION_STATUS
from gm_types.msg import CONVERSATION_ORDER, CONVERSATION_TYPE
from gm_types.trade import COUPON_TIME_TYPE
from message.views.message import internal_message_send
from api.models.message import ConversationUserStatus

from rpc.exceptions import RPCPermanentError


def get_conversations_v3(user_ids, offset, size, order_by=CONVERSATION_ORDER.LAST_REPLY_TIME,
                         last_reply_time_start=None, last_reply_time_end=None, reply_status=None,
                         is_star=None, user_id=None, user_last_name=None, comment=None):
    """
    获取会话列表， 与v2不同的是全部走es获取conversation_ids
    :param user_ids: LIST[USER_ID]
    :param offset: INT 偏移量
    :param size: INT 每页数量
    :param order_by: 排序规则
    :param last_reply_time_start: str 最后回复开始时间筛选
    :param last_reply_time_end: str 最后回复结束时间筛选
    :param reply_status: bool 回复状态（新消息、旧消息）
    :param is_star: bool
    :param user_id: int 精确搜索用户ID
    :param user_last_name: int 搜索用户昵称
    :param comment: int 搜索备注
    :return:
    """
    if not isinstance(user_ids, list):
        raise RPCPermanentError
    if not isinstance(size, int):
        raise RPCPermanentError
    if size <= 0 or size >= 50:
        size = 50
    es_filters = {'user_ids': user_ids}
    if is_star is not None and isinstance(is_star, bool):
        es_filters['is_star'] = is_star
    if last_reply_time_start is not None:
        es_filters['last_reply_time_start_gte'] = parser.parse('{}+0800'.format(last_reply_time_start)).isoformat()
    if last_reply_time_end is not None:
        es_filters['last_reply_time_end_lte'] = parser.parse('{}+0800'.format(last_reply_time_end)).isoformat()
    es_query = dict()
    # if search_content is not None:
    #     es_query = {
    #         'comment': search_content,
    #         'user_last_name': search_content,
    #     }
    if user_id is not None:
        es_query['user_id'] = user_id
    if user_last_name:
        es_query['user_last_name'] = user_last_name
    if comment:
        es_query['comment'] = comment
    es_sort_type = order_by
    # 全部
    es_result_total = search_conversation_from_es(offset=offset, size=size, filters=es_filters,
                                                  query=es_query, sort_type=es_sort_type)
    # 已回复
    es_filters['multi_filter_status'] = {
        'status': True,
        'user_ids': user_ids,
    }
    es_result_reply = search_conversation_from_es(offset=offset, size=size, filters=es_filters,
                                                  query=es_query, sort_type=es_sort_type)
    # 未回复
    es_filters['multi_filter_status'] = {
        'status': False,
        'user_ids': user_ids,
    }
    es_result_not_reply = search_conversation_from_es(offset=offset, size=size, filters=es_filters,
                                                      query=es_query, sort_type=es_sort_type)
    # es_result = search_conversation_from_es(offset=offset, size=size, filters=es_filters,
    #                                         query=es_query, sort_type=es_sort_type)
    if reply_status == None:
        es_result = es_result_total
    else:
        if reply_status:
            es_result = es_result_reply
        else:
            es_result = es_result_not_reply
    _conversation_ids = es_result['conversation_ids']
    cus_ids = []
    for conversation_id in _conversation_ids:
        sub_ids = ConversationUserStatus.objects.filter(
            user_id__in=user_ids, conversation_id=conversation_id
        ).values_list('id', flat=True)
        cus_ids.extend(list(sub_ids))
    conversation_user_status = ConversationUserStatus.objects.filter(id__in=cus_ids).order_by('-last_reply_time')

    conversation_info_list = [cs.conversation.conversation_info_v3(cs, user_ids) for cs in conversation_user_status]
    conversation_id_to_user_id = {cs.conversation_id: cs.user_id for cs in conversation_user_status}
    for c in conversation_info_list:
        default = 1 if c['is_new'] else 0
        c['unread_num'] = UserUnread(conversation_id_to_user_id[c['id']]
                                     ).get_conversation_unread(c['id'], default=default)
    return {
        'conversation_list': conversation_info_list,
        'total_count': es_result_total['total_count'],
        'reply_total': es_result_reply['total_count'],
        'not_reply_total': es_result_not_reply['total_count']
    }


class Command(BaseCommand):
    DOCTOR_BUSINESS_CHANNAL_ID_ONLINE = 35

    # only check merchant id can use
    availableMerchantIDList = []

    # check merchant id and coupon can use
    availableMerchantIDListWithCoupon = []

    onlineTestuid = 33703995

    # content = [["area", "merchant_name", 6477328345687146383, 533, 0, "福建最大的整形机构在线独家补贴，领取红包为美减压，爱美就到爱美尔"]]

    c = [
         ['大区', '北京炫美医疗美容', 6328834428978124832, 60936, 60935,
          '北京炫美医疗美容，16年京城老牌机构，美颜黄金周3大焕颜专场，充值返现 高端补水组合任选 另有三甲副主任手术案例招募，详询了解 祝您生活愉快，越来越美哒~~'],
         ['大区', '北京领医医疗美容', 6516969463013739779, 57089, 0,
          '尊贵的客人您好，感谢您对北京领医的关注，我们倍感荣幸！\n作为一家纯正的日式医疗美容诊所，我们始终秉承着日本医疗美容标准，由日本知名医师定期在国内坐诊，让您不出国门尊享日式医美体验！\n【热玛吉全面部 （仪器+医生双重认证）】卷后 ¥10160——9月再加送水光针！\n【伊肤泉微针】卷后 ¥1160\n【皮秒祛斑全面部】卷后¥2960\n还有更多特惠项目，咨询小姐姐在等您哦～'],
         ['大区', '北京美莱医疗美容医院', 6328834567562186390, 61664, 61663,
          '北京美莱钜惠速达 约你共享低价爆款，万人亲选系列 ❤️超微小气泡/无针水光 99元 ❤️伊婉C玻尿酸 980元 ❤️超皮秒祛斑 2888元 ❤️韩式三点定位 2999元 ❤️韩式高挺小翘鼻B 8800元 更多爆品尽在北京美莱  即刻预约，美丽开启~'],
         ['大区', '北京凤凰妇儿医院', 6447645076846916607, 62264, 0, '金秋十月，国庆来京变美七折优惠+旅游，北京凤凰女性专科医院，展现内在美，做私密我们更专业~'],
         ['大区', '北京和颜美帕医疗美容', 6516925349790571330, 61574, 61573,
          '本月抗衰季活动专场： 全面部线雕 5280元 热玛吉（面+颈) 8800元 全面部吸脂 5280元 艾莉薇玻尿酸 1980元 瘦脸100单位 799元 菲洛嘉8ml 1980元 更有超特价活动，等你来享~'],
         ['大区', '武汉诠美医疗美容门诊部', 6328834462117380177, 62222, 62223, '周年庆让利回馈好礼，咨询项目医生再领赠项目'],
         ['大区', '武汉华美整形外科医院总院（中整协修复基地）', 6329570416843092905, 62287, 59742,
          '『华美26周年超值盛典』\n\n超多项目超低价~~\n\n众多精美礼品/医美护肤套装免费送\n\n698爆款周年庆特权卡手慢无‼\n\n具体活动细节滴滴咨询我们哦~'],
         ['大区', '仁爱时光医疗美容', 6462616097651816433, 62279, 62278,
          '终于等到你，时光双节惠劲爆来袭！！！\n①任意下单即可参与抽奖（100%中奖）\n②吸脂/填充/胸臀整形满额送检查费+留观费\n③消费满额送进口玻尿酸/进口除皱针/伊肤泉微针/菲洛嘉等...\n④5000元宝丽太专享红包大放送\n更多优惠可以私信咨询哦~ '],
         ['大区', '武汉中翰整形外科医院', 6424920729401199852, 62283, 0, '欢迎小仙女来到武汉中翰，中秋国庆即将来临，我给您准备了一些项目特惠，还有国庆专属红包等小惊喜哦！'],
         ['大区', '武汉五洲整形美容医院', 6329571025797364148, 62259, 36913,
          '欢迎小仙女来到武汉五洲，中秋国庆即将来临，我给您准备了一些项目秒杀特惠，还有国庆专属红包等小惊喜哦！\n1、水光针秒杀188元\n2、玻尿酸低至 399元\n3、进口激光祛斑低至380元\n4、四代热玛吉全面部低至9800元\n5、瘦脸针低至399元\n6、手术项目低至6折\n还有超多低价特惠项目，私信回复您想了解的项目，我给您送优惠噢！'],
         ['大区', '武汉华美颌面中心（三级整形外科医院）', 6666977204624175933, 62285, 62286,
          '感谢宝宝的关注，是有什么困扰呢，现在预约到院面诊可以立减2000优惠，更多活动请咨询客服！'],
         ['大区', '武汉美仑医疗美容门诊部', 6328834555612612562, 62289, 62291,
          '哈喽--宝宝---吸脂/抗衰了解一下！！！武汉美仑前所未有专属特价！！！\n一：限时大腿/腰腹吸脂模招募仅需4280！！！\n二：限时全面部溶脂模招募仅需2980！！！\n三：五代热玛吉满200发加1000元再送100发！！！\n四：全面部线雕提升新客优享体验仅需4800！！！\n焕美惊喜来袭更有多个元变美红包私信客服等你来以随时私信我哟！'],
         ['大区', '武汉伊美馨医疗美容门诊', 6328834444970968889, 62290, 62294,
          '亲亲你好，我是武汉伊美馨整形医院唐医生，伊美馨专业塑美22年。\n9月特惠项目：脱毛年卡99，润百颜玻尿酸199，瘦脸针足量480，水光特价190，双眼皮1980，吸脂活动680单部位，填充680单部位，唇综合2500起。\n亲亲有需要详细了解的吗？'],
         ['大区', '华中科技大学同济医学院医院', 6335684117300101907, 61813, 60907,
          '您好 同济医美科，金秋十月喜迎双节，院内针剂项目足量注射，产品安全保障，手术项目专家教授亲诊21年临床经验，效果技术保障，更多活动优惠等您领取！'],
         ['大区', '紫馨', 6328834427262649614, 60309, 62205,
          '亲爱哒，本院国庆周年庆返场特惠活动热烈进行中~\r\n进口玻尿酸 低至580\r\n瘦脸针100单位 仅需499\r\n光子嫩肤M22 仅需380\r\n还有整形项目一口价、2人同行折扣价，无套路等您来了解~\r\n领取优惠券，预付款和尾款还可以再减哦！'],
         ['大区', '丽格', 6473497161664304578, 61599, 61600,
          '【广州联合丽格双节惠】腰腹、大腿吸脂低至8800元。到院还有拼团特价礼、车费限额报销和充值礼等活动。领红包下单更优惠哦，更多优惠请咨询我们~'],
         ['大区', '海珠今生', 6328834445944122018, 62240, 0, '今生医美大腿吸脂低至3980，脂肪丰胸低至3880，新客免费上门礼，等你来撩~'],
         ['大区', '深圳米兰柏羽', 6328834519763820324, 62116, 0,
          '宝宝您好！米兰柏羽16周年院庆啦，特推出特惠秒杀活动：1.海月兰水光针66；2.光子嫩肤99；3.伊婉C玻尿酸680；4.进口瘦脸针19805.法思丽大分子玻尿酸1280元；6.热拉提PLUS双探头全面部 2380元平台券与满减券可叠加使用！时间有限，优惠多多，宝宝们不要犹豫了哦！您想要了解的项目都可以咨询我哈'],
         ['大区', '深圳健丽', 6329571036681598886, 62117, 0,
          '【健丽双节购美狂欢季】特价针剂二人团购再打8折，健丽王牌抗衰套餐折上折，到院还有报销车费、充值礼、老带新返现等超值活动噢，更多优惠扣【1】获取~'],
         ['大区', '深圳鹏爱秀琪', 6358965722089031521, 62101, 62100,
          '【鹏爱秀琪与您庆双节】全场全面部填充特惠4500起！腰腹、大腿环吸低至7880！国庆来院至高可抵扣1000元！优惠超值错过要哭哭哦~'],
         ['大区', '深圳福华医疗美容医院', 6328834432161633105, 60119, 57180,
          '哈喽，小仙女是想要了解什么问题呢？咱们福华10月国庆大促到院就可以领取价值380元的医用面膜一盒，以及预存100就能抵1000使用哦。消费1W就可砸金蛋！还有更多的现金红包、储值礼以及消费礼，亲亲都可以线上直接了解哦！'],
         ['大区', '深圳微姿医疗美容门诊部', 6328834430152605159, 62162, 62161,
          '亲亲您好呀，微姿整形10月国庆特惠来啦~手术项目拍单即送除皱针1次！全身吸脂3个部位只要9800元！做吸脂另送面部填充，做假体丰臀送脂肪丰胯！交定金翻3倍使用，还可以享受折上折！比平时省下大几千！现在赶紧咨询客服小姐姐领取优惠礼包哦！'],
         ['大区', '深圳美颜医疗美容门诊部', 6328834713129691180, 62160, 0,
          '你好， 我院有国庆双节优惠活动正在进行中！  \n满10000减1000. 满5000减500.满1000减100.  满500减50.\n来院还可报销来院车费'],
         ['大区', '深圳广和医疗美容', 6328834422183305150, 62141, 62141,
          '哈喽，广和双节送福利！1.医院修复面膜；2.visia皮肤检测；3.小气泡清洁补水；4.3D思维成像设计；5.专家亲诊形象设计，以上福利五选一私信来院即可免费送'],
         ['大区', '深圳艺星医疗美容', 6456040881702122545, 58171, 58172,
          '艺星品牌15周年庆火热进行中！！！\n国庆好礼相送：\n①水光针；\n②进口除皱；\n③进口玻尿酸；\n④全城滴滴免费接送（整形项目消费满5000元）；\n回复数字即可领取，无套路限时领取。\n有任何不明白的可以随时私信我们哟！'],
         ['大区', '深圳联合丽格', 6328834635891485833, 36259, 36260, '30国庆节到院红包，狂欢国庆优惠红包满1000减50，无门栏优惠券20元\r\n'],
         ['大区', '深圳美莱', 6425625439502380754, 62198, 0,
          '您好，美莱22周年庆典活动盛大开启，私信预约即送10000元大礼包。\n热卖项目推荐：\n①国产瘦脸针599元；\n②进口瘦脸针1999元；\n③进口除皱针599元；\n④进口玻尿酸980元；\n⑤艾莉薇玻尿酸2980元；\n⑥超光子嫩肤999元；\n⑦超皮秒祛斑+水光1980元；\n⑧正版热玛吉12800元；\n⑨腰腹/大腿环吸10888元；\n⑩全脸脂肪填充12800元；\n全年最划算，错过很遗憾；更多活动详情，私信撩我帮你解答！'],
         ['大区', '成都武侯格莱丽', 6418755372676294081, 61649, 61646,
          '乘风破浪，未“莱”可期，格莱丽品牌战略升级计划正式启动，打造城东医美新格局-格莱丽锦江旗舰院，致力给大家带来VIP至尊式的医疗与服务体验，领红包再下单，价格更优惠，格莱丽锦江旗舰院地址：成都市锦江区华润路175号'],
         ['大区', '重庆新瑞俪医疗美容', 6328834488243667252, 60029, 62187,
          '重庆新瑞俪整形美容医院国庆红包大放送，无门槛神卷、满减神卷在线免费领，单部位吸脂520元起、双眼皮特价1030元起、鼻综合5780元，更多优惠详情在线免费咨询'],
         ['大区', '重庆玛恩医疗美容医院', 6691895305748714310, 60277, 60278,
          '亲亲您好，是想改善皮肤什么问题呢？我院9月购美狂欢节很多项目都有活动哦~玻尿酸买一送一，光子嫩肤疗程仅需234元，热拉提（定点+射频）疗程仅需5666元，下单前还可领取红包，性价比超高！您在浏览过程中有任何问题都可私信小恩哦~'],
         ['大区', '重庆沙坪坝玺悦台医疗美容门诊有限公司', 6328834618296466352, 61520, 62183,
          '亲亲您好，玺悦台喜迎中秋国庆双节，建院历史最低价，8万元红包补贴全额抵扣。全身吸脂3888元，厚唇改薄（双侧）2380元，双眼皮2980元......本院承诺顾客到院，院长亲自手术，无套路及任何隐形消费，左下角咨询客服小姐姐即送医用面膜哦！'],
         ['大区', '重庆美莱整形美容医院', 6381312986409169226, 62186, 62188,
          '【双节重庆 美莱有礼】综合美眼4项￥3800，美鼻可减5000，进口假体隆胸立减20000；换季美肤买1得5，热玛吉立减4000，超皮秒祛斑买1送1；玻尿酸买1送1，口腔矫正5折起，脱毛年卡低至￥38…更多上门礼、消费礼 敬请咨询~'],
         ['大区', '重庆宝丽医疗整形', 6501632206673289789, 62173, 62170,
          '宝宝节日快乐~超长假期美丽计划，全城最具诚意的活动价全都给你——【只做口碑 价格透明无套路 院长全程亲诊】\n买射频送水光针双节价199\n日式无痕美眼眼综合5项3500\n眼综合7项4800\n无隐形消费：耳软骨鼻综合5项5999\n【无套路】双C妈生鼻膨体鼻综合6项8800\n半肋鼻综合7项12800\n全脸脂肪填充4699\n全脸线雕3999\n篇幅有限还有好多超级划算的活动都写不下啦，快来找我给你说最划算最值得的方案呀~~ '],
         ['大区', '重庆艺星医疗美容医院', 6329570408446085371, 62184, 62185, '周年庆到店有礼'],
         ['大区', '重庆联合丽格美容医院', 6328834429254943862, 58448, 48889, '双节专属 周年钜惠'],
         ['大区', '昆明艺星医疗美容医院', 6328834427111659985, 62166, 62165, '艺星周年庆特惠膨胀金红包'],
         ['大区', '昆明梦想医疗美容医院', 6329570529611180848, 62163, 57282, '双节专属 周年钜惠'],
         ['大区', '昆明吴氏嘉美卓星美容医院', 6374079359682144370, 62176, 62175, '周年庆返场狂欢+国庆大礼包'],
         ['大区', '昆明韩辰医疗美容医院', 6328834523131835983, 61056, 61053, '双节专属 周年钜惠'],
         ['大区', '成都玛丽亚医疗美容', 6329571003697540960, 62190, 62189,
          '感谢关注国际JCI认证、国家二级医院，全国十大微整抗衰名院一玛丽亚! [玛丽亚10月购物狂欢节]①888元水光针/玻尿酸/光子嫩肤/水杨酸祛痘/眼周热拉提/除皱针/敏博士舒敏补水/盆底肌康复/妈妈臀塑型12大热门]项目任选3次(可重复选)。②888元购2020元热玛4代5代抵用券,888元当2020元花。'],
         ['大区', '成都中医大银海医美', 6456090571630042287, 54244, 53015,
          '银海中国眼，让美更经典，欢迎来到国家三级医院，成都中医大银海医美。眼鼻整形5折起，热玛吉3350起  热拉提年卡2999，菲洛嘉、姣兰买一赠一，更多优惠活动详询在线客服'],
         ['大区', '四川省人民医院医疗集团友谊医院', 6382427920501107262, 61844, 61843, '惠满双节  为美加冕 下单立减1千'],
         ['大区', '成都润美玉之光医疗美容', 6328834422745359471, 57055, 61703,
          '亲爱的，品牌&CCTV-1专访的成都润美玉之光和您一起开学啦，9月【开学季·焕颜新生】铃声响起，上千个项目为你添美，更有14项国家专利技术加持，让我们加倍焕颜！现在开始，小姐姐们，捉紧时间跟我们联系吧！'],
         ['大区', '成都军大整形美容医院', 6328834424494408282, 62086, 0,
          '您好，“成都军大医院”欢迎您！  我院为中国人民解放军、西部战区空军医院技术指导医院 • 中整协分会指定西南教学示范基地 •  并由中国平安&amp;中国人保承保西南地区医美保险官方指定首家合作机构，我是专属您的客服笙笙**，接下来我将为您一对一提供咨询服务！'],
         ['大区', '西安美莱医学美容医院', 6419735170521299284, 62204, 61589,
          ' 美莱集团，创立于1999年，全球29城连锁36院，是亚洲规模至大的专业医疗美容连锁集团。旗下拥有“美莱”、“华美” “紫馨”三大知名品牌，致力于整形美容、皮肤美容、微整形、牙齿美容、中医美容等专业领域的应用与发展，集团拥有300多位院长/主任级骨干医师，195名博士专家，100多个专利技术，1000多项学术著作发表国内外权威期刊杂志，300多万例个性美丽案例。'],
         ['大区', '西安画美医疗美容医院', 6436029762384264000, 39227, 45971, '画美周年庆预约金10倍膨胀颗抵扣特价，充5000得10000'],
         ['大区', '成都棕南医院（私密整形医学研究中心）', 6400946260789962410, 61510, 61511,
          '很感谢亲亲的关注！我们是棕南医院，私密整形是我院开展10余年的专业项目，知名专家操作保证效果，更有满2000减1500活动优惠，搭配专属预付款券价格更优！'],
         ['大区', '四川娇点医学美容医院', 6328834489829137966, 61868, 61826,
          '亲亲您好，四川娇点是成都电视台指定的美肤抗衰基地，看您对我们的项目很感兴趣，可以在线咨询了解给您更多专业性解答，线上无门槛减免20元，满1000减100元。'],
         ['大区', '云南华美美莱美容医院', 6395829340461939297, 48947, 0, '【华美美莱22周年庆】 22款口碑爆品，全年最低价！预存22元小金库，到院最高可抵22000元'],
         ['大区', '昆明美伊莱医疗美容', 6328834616790646964, 43295, 0, '美丽畅选，颜值加倍，素颜卡59.9起'],
         ['大区', '昆明吴氏嘉美美容医院', 6374080848668443421, 61356, 61357, '37周年感恩钜惠，充1万得2万，预存100抵3700，水光99，纹眉299，瘦脸针399，热拉提88'],
         ['大区', '云南美诗沁医疗美容', 6464376332737330715, 61694, 0, '美诗沁周年庆，私信领红包'],
         ['大区', '重庆当代整形美容医院', 6328834449660238121, 62025, 0,
          ' 重庆当代9月18日-10月31日，全城10周年庆，眼鼻项目满5000减800可叠加，手术类项目冲1万送1万，购买国产瘦脸针（680元）赠送玻尿酸一支，更多特价项目，详情请私信客服了解！'],
         ['大区', '成都军建医院', 6328834526684469274, 60921, 62218,
          ' 到院送医用面膜2张    \n美丽膨胀金 直接抵钱花\n预存100抵1000\n最高抵扣5000元。军建整形 变美0元购go\n1赞=5元 集赞当钱花\n玻尿酸 瘦脸针 水光针 双眼皮 去眼袋免费送🆓\n66个赞可到院免费领取精品厨具\n10倍膨胀金💥最高抵扣5000元 '],
         ['大区', '成都医大医院', 6503522319443435746, 62212, 59104,
          ' 亲亲，您好~感谢关注成都医大整形，我院专注唇▪眼▪鼻整形，持续为你变美护航。国庆长假变美有礼啦！！！热门爆品玻尿酸¥299、瘦脸针100单位¥480、四代热玛吉脸部¥4980、微笑唇¥6800、全鼻综合¥16800，限时抄底等你来享！\n上门就送：补水喷雾/品牌医用面膜/脱唇或腋毛3次\n预约金膨胀：100元抵500元，500元抵2000元\n网络预约整形项目全线折上折，低至7.5折~ '],
         ['大区', '重庆华美整形美容医院', 6329571006155434729, 61642, 61641,
          ' 轻松变美，到华美-在线预约可领取价值198元医院面膜，美国进口M22王者之冠光子嫩肤3次疗程套餐仅790元，还可叠加到院红包哟！更多整形优惠及在线面诊快快咨询吧！！！'],
         ['大区', '重庆军美医疗美容医院', 6329153871742549047, 60780, 60779,
          '喜迎国庆~预约100抵500,200抵1000,300抵2000~平台预约新客可获得润百颜喷雾/医用面膜/小气泡/无针水光4选1~~M22嫩肤体验单次199，激光祛斑年卡998，热玛吉8800，菲洛嘉680，液态精雕下颌缘1880，大腿环吸4880，双眼皮2580，隆鼻7800，进口隆胸15800。。。。还有更多优惠，等你来'],
         ['大区', '重庆珂琳雅医疗美容门诊部', 6328834459202328430, 62138, 0,
          ' 哈喽，亲亲，即日起在线预约的宝宝即可赠送医用面膜1盒/卡通水杯1个/医用精华液1支，并享受重庆主城区内专车接车送福利，还有更多优惠哦！！'],
         ['大区', '西安壹加壹医疗美容医院（西北旗舰店）', 6497039796123247593, 62191, 61798,
          ' 来院礼（仅限外科/口腔初诊）：来院送院庆 1280 元大礼包 \n（腋毛 6 次+雪肌冰导 3 次） \n6、同行礼（仅限 1 次）：同行送蚕丝面膜（1 人消费即送） '],
         ['大区', '西安米兰柏羽医疗美容医院（原西安美立方）', 6328834443008031526, 58162, 58161,
          '西安美立方上市更名《米兰柏羽》，携手成都米兰、深圳米兰两地医疗资源，服务品质、医疗技术、优质产品全新焕变升级。用心做品质，用爱做医美，我们一直在路上，欢迎您随时咨询！'],
         ['大区', '西安现代医美中心', 6328834635807602879, 31205, 31201, ' 亲亲，看你浏览了我们医院的爆款项目，欢迎咨询了解哦，凡到院即送医用面膜！'],
         ['大区', '太原军大医疗美容医院', 6526403647813837133, 60181, 61980,
          ' 军大是一家集医疗美容服务、美容科研、医美培训、学术交流于一体的专业整形机构，拥有独栋8000平方精奢大楼，设有整形外科，皮肤美容，微创美容中心、私密中心，口腔美容等核心科室，并引进海外设器，打造千级层流净化的手术环境，营造五星级舒适优雅的医疗环境，建立了完善的个性化美容体系，用心打造艺术自然之美。 军大专家团队云集了军区整形巨匠10余位，在医疗美容的各个领域内都有其出类拔萃的人才，并定期赴外学习、交流技术，为培养军大医美人才奠定扎实基础，为您的美丽保驾护航！'],
         ['大区', '成都铜雀台医学美容医院', 6328834478550578937, 60550, 60551, ' 成都铜雀台金秋美肤抗衰季，爆品项目低至1折起，来院即送防晒喷雾！'],
         ['大区', '成都恒博医疗美容', 6328834489678147584, 56895, 56894, '恒博天姿双节大庆，为美丽保驾护航'],
         ['大区', '西安悦华医疗美容门诊部', 6328834472083039460, 61990, 61989, '喜迎双节，悦华百万优品大放价 '],
         ['大区', '西安华都妇产医院', 6366822194550487592, 62203, 62195,
          ' 亲亲 国庆双节活动来咯 您所关注的【小阴唇修复 手术紧缩等】爆款项目 从今日起可以同时享受以下福利： \n①每日下单前三名可以领取到院福利礼包/交通补贴福利 ； \n②尾款大满减红包福利【满90减50；满3000减500 ；满6000减1000】 \n③私密手术类项目可以享受赠送【私密漂红】体验 ! \n无套路,无隐形消费,更多真实案例私信了解 ！ '],
         ['大区', '重庆时光整形美容医院', 6328834442743859892, 61751, 61750, ' 亲亲，您好，看到您浏览我们的项目，是想了解哪方面呢？有任何疑问随时为您解答！'],
         ['大区', '重庆光博士医疗美容', 6329571006646101183, 59466, 59467, ' 你好，亲亲最近在了解什么项目呢？国庆优惠活动来咯~领红包可抵现金用，到院还有送皮肤项目哦~'],
         ['大区', '上海华美医疗美容医院', 6328834441472921182, 61236, 0,
          '美肤黄金周 惊喜不断 \n皮秒全面部3980 \n超皮秒全面部1880 \n微针祛痘套组1980 \n热拉提plus医师版2580 \nDPL超光子嫩肤全面部980 \n基础水光单次499 3次卡1280 \n新一代热玛吉全面部600发26800'],
         ['大区', '上海首尔丽格医疗美容', 6328834457289693664, 62207, 62206,
          '国庆出行堵又挤，不如整容悦自己\n享礼包的价，变更靓的颜\n①颧骨缩小29800，下颌角➕皮质骨38000\n还有1万元颧骨固定费，5千元下巴补助金，直接当场抵扣（9月30号结束）\n②INMODE面部射频提拉 初享体验1999，送碧萃诗透明质酸和法国迪玛丝缇面部刷酸\n颜值小改变 气质新体现\n③切开双眼皮➕  上睑下垂矫正 韩国院长价15800\n另韩国院长眼修复招募，还你姣好面目\n800元轮廓修整面诊费，500元CT补助金\n在首尔丽格，不允许你不好看，赶紧联系客服预约哦'],
         ['大区', '上海薇琳医疗美容医院', 6328834454294911932, 62003, 61618,
          '#欢迎咨询上海薇琳中美韩（国际）连锁整形医院#国际造美节，品牌连锁钜惠~现临薇琳品牌23周年，为回馈广大客户，特突出N大优惠活动时间9月16日至10月31日\n①水光补水2ml特价秒杀188元\n②光子嫩肤M22只需399元\n③进口去鱼尾纹白菜价699元\n④伊婉C玻尿酸680元伊婉V1580元\n⑤"瘦身季"美国酷塑溶脂首点2980元\n⑥买一送一"热玛吉全面部"900发仅需10800元\n⑦2300元十选三套餐卡，2000元十二选一无限购卡\n更多“颧骨颌面，鼻综合、眼综合、吸其整形等咨询客服小姐姐鸭~'],
         ['大区', '上海艺星医疗美容医院', 6329570408508957823, 62229, 0,
          'Yestar上海艺星整形「15周年庆」强势来袭‼️\n❗️全场充值满一万送一万❗️充值越多折扣越多；还有更多抽奖等您来\n1、院庆期间：新老顾客上门就送价值1280元的中医亚健康调理1次（肩颈疲劳/头痛/微波暖宫）\n2、变美项目全场按价目表8.8折优惠，年中最大力度优惠力度\n3、每日10点前到院缴费治疗（不含手术）的顾客，享“早市9折”；\n4、老带新当天两人同时消费，送价值4800元针刺疗法穴位埋线各一次，消费满10000元以上，送玻尿酸补水面膜1盒价值298元，以此类推赠送不停。\n快来小窗撩我下单囤货吧?~错过又要等一年！\n活动时间:2020.9.18～2020.11.10'],
         ['大区', '杭州珈禾医疗美容医院', 6433496798190426143, 62232, 0, '当国庆遇上中秋；珈禾17年院庆拉开帷幕；即日起咨询线上客服寻求变美礼包，80种爆款项目，低至2折起售'],
         ['大区', '上海百达丽医疗美容门诊部', 6328834501472520725, 62227, 0,
          '美肤黄金周 惊喜不断 \n超皮秒嫩肤全面部800 \n伊肤泉微针套组1380  \nAOPT超光子嫩肤全面部580 \n基础水光单次399 3次卡1111 \n除皱单次388  瘦脸单次588            叠加预约金红包更优惠'],
         ['大区', '上海天大医疗美容整形', 6328834467477712267, 62233, 0,
          '亲爱的小仙女，感谢关注上海天大医疗美容医院。\n天大医院是具备整形最高四级手术资质的专业医院，皮肤科被评为上海皮肤美容优势专科...\n以下是为您推荐的本月爆款项目\n①无针水光（本月特惠）51元\n②深蓝射频眼周299元\n③热拉提PLUS    999元\n④埋线提升          3800元    \n⑥胶原蛋白0.5ml   2680元\n⑦热玛吉闺密双人行   7900元/人     \n红包可抵扣进口玻尿酸项目哟，详情可咨询咱们的客服宝宝'],
         ['大区', '上海虹桥医院（私密整形中心）', 6411769888783819957, 61236, 61237,
          '亲爱的十月一黄金周来啦 惊喜不断 \n活动期间到院，均赠送伴手礼一份 消费满5000元赠送脱毛两次（比基尼，脱毛，腋毛三选一） 消费满8000元，赠送乳晕粉嫩1次'],
         ['大区', '上海铂曼医疗美容门诊部', 6328834432585202327, 62239, 0,
          '"双节同庆&铂曼热卖推荐：\n①双美胶原蛋白 全国御用讲师手打价5580元\n②乔雅登玻尿酸 乔雅登风云讲师手打价5980元\n③酷塑冷冻溶脂 季末打包价4个点位15880元\n④超皮秒下单就减百元红包\n⑤OPT光子嫩肤下单立减200元，套餐价格更优惠\n###当季热品·铂曼热玛吉###\n支持四维认证+沪上热玛吉认证医师团"\r\n'],
         ['大区', '上海光博士医疗美容', 6328834508275693538, 62236, 62237,
          '光博士国庆巨惠来袭： 1.光子嫩肤闲时卡 166元  2.皮秒嫩肤+C6祛黄 399元 3.艾莉微 1800元 4.私密漂红 380元 5.伊婉C 880元 6.伊婉大分子 1180元 7.水光2次+C62次  666元 8.切开双眼皮 980元 9.基础水光年卡1080元 10.蜂巢皮秒全面部 1200元 11.菲蜜丽私密紧致 1280元 12.热拉提全面部  1980元 13.自体脂肪隆胸 4900元 14.乔雅登雅致 3980元 15.热玛吉全面部 4999元 光博士现有常驻医生8位 两位副高 四位主治  医院逾4000平 大国匠心值得信赖'],
         ['大区', '上海时光整形', 6328834558070481766, 55485, 55486, '上海时光祝您国庆节快乐！宝宝您好，我是上海时光医院咨询师，有什么可以帮您吗？'],
         ['大区', '上海瑞格医疗美容', 6643082665715062466, 62262, 62263,
          '亲爱的，您好哦。感谢您对咱们医院的关注！\n看到您有浏览咱们家的爆款项目。\n咱们医院国庆大促活动正在进行。\n1、激光阴道紧缩首次体验399元\n2、第七代M22黄金超光子首次特惠价格\n仅需580元！3次卡1440元！5次卡1990元！\n3、超皮秒首次体验999元\n更有多重优惠在等着您呢，详情可以询问在线客服小欧哦！'],
         ['大区', '青岛艾菲医疗美容', 6589753748900393876, 62120, 0, '终于等到你，还好我没放弃~，下单咨询客服有惊喜~ 十一期间线上预约还可以免费升级院长操作哦，详情咨询客服小姐姐'],
         ['大区', '济南吉芙医疗美容', 6575303956036136234, 49989, 49987, '亲亲，我们国庆期间做活动，下单就送项目，更有诸多产品1折抄底秒杀，详情咨询客服。'],
         ['大区', '青岛诺美德医疗美容', 6329571820945023082, 61286, 61288, '您好亲，感谢关注青岛诺美德医疗美容医院，私信预约即可领取医用面膜一贴哦！'],
         ['大区', '济南艺星医疗美容医院', 6361823477556696944, 62121, 0,
          '艺星品牌15周年庆盛大开启\n九大爆款项目任选三项仅需999元！\n外科项目8.5折起！\n美丽储值最高可返60%。\n超多惊爆特惠，奢华礼品相送。欢迎宝宝私信在线客服预约。'],
         ['大区', '临沂微整美容医院', 6672743453216063095, 62167, 62168,
          'A-Durian:\n★★感谢关注临沂微整美容医院，院内项目钜惠，私信预约的话即可领取医用面膜一贴哦~~~\n'],
         ['大区', '青岛华韩整形美容医院', 6451001379719868551, 61709, 61708,
          'hello亲爱的 感谢您浏览咱们家的项目，咱们华韩是上市整形美容医院，是青岛最大的中韩专家基地，执行国际JCI金标准，为了给大家谋福利，更美平台推出华韩所有项目4.9折的优惠，并且到院送你们每人一份到院礼，市内四区初客可以免费打车接诊服务，对项目有什么疑问，随时咨询我们哦！'],
         ['大区', '济南韩氏整形外科医院', 6328834840464494810, 62169, 62172, '专属于小仙女的红包到啦，快快签收哦~上帝欠您的，韩氏还给您！'],
         ['大区', '青岛元美整形', 6451281159954399369, 62148, 62158,
          '您关注的整形医院给您发了一个红包~终于等到你，欢迎关注！我们是更美全国十佳整形名院—青岛元美整形，原九院医生团，主打眼鼻脂肪及皮肤针剂项目，戳红包开始变美↑在线咨询客服可以领取大额满减红包，您可以在下方输入您想咨询的问题'],
         ['大区', '济南海峡美容整形医院', 6355971291014606078, 62122, 62123,
          '\u200d♀️听说漂亮的小姐姐都会回信息，堪比年度大促的【宠粉节】 抓住=赚钱！全网性价比最高，美丽超划算！诚意如何？随时等你来撩~立即回复想变美的项目/部位，特权专属，马上安排妥妥的~🙋  '],
         ['大区', '青岛新阳光妇产医院', 6572414173580372892, 62132, 62131,
          '当中秋遇上国庆\n热闹的小长假即将来临\n你做好准备了吗？\n双节来临之际\n为了回顾新老顾客，\n线上预约来院有诸多好礼，\n下单赠送私密漂红（数量有限，先到先得），\n详情细节咨询客服，还可以领取满减红包哟！\n私密整形可能改变不了生活，但却能防止生活被改变！'],
         ['大区', '青岛伊美尔国宾整形外科医院', 6474155179343091800, 61057, 0,
          '青岛伊美尔23周年狂欢庆典超燃开启啦，一大波钜惠来袭，秒杀、满减、优惠不停歇，更有多种福利享受。现在预约手术项目，尊享手术项目预存500抵1000元特权福利！活动力度空前，先到先得哟~小仙女，还在等什么，咨询小伊在线等撩啦~'],
         ['大区', '杭州华山连天美医疗美容医院', 6328834550013200855, 61435, 0,
          'call我的小可爱我教你领大额独家美券~~~~现在下单10.1变美，中整协首批5A级机构，四级资质。品质之选，安心变美！'],
         ['大区', '杭州丽脂医疗美容', 6379635575154122552, 62177, 62178,
          '宝宝您好，本院是国家脂肪研究基地，全流程升级手术方案，术前进口仪器精准评估检测脂肪，定制化术后恢复方案。颠覆传统吸脂手术，全程愉悦舒适。目前国庆特价活动进行中，超值红包等您领取，有什么问题可以随时咨询我哦'],
         ['大区', '杭州华颜医疗美容', 6573088221398306401, 62159, 0,
          '亲爱的宝宝\n感谢关注杭州华颜医疗美容\n十月小长假，华颜项目推荐：\n全身吸脂：￥9800\n超皮秒：￥880\n玻尿酸：￥380\n菲洛嘉套餐：¥1212\n（套餐含：菲洛嘉、英诺小棕瓶、碧萃诗水光针、VC/谷胱甘肽、医用面膜1片）\n\n更多折扣，直接私信详询哦~'],
         ['大区', '杭州天鑫医疗整形（旗舰店）\t', 6679646006415478064, 61658, 0,
          '亲爱的顾客，天鑫开业狂欢季已经开始！\n活动时间：9月1日至10月22日\n\n礼包一秒杀价：3980元\n1.赛诺秀皮秒3次\n2.中胚层水光针2次\n3.医用修复面膜2盒\n\n礼包二秒杀价：4980元\n1.DPL敏感肌修复3次\n2.中胚层伊肤泉2次\n3.医用修复面膜2盒\n\n礼包三秒杀价：5980元\n1.M22超光子抗衰3次\n2.中胚层水光针2次\n3.医用修复面膜2盒\n\n各种特惠活动，秒杀项目\n私信客服小姐姐哦~'],
         ['大区', '杭州格莱美医疗美容医院\r\n', 6328834449207214509, 62136, 62139,
          '#杭州格莱美9周年庆#\n这“礼”是，我们的9周年❗\n眼综合4项：￥4999\n韩国清新鼻套餐三项：￥6980 \n国产瘦腿针：￥1999\n国产瘦肩针：￥1599\xa0\n形体吸脂12选5：￥8888 \n热玛吉4代全面部：￥6800 \nFotona 4D 祛黑眼圈：￥3980\nM22光子嫩肤：￥380 \n水光针：￥280\n皮秒全脸祛斑：￥899\n菲洛嘉：￥980 \n伊婉C玻尿酸：￥599\xa0\n国产瘦脸针100单位：￥599\n更多折扣，直接详询哦 ~ヾ(❀╹◡╹)ﾉ~'],
         ['大区', '浙江艾博医疗美容\t', 6549130697758602613, 61922, 61923,
          '宝宝您好欢迎光临浙江艾博医疗美容；双节同庆已经来临💥小艾特意为您安排了一大波福利，快来领取吧💞①360会员专享万元大礼包；②消费满减高至1200元；③热玛吉💥两人拼团价1980元；④全身美白spa纳米美白太空舱520元；⑤甄选好货十余款超值变美福利产品；礼品数量有限❗快来撩我'],
         ['大区', '杭州美莱医疗美容医院', 6328834497466989650, 61585, 0,
          '【杭州美莱6周年庆】\n整形6.6折，双人同行6折~~\n预约金+尾款双红包满减~~\n多充多送还享折上折~~\n私信抽奖，中奖率100%'],
         ['大区', '艾尔莎·奥拉克', 6476030498811491675, 61427, 54976, '无门槛红包+大额满减红包。只为您轻松变美'],
         ['大区', '杭州4ever', 6440416859425683293, 62179, 0,
          '推送内容：国庆狂欢，领取红包到院消费满1500立减200元，伊婉C660一支不限购，面部吸脂380起，全脸脂肪填充特惠价4980，全脸线雕5880不限根数，小眼综合4280……更多特惠活动欢迎咨询客服~'],
         ['大区', '杭州艺星医疗美容医院', 6514734530362971298, 62309, 0,
          '艺星品牌15周年庆回馈星粉 已全城启幕 预约8大爆款 任意项目2020元超值抢购 更有星粉到院畅享1500元美力金 星粉消费3倍积分 砸金蛋100%中奖等超值豪礼相送 充值1万得2万 全场85折 更多超值惊喜项目引爆福利~ 全年大优惠，错过再等一年 星粉随时来撩哦！！！'],
         ['大区', '河南汇星医疗美容整形', 6328834445352728936, 62075, 62074,
          '亲爱的汇星粉丝宝宝，本次2020国庆嗨购 前所未有大放送！不止5折！\n①报路费 ②免住宿 ③免费体验皮肤护理588元套餐，\n福利不易，大家想做的项目，多领红包！尽快下哦！'],
         ['大区', '郑州美莱医疗美容医院', 6382856553178943526, 62084, 62082,
          '美莱中秋国庆狂欢声调-美丽跨“月”时光，到院即抽奖（戴森、迪奥、香奈儿）+200元丹尼斯购物卡+保价爆品超低价，更多优惠私信客服哦'],
         ['大区', '河南中医药大学第一附属医院', 6341949409768536792, 62094, 62096,
          '金秋双节·限时优惠为您金秋送礼！①全切双眼皮：原价5800，活动购价\n2680②眼综合：原价12800活动价4280③假体隆鼻惊爆活动价4280，\n更有除皱季卡980活动时间：10.1-10.8，国庆假期让自己变美吧，\n带给你专属钜惠，更多详情宝宝可以回复咨询哦。'],
         ['大区', '巨星医疗整形连锁（紫荆山分院）', 6494489572083337095, 62119, 50944,
          '🚩国庆节医美购┈┈避坑大全    \n最强优惠在这里，盘它更划算️️️\n️水动力全身吸脂八折优惠，仅需6240\n️凡到院消费顾客即送医用修复贴一盒\n️闺蜜同行更有多项变美项目体验，其他项目同样优惠。\n\n名额有限，宝宝速速抢购，赶快咨询我哦！'],
         ['大区', '郑东安琪儿医疗美容门诊部', 6328834450230692023, 61263, 0,
          '安琪儿免费项目拍了拍你，国庆特惠福利大放送，免费招募名额有限，私信即可领红包，来院还有MAC口红等更多大奖免费抽哦～～'],
         ['大区', '郑州善雅整形', 6670976491028744546, 58447, 0, '宝宝你好呀，看宝宝领取了咱们医院的红包，到院可以抵现金使用哦，预约面诊可以免费领取皮肤项目。宝宝怎么考虑的呢？'],
         ['大区', '郑州妍琳整形', 6328834469566444797, 62099, 62102, '金秋十月，全院所有项目低至一折起，双人同行送水光针一支'],
         ['大区', '河南缔莱美整形美容医院', 6582094348182189887, 62174, 62171,
          '当中秋节偶遇国庆节\xa01.\xa0私信礼雾化水光1次 面膜1盒 VISAT7皮肤检测1次2.\xa0到院礼外地顾客最高可报销500元车费本市顾客可享受专车到院服务3.\xa0加购礼消费满2000 加101元即可获得水光针1支消费满3000加101元即可获得玻尿酸1支消费满5000加101元即可获得瘦脸针1支消费满10000加101元即可获得菲洛嘉1支\xa04.\xa0同行礼双人同行 诊外项目可享受8.8折优惠\xa05.\xa0红包\xa0尾款红包 预约金红包 优惠享不停'],
         ['大区', '郑州华领医疗美容医院', 6329571039579834406, 62181, 62180,
          '中秋＆国庆 双节联动 ，惊喜不断 \r外科新客来院即送医美面膜一份\r①微创双眼皮 1680元\r②耳软骨鼻综合 4680元\r③M22光子嫩肤 399元\r④超皮秒全面部 780元\r⑤瘦脸针100u 680元\r活动火爆，到院前一定记得跟小领预约哦！\r另有眼、鼻、吸脂、隆胸、免费案例招募（详情请在线咨询呦）'],
         ['大区', '河南gy广运整形（美博士）医疗美容', 6537149559146021770, 57920, 0,
          '小仙女，你好\n双节钜惠 本院众多项目低至一折起\n线上预约！即可报销车费\n预交10元当1000元使用\n0元招募眼鼻模特\n闺蜜同行送价值2680元嫩肤补水项目+医用面膜一盒\n更多优惠，欢迎私信我们小仙女，你好\n双节钜惠 本院众多项目低至一折起\n线上预约！即可报销车费\n预交10元当1000元使用\n0元招募眼鼻模特\n闺蜜同行送价值2680元嫩肤补水项目+医用面膜一盒\n更多优惠，欢迎私信我们'],
         ['大区', '谢景良医疗美容门诊部', 6496619164067714448, 62303, 0,
          '双节同庆，谢景良整形狂欢节开始啦~全场项目低至1折起，除皱99元，瘦脸针299元起，全面部填充6800元，大腿吸脂仅需5800元哦！更多活动请CALL我哦！！！'],
         ['大区', '郑州虎珂医疗美容', 6574840536383913572, 62308, 0,
          '当中秋节偶遇国庆节，美丽优惠大放价\n1.\xa0私信礼私信预约赠您无针水光1次或面膜2片\xa0\n2.\xa0到院礼手术顾客最高可报销500元车费\n3.\xa0加购礼消费满2000\xa0加101元即可获得小气泡1次；消费满3000加101元即可获得水光针1次；消费满5000加101元即可获得瘦脸针1支；消费满10000加101元即可获得3deep面部提升一次\n4.\xa0同行礼双人同行\xa0诊外项目可享受8.8折优惠\xa0\n5.\xa0红包\xa0尾款红包\xa0预约金红包\xa0优惠享不停 '],
         ['大区', '洛阳孔繁荣医疗美容门诊部', 6493018595419838307, 62307, 0,
          '国庆出行堵又挤，不如整容悦自己，宝宝们准备好变美了吗？\n孔繁荣整形医院十一钜惠来袭，水光、双眼皮、热玛吉……活动价格惊喜不已！\n赶紧咨询客服享受一年一次的最低优惠吧！爱你呦～\n享礼包的价，变更靓的颜！'],
         ['大区', '杭州美莱医疗美容医院', 6328834497466989650, 61585, 0,
          '【杭州美莱6周年庆】\n整形6.6折，双人同行6折~~\n预约金+尾款双红包满减~~\n多充多送还享折上折~~\n私信抽奖，中奖率100%'],
         ['大区', '苏州美贝尔美容整形医院', 6333228717778706316, 62083, 0,
          '【苏州美贝尔9周年庆·享3大报销4大特权6大秒杀】\n外科项目享三大报销 \n①报销路费 ②报销住宿费 ③报销体检费\n【购99元抗衰卡 享四大特权】\n5次大气泡+医用面膜2盒\n3次光子嫩肤980\n3次皮秒祛斑2980 \n3次菲洛嘉 2980\n3次热拉提2980\n【六大秒杀爆款】\n菲洛嘉 ￥699 瘦脸100单位+除皱 ￥699\n伊婉C￥699 皮秒祛斑 ￥699\n热拉提 ￥699 祛眼袋￥699'],
         ['大区', '原韩整形', 6432129803196862117, 62080, 62081,
          'hello亲爱的～💕感谢关注【原韩整形旗舰店】KIBM大韩美容医学会中国区合作机构💕吸脂买一送一，真的送没套路！超肋鼻综合9980起！韩籍博士院长金志昱，连续3年获消费者满意No.1，连续4年任韩国选美评委👉中籍院长郑伟（韩国S Propose整形外科认证医生）拥有1000+台鼻部手术👉中籍院长杨健为mini吸脂创始人，专攻脂肪项目研究！的丰富临床经验。💕私信客服到院免费领医用面膜❗️您可以回复您想要变美的部位，例如“鼻子”，免费帮您定制变美方案～'],
         ['大区', '南京美贝尔', 6328834490479193411, 61276, 0,
          '您好，欢迎光临南京美贝尔官方旗舰院！！！                    美贝尔是国际连锁整形医院，拥有多项国家专利技术，并设立美贝尔整形技术研究院，拥有国内资深专家最多的医院！您如果方便，可以发下您素颜的照片，我们美容师可以给您做个面诊，在线为您做一套美丽方案。       正值国庆，美贝尔福利放“价”，现预约来院，我们将送您价值1680元活动大礼包：【单部位除皱1次+焕氧美肤1次+皮肤检测1次】（数量有限哦）立即咨询客服抢占名额吧！！'],
         ['大区', '南京医科大学友谊整形外科医院', 6411890656733565465, 56073, 0,
          '双秋兼闰月，普天共同庆！\n南京医科大学友谊整形外科医院祝您节日快乐！节日福利已经来临\n①下颌角V颜定制25800\n②清新小鹿眼双眼皮+内眼角4680\n③纯欲感元气鼻综合5980\n④自体脂肪隆胸6800\n⑤热玛吉5980！\n火爆进行中！到院即送暖手宝，详情咨询客服 '],
         ['大区', '苏州薇琳美容医院', 6328834425756958961, 62027, 0,
          '快来快来，小薇过生日啦~薇琳23周年盛典，错过等一年！全线项目劲爆钜惠，性价比高到尖叫！\n新客来院即送医美面膜一份；购一万享3万；预存128元，升值至2300元周年庆礼包。\n回复数字即可询问：\n①补水水光+美白VC 198元；\n②超皮秒祛斑+嫩肤+嫩肤水光模式 999元； \n③M2光子嫩肤：299； \n④菲洛嘉2支：1980元；\n⑤热玛吉600发*2次，9999元\n⑥面部吸脂（面颊+下颌缘+双下巴）案例价仅需：2800元；\n⑦韩式内切祛眼袋980；\n⑧全身吸脂（大腿、腰、腹、背部、上臂五个部位）7999，另送塑身衣一套。\n周年盛典活动火爆，到院前一定记得跟小薇预约哦！'],
         ['大区', '苏州康美美容医院', 6328834553209176918, 61540, 61542,
          '在啊，亲\n苏州康美—中国整形美容协会指定机构，明星医院\n\n①M22光子嫩肤 520元\n② 瘦脸针100u 580元\n③ 瘦腿/瘦肩针 200u 1680元\n④超皮秒祛斑全面部 1980元\n⑤热玛吉全面部 5800元\n⑥全身吸脂（签约瘦身） 案例价6900元\n⑦牙齿矫正（免费拍片） 5666元起\n\n回复心仪项目编号💕\n让你花最少的钱，做最美的自己！'],
         ['大区', '广州建国医院', 6653595066751288394, 62095, 0,
          '\t\n小仙女在吗，感谢关注建国女性私密整形中心~~~\n三甲公立、大咖专场！一站式解决女性私密问题！\n1、在线预约即送价值590元私密检测项目；\n2、全线手术项目直减500元；\n3、所有项目均可报销车费；\n4、三甲公立副主任医师领衔限号接诊；\n欢迎私信了解，有更多疑问可咨询私密师哦'],
         ['大区', '南京柠檬', 6556456340485616955, 62112, 62114,
          '正在为女神派发红包～\n中秋＆国庆 双节联动 \n你的美，我们包了\nᐅ整形产品超大券：满10000减1000\n#私信get更多福利~#'],
         ['大区', '江苏施尔美整形美容医院', 6328834446480980730, 62118, 0, '国庆抄底 钜惠给你。足量正品可验瘦脸针仅售380元！眼综合2680元！'],
         ['大区', '南京艺星医疗美容医院', 6328834503410269786, 60014, 61087,
          '宝宝们，你们期待的艺星15周年庆活动来咯，眼综合3680，鼻综合4800，600发热玛吉只需5480，另外全场立减1000元，还有更多礼品免费领取，详情可以在线咨询客服哦'],
         ['大区', '苏州美莱美容医院', 6328834553209176918, 61211, 0,
          '小仙女你好呀，感谢关注苏州美莱\n①中国5A级整形医院，亚洲29城36院连锁\n②美莱专家团拥有11位副高级以上医师、15位主治医师\n③三位20余年经验的麻醉医师，为您变美保驾护航\n\n回电想要变美的部位，免费帮您制定专属变美方案！\n\n美莱15年，“潮”美前行，苏州美莱周年庆8大优惠等你来拿\n1，上门礼  2，回家礼  3，储值礼  4，消费礼  5，回馈礼\n6，专场礼  7，特价礼  8，国潮礼\n\n周年庆详情登陆更美搜索苏州美莱\n\n全场下单满20减20，满1000减100，满5000减500~\n整形专场活动：\n预交100抵500，满5000可用\n预交100抵1000，满10000可用\n预交100抵2000，满20000可用~\n\n\n'],
         ['大区', '苏州瑞芙臣医疗美容', 6328834553209176918, 62281, 0,
          '嗨！双节黄金周 \n苏州瑞芙臣医疗美容1折抄底闭馆嗨购会\n6重礼遇 礼献全城\n嗨够8天8夜                                                                                1、超强特惠礼  2、惊喜返现礼   3、院长特权日  4、免费霸王餐    5、高定套餐礼  6、奢美消费礼。'],
         ['大区', '南京凤凰岛整形外科', 6340464445805873076, 62234, 62235,
          '您好，感谢关注凤凰岛整形外科！\n①我获得最具影响力美容整形机构、南京市玄武区示范机构、明星医院等各种殊荣！\n②专注眼部整形15年之久的胡国强院长亲诊！\n③限时菲洛嘉688，润百颜白紫288、伊婉v厂家回馈1280限购1支、英诺小棕瓶1180送皮秒\n④凤凰岛10月抗衰季精品套餐，打造您的面部年轻化，送您10年前的面容！\n凤凰岛承诺：所售产品及项目均正品保障，当面可验。'],
         ['大区', '南京嘉怡美医疗美容医院', 6434951206803648916, 57024, 57023, '嘉怡美活动大促，这是送您的购美红包哦，用红包更优惠哦'],
         ['大区', '南京康美美容医院', 6328834424624479013, 61280, 61283,
          '小仙女你好呀，目前正值康美十八周年庆院庆月，\n1.⌠护肤年卡⌡🌱助你完胜乘风破浪的姐姐们！\n锁定本周末，＃扫码寻宝*^_^*终极专场＃\n2.⌠年轻复颜术⌡➰𝐂𝐎𝐌𝐄𝐈线雕术，༄紧致提拉༄抚平皱纹\n3.⁽⁽ଘ玻尿酸ଓ⁾⁾༻༄时下最时尚的美眼神器\n🐾跟我一起体验#玻尿酸丰唇#\n全场有⌠免单⌡^O^扫码必⌠中奖⌡ \n精彩福利不断！更多精彩好礼，欢迎私信咨询哦！'],
         ['大区', '南京芬迪', 6328834530547386754, 62049, 62052, '周年院庆活动进行中，超多惊喜优惠等你来噢！芬迪宠粉节，开心过中秋！'],
         ['大区', '南京韩辰整形医院', 6328834915420966193, 62230, 62231,
          '\r\n亲爱的更美宝宝，欢迎回来南京韩辰~~ 9月品牌升级月（9.1-9.30） --不畏谣言，破浪前行-- 1、国产瘦脸足量 499元 2、BOTOX瘦脸针100单位 2580元 3、天鹅瘦肩针200单位 1280元 4、热玛吉4代600发 4999元 5、头发种植1000单位 3999元 6、头顶加密种植试种200单位限时价 78元 品牌升级，预约上门即送点痣3颗，详情可咨询我们客服人员哦~~~'],
         ['大区', '南京华韩奇致', 6328834657387391605, 62265, 62266,
          '欢迎咨询华韩奇致整形医院  全面部脂肪填充1980，皮秒全面部祛斑仅需380，详情可咨询客服 宝宝现在下单到院还可领取扫地机器人。'],
         ['大区', '南京安安医疗美容', 6328834647476177109, 62217, 62216,
          '☞☞来院就领蟹！享礼包的价，变更靓的颜！①268元豪华蟹礼！！提前庆双节~整形代金券2000元+豪华中秋蟹礼盒+优斯玛臻贵礼盒②只要你咨询我们就大放送！！私信礼+消费礼+免单礼~来问我啊③1001元好事成双6选2~不来后悔！！超声提升/脂肪填充/可塑人造鼻/可以反悔的双眼皮/熊猫眼半年卡/美肌修复卡！！来安安，不允许你不好看，赶紧联系客服预约哦~回复数字即可询问：①一众皮肤项目9元起；②光子嫩肤179元，黑脸娃娃97元；③小菲洛嘉套餐（水光针+除皱+美白）379元；④脂肪填充单个部位519元，双部位979元，全面部4779元；⑤黑眼圈双次97元，眼袋复原术1479元；⑥埋线提升单个部位1579元，全面部4779元；⑦面颊吸脂/双下巴吸脂仅需：2979元；⑧新一代热玛吉全面部9800元'],
         ['大区', '南京健丽医疗美团', 6328834620162897725, 62312, 0,
          '\n外科项目享三大报销 \n①报销路费 ②到院就送项目 ③消费立减大量金额\n【购880元抗衰卡 享全年无限次数】\n\n【六大秒杀爆款】\n菲洛嘉 ￥880  英诺小棕瓶￥ 880\n瘦脸100单位+下颌缘提升 ￥680\n3支伊婉C￥1960 润百颜黑金版￥560 \n 不开刀去眼袋￥5440 不开刀双眼皮￥ 5440'],
         ['大区', '阜阳微悦医疗美容', 6328834794230675662, 56764, 62089,
          '女神你好呀，由于国庆和中秋的到来，特此推出特价项目                                         \n双节大促：\n1、眼综合2600元。\n2、鼻综合4980元。\n3、全脸填充9800元。\n4、瘦脸针680元。\n5、水光针190元。\n欢迎女神来咨询了解，回复即送价值98元的小气泡1次。'],
         ['大区', '安徽维多利亚整形外科医院', 6481430998646024191, 58808, 58807,
          '9月20日—10月31日 荣耀开启！\n————🎊🎊—————\n八大回馈美耀全城 \n豪礼狂欢感恩同贺 \n1️⃣ 转发集赞到院得精美礼品\n2️⃣ 预充得超值增额 还可叠加500元无门槛现金券\n3️⃣ 院庆储值翻倍增 充1万得2万\n4️⃣ 880、3880院庆大礼包超值购。'],
         ['大区', '合肥亚典美容医院', 6584368185951251314, 62110, 62109,
          '金秋十月家国共欢庆，共聚合肥亚典焕新颜~变美套餐无需等待①眼部综合4项2388元②鼻综合8项抄底价仅8999元！③窈窕女神必备款：全身吸脂5300！④自体丰胸5800！混血芭比翘眼部综合案例模特全免打版在线招募中！即日起手术项目来院企业滴滴专车接待，到院即可领取价值199元皮肤清洁护理项目！'],
         ['大区', '合肥台美丽格医疗美容', 6566596510962754122, 62133, 0,
          '亲亲您好 欢迎关注合肥台美丽格，为了庆祝双节，现在预约到院即送医用面膜一盒+巨补水1次+超微小气泡1次，所有手术项目消费即送全年医用面膜12盒，更有膨胀金100抵1000元，更多优惠请私信客服哟'],
         ['大区', '合肥华美整形外科医院', 6328834490659525492, 61474, 61475,
          '#合肥华美26周年庆#\n【安徽省二级专科医院】\n【安徽国际胸模大赛主办机构】\n【安徽首家申请5A医美机构】\n国产瘦腿针年卡3660元；\n瘦身吸脂480元起；\n切开双眼皮1480元；\n脂肪填充4800元；\n祛斑888元|美白780元|紧致980元\n【留   电咨询还赠依云补水喷雾1瓶哦~】\n数量有限，先到先得'],
         ['大区', '合肥白领安琪儿医疗美容', 6330588843795906116, 62225, 62228,
          '国庆出行堵又挤，不如整容悦自己\n①\u3000到院即送项目（小气泡/水氧/无针水光/私密检测4选1）\xa0\n②\u3000整形外科项目享来院报销打车费 \n③\u3000活动期间，院长亲诊眼鼻胸吸脂免费案例名额等你来约\n④\u3000美肤特惠：皮秒全面部980王者之冠全面部580光子嫩肤98\n⑤\u3000手术特惠:激光紧缩1980 全身吸脂9111 全面部填充3800 全面部线雕3777\n在安琪儿，不允许你不好看，赶紧联系客服预约哦 '],
         ['大区', '合肥恒美医疗美容医院', 6328834488050755184, 61266, 0,
          '欢迎咨询合肥恒美整形医院#国庆大促超值价 ①水光补水2ml特价秒杀199元 ②"瘦身季"衡力瘦腿针1780元 ③全国热销"埋线提升"仅需4999元 ④全国热销"热玛吉"仅需15800元 ⑤全国热销祛斑"全脸皮秒"1980元 ⑥国产瘦脸白菜价488元 ⑦国产除皱75元'],
         ['大区', '蚌埠国色整形美容医院', 6493370389144542351, 62260, 62261,
          '亲爱的宝宝\n感谢关注蚌埠国色整形美容医院\n十月小长假，变美项目推荐：\n全身吸脂：￥9800\n切开双眼皮：￥1680\n假体隆鼻：￥1680\n超皮秒：￥380\n玻尿酸：￥380\n瘦脸针100单位：￥650\n果酸换肤：￥1\n小气泡洁肤：￥1\n小气泡洁肤季卡：￥199\n有针水光季卡：￥299\n小气泡洁肤季卡+有针水光季卡：￥399\n更多折扣，直接私信详询哦~ '],
         ['大区', '贵阳华美整形美容医院', 6328834421990465727, 62315, 0,
          '\xa0轻松变美，到华美紫馨-在线预约可领取价值198元医院面膜，美国进口M22王者之冠光子嫩肤只需980元，OPT三次卡1980元，还可叠加到院红包哟！更多整形优惠及在线面诊快快咨询吧！！！'],
         ['大区', '贵阳美贝尔整形美容医院', 6328834457629474887, 62320, 0,
          '\xa0亲亲，您好，看到您浏览我们的项目，是想了解哪方面呢？有任何疑问随时为您解答！爆品项目低至1折起活动优惠中快找我了解吧'],
         ['大区', '杭州薇琳医疗美容医院', 6503805751641777606, 58415, 0,
          '\xa0亲亲，您好，看到您浏览我们的项目，是想了解哪方面呢？有任何疑问随时为您解答！爆品项目活动优惠 快找我了解吧'],
         ]

    def handle(self, *args, **options):
        self.checkMerchant()
        self.checkCoupon()

        self.sendCoupon()

    def sendCoupon(self):

        resMerchantFailedSend = {}
        resMerchantSucceedSend = {}
        resConversationMap = {}

        sendReqNum = 0
        timeStart = time.time()

        for line in self.availableMerchantIDListWithCoupon:

            failedSendUserList = []
            succeedSendUserList = []
            allConversationID = []

            doctorIDList = []
            doctorUIDList = []
            sendCouponAdminUID = 0
            targetUserSet = set()
            sendCouponList = []
            conversationList = []

            merchantID = line[2]

            print()
            print()
            print("start send coupon of merchant id: {merchantID}".format(merchantID=merchantID))
            print("merchant id: ", merchantID)
            couponID1 = line[3]
            print("coupon id 1: ", couponID1)
            couponID2 = line[4]
            print("coupon id 2: ", couponID2)
            sendCouponList.append(couponID1)
            sendCouponList.append(couponID2)

            sendGiftList = self.getGiftID(sendCouponList)
            print "gift list: "
            print sendGiftList
            msgContent = line[5]
            print("msg content: ", msgContent)

            # get merchant info
            merchant = Merchant.objects.filter(id=merchantID).first()
            if not merchant:
                continue

            # get admin doctor id
            adminDoctorID = merchant.doctor.id

            # add admin doctor id to doctorIDList
            doctorIDList.append(adminDoctorID)

            # get other doctors
            doctors = MerchantRelevance.objects.filter(merchant_id=merchantID, doctor__doctor_type=DOCTOR_TYPE.DOCTOR)
            doctorCount = doctors.count()
            ds = doctors[0: doctorCount]
            otherDoctorList = [d.doctor.id for d in ds]

            # add other doctor id to doctorIDList
            doctorIDList.extend(otherDoctorList)

            # get all doctor info
            doctors = Doctor.objects.filter(pk__in=doctorIDList)

            for doctor in doctors:

                ddid = doctor.id
                duid = doctor.user_id

                # add every doctor uid to doctorUIDList
                doctorUIDList.append(duid)
                if ddid == adminDoctorID:
                    sendCouponAdminUID = duid

            # get message target user' user_id list at 2020/4/1 - 2020/7/31
            offsetNumber = 0
            while True:

                # get conversations between 2020/4/1 -- 2020/6/30 of merchant
                conversationResList = get_conversations_v3(
                    user_ids=doctorUIDList,
                    offset=offsetNumber,
                    size=50,
                    is_star=0,
                    last_reply_time_start="2020-03-01T00:00:00",
                    last_reply_time_end="2020-09-30T00:00:00",
                )

                # print("msg res: ")
                # pprint.pprint(conversationResList)
                resCount = conversationResList["total_count"]

                # no conversation
                if resCount <= 0:
                    break

                conversationListOne = conversationResList["conversation_list"]

                conversationList.extend(conversationListOne)

                if len(conversationListOne) < 50:
                    break

                offsetNumber += 50

            conversationNum = len(conversationList)
            print("conversation number of merchant: {merchantID} in 2020-03-01T00:00:00 and 2020-09-27T00:00:00 is ".format(
                merchantID=merchantID), conversationNum)

            # no conversation, handle next merchant
            if conversationNum == 0:
                continue

            # get target user id set
            # 因为发送者和接收者，顺序不定，所以都取出来去判断
            for conversationRow in conversationList:

                targetUserID0 = conversationRow["user_ids"][0]
                targetUserID1 = conversationRow["user_ids"][1]
                conversationID = conversationRow["id"]

                allConversationID.append(conversationID)

                if targetUserID0 not in doctorUIDList:
                    targetUserSet.add(targetUserID0)

                if targetUserID1 not in doctorUIDList:
                    targetUserSet.add(targetUserID1)

            # print("target user id set: ", targetUserSet)
            targetUserNum = len(targetUserSet)
            print("target user number of merchant id:{merchantID} is ".format(merchantID=merchantID), targetUserNum)

            targetUserSet.add(self.onlineTestuid)

            for targetUserOne in targetUserSet:

                try:
                    for giftID in sendGiftList:

                        resSendCoupon = internal_message_send(
                            sender_user_id=sendCouponAdminUID,
                            target_user_id=targetUserOne,
                            msg_type=MESSAGE_TYPE.GIFT,
                            conversation_type=CONVERSATION_TYPE.MESSAGE,
                            content={"gift_id": giftID, "channel_id": self.DOCTOR_BUSINESS_CHANNAL_ID_ONLINE}
                        )

                        sendReqNum += 1

                        errSendGift = resSendCoupon["error"]
                        # send failed
                        if errSendGift == CODES.MESSAGE_CONTENT_FILTERED_OUT:
                            # print(
                            #     "Failed to send merchant:{merchantID}'s coupon: {couponGiftID} to user: {targetIDOne}".format(
                            #         merchantID=merchantID, couponGiftID=giftID, targetIDOne=targetUserOne))

                            failedGiftInfo = str(giftID) + "_" + str(targetUserOne)
                            failedSendUserList.append(failedGiftInfo)

                        # send success
                        # if errSendGift == CODES.SUCCESS:
                        #     print(
                        #         "Succeed to send merchant:{merchantID}'s coupon: {couponGiftID} to user: {targetIDOne}".format(
                        #             merchantID=merchantID, couponGiftID=giftID, targetIDOne=targetUserOne))

                    # send msg
                    resSendMsg = internal_message_send(
                        sender_user_id=sendCouponAdminUID,
                        target_user_id=targetUserOne,
                        msg_type=MESSAGE_TYPE.TEXT,
                        conversation_type=CONVERSATION_TYPE.MESSAGE,
                        content={"text": msgContent}
                    )

                    sendReqNum += 1

                    errSendMsg = resSendMsg["error"]
                    # if errSendMsg == CODES.MESSAGE_CONTENT_FILTERED_OUT:
                    #     print(
                    #         "Failed to send merchant:{merchantID}'s msg to user: {targetIDOne}".format(
                    #             merchantID=merchantID, targetIDOne=targetUserOne))

                    if errSendMsg == CODES.SUCCESS:
                        # print(
                        #     "Succeed to send merchant:{merchantID}'s msg to user: {targetIDOne}".format(
                        #         merchantID=merchantID, targetIDOne=targetUserOne))
                        succeedSendUserList.append(targetUserOne)
                except:
                    continue

            resMerchantFailedSend[merchantID] = failedSendUserList
            resConversationMap[merchantID] = allConversationID
            resMerchantSucceedSend[merchantID] = succeedSendUserList

            print("send coupon complete of merchant: {merchantID}".format(merchantID=merchantID))

        print("res failed to send: ")
        print(resMerchantFailedSend)

        print("res succeed to send: ")
        print(resMerchantSucceedSend)

        print("conversation id list: ")
        print(resConversationMap)

        print("total send request number: ", sendReqNum)

        timeEnd = time.time()
        timeUse = timeEnd - timeStart
        print("time use: ", timeUse)

    def checkMerchant(self):

        unavailableMerchantList = []

        print "===== start to check merchant id ====="

        for line in self.c:
            merchantID = line[2]

            m = Merchant.objects.filter(id=merchantID).first()
            if not m:
                unavailableMerchantList.append(line)
            else:
                self.availableMerchantIDList.append(line)

        print "=== result of check merchant id ==="
        print "num of unavailable line with merchant:"
        print len(unavailableMerchantList)
        print "content of unavailable line with merchant:"
        print unavailableMerchantList
        print "===== end to check merchant id ====="

    def checkCoupon(self):

        unavailableMerchantInfoSet = set()

        print "===== start to check coupon id ====="

        for line in self.availableMerchantIDList:

            doctorIDList = []
            couponList = []

            merchantID = line[2]
            couponID1 = line[3]
            couponID2 = line[4]
            couponList.append(couponID1)
            couponList.append(couponID2)

            # get merchant obj
            m = Merchant.objects.filter(id=merchantID).first()

            # get admin doctor
            adminDoctorID = m.doctor.id
            print "admin doctor", adminDoctorID

            # add admin doctor id to doctorIDList
            doctorIDList.append(adminDoctorID)

            # get other doctors
            doctors = MerchantRelevance.objects.filter(merchant_id=merchantID, doctor__doctor_type=DOCTOR_TYPE.DOCTOR)
            doctorCount = doctors.count()
            ds = doctors[0: doctorCount]
            otherDoctorList = [d.doctor.id for d in ds]

            doctorIDList.extend(otherDoctorList)
            print "doctor list: ", doctorIDList

            # get doctor of every coupon, find it is exist in doctor list
            for couponID in couponList:
                if couponID == 0:
                    continue

                c = Coupon.objects.filter(id=couponID).first()
                if not c:
                    unavailableMerchantInfoSet.add(merchantID)
                    continue

                createDoctor = c.doctor
                if not createDoctor:
                    unavailableMerchantInfoSet.add(merchantID)
                    continue

                createDoctorID = createDoctor.id

                # coupon's create doctor is not belong to merchant,
                # add to unavailableMerchantInfoSet
                if createDoctorID not in doctorIDList:
                    unavailableMerchantInfoSet.add(merchantID)
                    continue

                couponType = c.coupon_type

                # coupon's type is not COUPON_TYPES.DOCTOR,
                # add to unavailableMerchantInfoSet
                if couponType != COUPON_TYPES.DOCTOR:
                    unavailableMerchantInfoSet.add(merchantID)
                    continue

                distributionStatus = c.distribution_status

                # coupon's distribution status is not COUPON_DISTRIBUTION_STATUS.OPEN,
                # add to unavailableMerchantInfoSet
                if distributionStatus != COUPON_DISTRIBUTION_STATUS.OPEN:
                    unavailableMerchantInfoSet.add(merchantID)
                    continue

                timeType = c.time_type

                # if time type is COUPON_TIME_TYPE.START_END,
                # check start_time <= now <= end_time,
                # if incompatible, add to unavailableMerchantInfoSet
                if timeType == COUPON_TIME_TYPE.START_END:
                    startDataTime = c.start_time
                    endDataTime = c.end_time

                    nowTime = datetime.datetime.now()

                    if nowTime < startDataTime or nowTime > endDataTime:
                        unavailableMerchantInfoSet.add(merchantID)
                        continue

                # if can't find GiftToCoupon obj where coupon.id=couponID,
                # add to unavailableMerchantInfoSet
                couponGift = GiftToCoupon.objects.filter(coupon_id=couponID).first()
                if not couponGift:
                    unavailableMerchantInfoSet.add(merchantID)
                    continue

                # if coupon is not enable, add to unavailableMerchantInfoSet
                isEnable = couponGift.coupon_enable
                if not isEnable:
                    unavailableMerchantInfoSet.add(merchantID)
                    continue

        print "===== result of check coupon id ====="
        print "num of unavailable line with merchant and coupon:"
        print len(unavailableMerchantInfoSet)
        print "content of unavailable line with merchant and coupon:"
        print unavailableMerchantInfoSet
        print "===== end to check coupon id ====="

        for line in self.c:

            mid = line[2]

            if mid in unavailableMerchantInfoSet:
                continue

            self.availableMerchantIDListWithCoupon.append(line)

        print "===== available line with merchant and coupon ====="
        print self.availableMerchantIDListWithCoupon
        print "================ END ======================"

    @staticmethod
    def getGiftID(couponList):

        giftList = []

        for couponID in couponList:
            if couponID == 0:
                continue

            couponGift = GiftToCoupon.objects.filter(coupon_id=couponID).first()
            giftID = couponGift.coupon_gift.id
            giftList.append(giftID)

        return giftList
