# coding: utf-8

from datetime import datetime

from agile.models.tag import TagV3, TagV3Relation
from django.db.models import Q
from group.models import Group, GroupContent, GroupFollow, GroupTagV3, GroupTopicRelation, Topic
from agile.services.tag import get_tag_v3_attr_by_tag_v3_ids_by_url, get_ids_by_names
from .es import to_epoch, tzlc
from gm_types.gaia import TAG_V3_TYPE

_base_query = Q(is_online=True)


def get_group_followers(group_id):
    query = _base_query & Q(group_id=group_id)
    return list(GroupFollow.objects.filter(query).values_list("user_id", flat=True))


def get_group_last_content_create_time(group_id):
    query = _base_query & Q(group_id=group_id)
    obj = GroupContent.objects.filter(query).order_by("-create_time").first()
    if obj:
        return tzlc(obj.create_time)
    return tzlc(datetime(1970, 1, 1))


def get_group_last_content_comment_time(group_id):
    query = _base_query & Q(group_id=group_id)
    topic_ids = list(GroupTopicRelation.objects.filter(query).values_list("topic_id", flat=True))
    if topic_ids:
        obj = Topic.objects.filter(id__in=topic_ids).order_by("-update_time").first()
        if obj:
            return tzlc(obj.update_time)
    return tzlc(datetime(1970, 1, 1))


def get_group(instance):
    g = instance
    assert isinstance(g, Group)

    group_id = g.id
    tag_list = g.tag_list
    tag_ids = []
    tag_names = []
    for i in tag_list:
        tag_ids.append(i.get("id"))
        tag_names.append(i.get("name", ""))

    is_show_index = True
    try:
        is_show_index = g.is_show_index
    except Exception as e:
        print(e)
        pass

    res = {
        "id": group_id,
        "name": g.name,
        "mapping_type": "group",
        "is_online": g.is_online,
        "create_time": tzlc(g.create_time),
        "update_time": tzlc(g.update_time),
        "last_content_create_time": get_group_last_content_create_time(group_id),
        "last_content_comment_time": get_group_last_content_comment_time(group_id),
        "tag_ids": tag_ids,
        "tag_names": tag_names,
        "user_ids": get_group_followers(group_id),
        "user_follow_number": len(get_group_followers(group_id)),
        "is_show_index": is_show_index
    }
    tag_v3_ids = list(GroupTagV3.objects.filter(group_id=group_id, is_online=True).values_list("tag_v3_id",
                                                                                               flat=True))  ##该组关联的新标签

    tags_v3_name = []
    tags_v3_ids = []

    tagv3_ids = get_ids_by_names(tags_ids=tag_v3_ids, tags_type=TAG_V3_TYPE.NORMAL, get_ids=True)
    tagv3_names = get_ids_by_names(tags_ids=tag_v3_ids, tags_type=TAG_V3_TYPE.NORMAL)
    tags_v3_ids.extend(tagv3_ids)
    tags_v3_name.extend(tagv3_names)

    second_classify_ids = get_tag_v3_attr_by_tag_v3_ids_by_url(tags_ids=tag_v3_ids, get_name=False,
                                                               agg_type=TAG_V3_TYPE.SECOND_CLASSIFY,
                                                               tag_type=TAG_V3_TYPE.FIRST_CLASSIFY)
    tag_v3_ids.extend(second_classify_ids)
    second_by_get_protag_names = get_tag_v3_attr_by_tag_v3_ids_by_url(tags_ids=tag_v3_ids, get_name=True,
                                                                      agg_type=TAG_V3_TYPE.NORMAL,
                                                                      tag_type=TAG_V3_TYPE.SECOND_CLASSIFY)
    tags_v3_name.extend(second_by_get_protag_names)
    second_by_get_protag_ids = get_tag_v3_attr_by_tag_v3_ids_by_url(tags_ids=tag_v3_ids, get_name=False,
                                                                    agg_type=TAG_V3_TYPE.NORMAL,
                                                                    tag_type=TAG_V3_TYPE.SECOND_CLASSIFY)
    tags_v3_ids.extend(second_by_get_protag_ids)

    res["tags_v3_name"] = list(set(tags_v3_name))
    res["tags_v3_ids"] = list(set(tags_v3_ids))

    return res
