# coding=utf8

from __future__ import unicode_literals, absolute_import, print_function

from django.db import models
from django.db.models import ForeignKey, IntegerField, CharField, TextField
from django.db.models import BooleanField, ManyToManyField, DateTimeField
from django.db.models import OneToOneField

from gm_upload import ImgUrlField, IMG_TYPE
from gm_types.wiki import (
    TREATMENT_TYPE,
    ANESTHESIA_TYPE,
    ACHE_LEVEL_NEW,
    NEW_WIKI_WORDREL_CATEGORY
)

from wiki.models.product import Product
from api.models import Tag
from api.models.wordrel import WordRel


class Item(models.Model):
    class Meta:
        verbose_name = u'项目百科'
        db_table = 'wiki_item'

    soyoung_id = IntegerField(u'新氧id', null=True)
    name = CharField(u'项目介绍', max_length=48)
    icon = ImgUrlField(u'图标', max_length=128, img_type=IMG_TYPE.ITEMWIKI)
    other_name = CharField(u'别称', max_length=100)
    description = TextField(u'介绍')
    effect = TextField(u'功效特色')
    use_result = CharField(u'使用功效', max_length=128)
    fit_people = TextField(u'适应人群')
    avoid_people = TextField(u'禁忌人群')
    advantage = TextField(u'优点')
    disadvantage = TextField(u'缺点')
    treatment_method = CharField(u'治疗方式', max_length=8, choices=TREATMENT_TYPE)
    anesthesia_method = CharField(u'麻醉方法', max_length=8, choices=ANESTHESIA_TYPE)
    ache_level = CharField(u'疼痛感', max_length=8, choices=ACHE_LEVEL_NEW)
    price = CharField(u'参考价格', max_length=50)
    hospitalized = BooleanField(u'是否住院', default=False)
    treatment_time = CharField(u'操作时长', max_length=50)
    maintain_time = CharField(u'维持时长', max_length=50)
    recover_time = CharField(u'恢复周期', max_length=50)
    treatment_num = CharField(u'治疗周期', max_length=50)
    treatment_doctor = CharField(u'治疗人员要求资质', max_length=50)
    pre_know = TextField(u'术前锦囊')
    similar_item = ManyToManyField('self', u'相似项目')
    related_product = ManyToManyField(Product, related_name='similar_items')
    is_hot = BooleanField(u'是否热门', default=False)
    is_online = BooleanField(u'是否上线', default=True)
    create_time = DateTimeField(verbose_name=u'创建时间', auto_now_add=True)
    similar_sy_item = CharField('新氧相似项目', default='[]', max_length=500)
    related_sy_drug = CharField('新氧关联product', default='[]', max_length=500)

    def detail_info(self, extra=True):
        fields = ['id', 'name', 'description', 'effect', 'use_result', 'fit_people',
                'avoid_people', 'advantage', 'disadvantage', 'price',
                'recover_time', 'pre_know', 'is_hot',  'treatment_time']
        detail_info = {}
        for _f in fields:
            detail_info[_f] = getattr(self, _f)

        detail_info['after_know'] = [
            {
                'title': x.title,
                'rank': x.rank,
                'recover_tips': x.recover_tips,
                'care_method': x.care_method
            } for x in self.after_know.filter(is_online=True).order_by('rank')
        ]
        if extra:
            detail_info['similar_item'] = [s.detail_info(extra=False) for s in self.valid_similar_items()]
        return detail_info

    def valid_similar_items(self):
        return self.similar_item.filter(is_online=True).exclude(id=self.id)


class ItemAfterKnow(models.Model):
    class Meta:
        verbose_name = u'项目术后锦囊'
        db_table = 'wiki_itemafterknow'

    item = ForeignKey(Item, related_name='after_know')
    title = CharField(u'名字', max_length=50)
    rank = IntegerField(u'展示顺序')
    recover_tips = CharField(u'恢复提示', max_length=128)
    care_method = TextField(u'护理方法')
    is_online = BooleanField('是否上线', default=True)


class ItemTag(models.Model):
    class Meta:
        verbose_name = u'项目百科关联标签'
        db_table = 'wiki_item_tag'

    item = ForeignKey(Item, verbose_name=u'项目百科', related_name='tags')
    tag = ForeignKey(Tag, verbose_name=u'标签')


class ItemTagV3(models.Model):
    class Meta:
        verbose_name = u'项目百科关联新标签'
        db_table = 'wiki_item_tag_v3'

    item_id = IntegerField(verbose_name=u'项目百科', db_index=True)
    tag_v3_id = IntegerField(verbose_name=u'标签V3', db_index=True)
    create_time = DateTimeField(verbose_name=u'创建时间', auto_now_add=True)
    update_time = DateTimeField(verbose_name=u'更新时间', auto_now=True)


class ItemRelationKeyWords(models.Model):
    class Meta:
        verbose_name = '项目百科 关联的关键词'
        db_table = 'wiki_item_keywords'

    item = ForeignKey(Item, related_name='words')
    wordrel = ForeignKey(WordRel, verbose_name=u'关键字')
    category = CharField(verbose_name=u'分类', max_length=5, choices=NEW_WIKI_WORDREL_CATEGORY)
    is_effect = BooleanField(verbose_name=u'是否有效', default=True)
    created_time = DateTimeField(verbose_name=u'创建时间', auto_now_add=True)
    updated_time = DateTimeField(verbose_name=u'最后更新时间', auto_now=True)

