# coding=utf-8
import json
from django.db import models
from django.utils import timezone
from gm_upload import ImgUrlField, IMG_TYPE
from gm_types.gaia import SIGN_USER_ACTION, SIGN_TYPE_V2, SIGN_TASK_TYPE
from django.contrib.auth.models import User


class SignConfigV2(models.Model):
    class Meta:
        verbose_name = u'签到活动配置'
        db_table = 'api_sign_config_v2'
        app_label = 'api'

    cycle = models.IntegerField(max_length=4, verbose_name=u'按照规则每天新增的积分', default=5)
    rule = models.TextField(max_length=2048, verbose_name=u'活动规则', default="", null=True)
    prize = models.TextField(max_length=2048, verbose_name=u'盲盒活动规则', default="", null=True)
    operate_config = models.TextField(verbose_name=u'运营专题配置', default="", null=True)
    is_online = models.BooleanField(verbose_name=u'是否在线', default=True)
    browse_add_points = models.IntegerField(verbose_name=u"逛商品任务停留秒数", default=20)
    friend_assist_img_url = ImgUrlField(img_type=IMG_TYPE.NOWATERMARK, verbose_name=u'好友助力图片', max_length=256)
    created_time = models.DateTimeField(auto_now_add=True, verbose_name=u'创建时间')
    updated_time = models.DateTimeField(verbose_name=u'更新时间', auto_now=True)


class SignRecordLogV2(models.Model):
    class Meta:
        verbose_name = u'用户签到log表'
        db_table = 'api_user_sign_v2'
        app_label = 'api'
        unique_together = ('user', 'sign_time')

    user = models.ForeignKey(User, verbose_name=u'用户')
    sign_time = models.DateField(verbose_name=u'签到时间', default=timezone.now)
    sign_type = models.CharField(verbose_name=u'签到类型', default=SIGN_TYPE_V2.NOMALR_SIGN, choices=SIGN_TYPE_V2,
                                 max_length=4)
    created_time = models.DateTimeField(auto_now_add=True, verbose_name=u'创建时间')
    updated_time = models.DateTimeField(verbose_name=u'更新时间', auto_now=True)


class SignBigPrizeV2(models.Model):
    class Meta:
        verbose_name = u'签到奖品'
        db_table = 'api_sign_prize_v2'
        app_label = 'api'

    name = models.CharField(max_length=256, verbose_name=u'名称')
    img_url = ImgUrlField(img_type=IMG_TYPE.NOWATERMARK, verbose_name=u'图片', max_length=256)
    count = models.IntegerField(default=0, verbose_name=u'数量')
    exchange_days = models.IntegerField(default=0, verbose_name=u'兑换天数')
    init_count = models.IntegerField(verbose_name=u'初始的奖品数量', default=0)
    online_time = models.DateField(verbose_name=u'上线时间', null=True)
    is_online = models.BooleanField(verbose_name=u'是否在线', default=False)
    created_time = models.DateTimeField(auto_now_add=True, verbose_name=u'创建时间')
    updated_time = models.DateTimeField(verbose_name=u'更新时间', auto_now=True)

    title = models.CharField(verbose_name=u"详情页title", default="", max_length=256)
    rule = models.TextField(max_length=2048, verbose_name=u"详情页规则", default="")
    image_detail = ImgUrlField(img_type=IMG_TYPE.NOWATERMARK, verbose_name=u'详情页图片', max_length=256)
    jump_url = models.CharField(verbose_name=u"下方banner跳转", default="", max_length=256)
    jump_img_url = ImgUrlField(img_type=IMG_TYPE.NOWATERMARK, verbose_name=u'详情页下方banner图片', max_length=256)


class ExchangedRecordV2(models.Model):
    class Meta:
        verbose_name = u'兑换记录'
        db_table = 'api_sign_exchanged_v2'
        app_label = 'api'
        unique_together = ('user', 'prize')

    user = models.ForeignKey(User, verbose_name=u'用户')
    prize = models.ForeignKey(SignBigPrizeV2, verbose_name=u'奖品')
    express_no = models.CharField(verbose_name=u'运单号', null=True, default='', max_length=64)
    created_time = models.DateTimeField(auto_now_add=True, verbose_name=u'创建时间')
    updated_time = models.DateTimeField(verbose_name=u'更新时间', auto_now=True)


class SignUserInfoV2(models.Model):
    class Meta:
        verbose_name = u'用户签到信息'
        db_table = 'api_userinfo_sign_v2'
        app_label = 'api'

    user = models.ForeignKey(User, verbose_name=u'用户', db_index=True)
    is_remind = models.BooleanField(verbose_name=u'签到提醒', default=False)
    add_sign_number = models.IntegerField(verbose_name=u'当月补签数', default=0)
    share_code = models.CharField(verbose_name=u'分享码', default="", max_length=256)
    last_add_time = models.DateField(verbose_name=u'最后补签时间', null=True)
    newest_time = models.DateField(verbose_name=u"最新签到时间", null=True)
    last_days = models.IntegerField(verbose_name=u"当前连续签到天数", default=0)
    show_tips = models.BooleanField(verbose_name=u'是否展示tip，只展示一次', default=False)
    points_count = models.IntegerField(verbose_name=u"积分总数", default=0)
    show_point = models.IntegerField(verbose_name=u"弹窗展示的积分", default=0)
    xcx_form_id = models.CharField(verbose_name=u'小程序推送的form_id', default="", max_length=256)
    rush_remind = models.BooleanField(verbose_name=u'抢购提醒', default=False)
    created_time = models.DateTimeField(auto_now_add=True, verbose_name=u'创建时间')
    updated_time = models.DateTimeField(verbose_name=u'更新时间', auto_now=True)


class SignBlindBox(models.Model):
    class Meta:
        verbose_name = u'盲盒卡片'
        db_table = 'api_sign_blind_prize'
        app_label = 'api'

    name = models.CharField(verbose_name=u"盲盒卡片名称", default=u"", max_length=128)
    img_url = ImgUrlField(img_type=IMG_TYPE.NOWATERMARK, verbose_name=u'盲盒图片URL', max_length=256)
    probability = models.IntegerField(verbose_name=u"概率", default=0)
    is_online = models.BooleanField(verbose_name=u"是否上线", default=False)
    is_final = models.BooleanField(verbose_name=u'是否是隐藏盲盒', default=False)
    show_order = models.IntegerField(verbose_name=u"展示顺序", default=0)
    created_time = models.DateTimeField(auto_now_add=True, verbose_name=u'创建时间')
    updated_time = models.DateTimeField(verbose_name=u'更新时间', auto_now=True)


class SignUserBlindBox(models.Model):
    class Meta:
        verbose_name = u'用户获取的盲盒'
        db_table = 'api_sign_user_blind_card'
        app_label = 'api'

    user_id = models.IntegerField(verbose_name=u"用户ID", null=False)
    blind_prize_id = models.IntegerField(verbose_name=u"盲盒卡片ID", null=False)
    count = models.IntegerField(verbose_name=u"盲盒卡片的数量", default=1)
    created_time = models.DateTimeField(auto_now_add=True, verbose_name=u'创建时间')
    updated_time = models.DateTimeField(verbose_name=u'更新时间', auto_now=True)


class SignBlindExchangePrize(models.Model):
    class Meta:
        verbose_name = u'盲盒兑换的奖品'
        db_table = 'api_sign_blind_relate_prize'
        app_label = 'api'

    name = models.CharField(max_length=256, verbose_name=u'名称')
    img_url = ImgUrlField(img_type=IMG_TYPE.NOWATERMARK, verbose_name=u'图片', max_length=256)
    coupon_id = models.IntegerField(verbose_name=u'美券id(关联coupon表索引ID)', null=True, blank=True)
    count = models.IntegerField(default=0, verbose_name=u'每天的数量')
    description = models.CharField(verbose_name=u'说明', max_length=256, default="")
    is_online = models.BooleanField(verbose_name=u'是否在线', default=False)
    blind_box_ids = models.CharField(verbose_name=u'关联的盲盒卡片ID', default='[]', max_length=256)
    show_order = models.IntegerField(verbose_name=u"展示顺序", default=0)
    created_time = models.DateTimeField(auto_now_add=True, verbose_name=u'创建时间')
    updated_time = models.DateTimeField(verbose_name=u'更新时间', auto_now=True)

    detail_introduce = models.TextField(max_length=2048, verbose_name=u'详情页说明', default="", null=True)
    detail_img = ImgUrlField(img_type=IMG_TYPE.NOWATERMARK, verbose_name=u'详情页盲盒图片URL', max_length=256)

    @property
    def blind_box_list(self):
        str_list = str(self.blind_box_ids)
        blind_list = json.loads(str_list.replace("u", "").replace("'", ""))
        return blind_list


class SignUserExchangedPrize(models.Model):
    class Meta:
        verbose_name = u'用户兑换的奖品'
        db_table = 'api_sign_exchange_prize'
        app_label = 'api'

    user_id = models.IntegerField(verbose_name=u"用户ID", null=False)
    prize_id = models.IntegerField(verbose_name=u"奖品ID", null=False)
    check_adress = models.BooleanField(verbose_name=u"是否确认地址信息（只针对小程序）", default=False)
    express_no = models.CharField(verbose_name=u'运单号', null=True, default='', max_length=64)
    created_time = models.DateTimeField(auto_now_add=True, verbose_name=u'创建时间')
    updated_time = models.DateTimeField(verbose_name=u'更新时间', auto_now=True)


class SignTaskUserFinish(models.Model):
    class Meta:
        verbose_name = u'用户任务完成记录表'
        db_table = 'api_sign_user_task'
        app_label = 'api'

    user_id = models.IntegerField(verbose_name=u"用户ID", null=False, db_index=True)
    task_type = models.IntegerField(verbose_name=u'任务类型', choices=SIGN_TASK_TYPE, max_length=4, db_index=True)
    task_value = models.CharField(verbose_name=u'任务关键值', default="", max_length=128)
    finish_time = models.DateField(verbose_name=u'任务完成时间', default=timezone.now, db_index=True)
    is_read = models.BooleanField(verbose_name=u"任务消息已读or未读", default=False, db_index=True)
    created_time = models.DateTimeField(auto_now_add=True, verbose_name=u'创建时间')
    updated_time = models.DateTimeField(verbose_name=u'更新时间', auto_now=True)


class SignAssistUser(models.Model):
    class Meta:
        verbose_name = u'用户助力表'
        db_table = 'api_sign_assist_user'
        app_label = 'api'

    assist_user_id = models.IntegerField(verbose_name=u"受助的用户ID", null=False)
    user_id = models.IntegerField(verbose_name=u"用户ID", db_index=True)
    created_time = models.DateTimeField(auto_now_add=True, verbose_name=u'创建时间')
    updated_time = models.DateTimeField(verbose_name=u'更新时间', auto_now=True)
    task_type = models.IntegerField(verbose_name=u'任务类型', choices=SIGN_TASK_TYPE, default=SIGN_TASK_TYPE.FRIEND_ASSIST)


class SignPrizeV3(models.Model):
    class Meta:
        verbose_name = u"签到奖品表"
        db_table = u"api_sign_prize_v3"
        app_label = u"api"

    name = models.CharField(max_length=256, verbose_name=u'名称')
    img_url = ImgUrlField(img_type=IMG_TYPE.NOWATERMARK, verbose_name=u'图片', max_length=256)
    count = models.IntegerField(default=0, verbose_name=u'总数量')
    exchange_days = models.IntegerField(default=0, verbose_name=u'兑换天数')
    is_online = models.BooleanField(verbose_name=u'是否在线', default=False)
    created_time = models.DateTimeField(auto_now_add=True, verbose_name=u'创建时间')
    updated_time = models.DateTimeField(verbose_name=u'更新时间', auto_now=True)


class ExchangedRecordV3(models.Model):
    class Meta:
        verbose_name = u'兑换记录'
        db_table = u'api_sign_exchanged_v3'
        app_label = u'api'

    user_id = models.IntegerField(null=False, verbose_name=u'用户ID')
    prize_id = models.IntegerField(null=False, verbose_name=u'奖品ID')
    express_no = models.CharField(verbose_name=u'运单号', null=True, default='', max_length=64)
    created_time = models.DateTimeField(auto_now_add=True, verbose_name=u'创建时间')
    updated_time = models.DateTimeField(verbose_name=u'更新时间', auto_now=True)