#! /usr/bin/env python
# -*- coding: utf-8 -*-
import json
from django.db.models import Q

from api.models.face.user_record import ScanRecord
from api.models.user_face_image import FaceOneImageV2
from api.models.face import SkinResult, AiFeedKyc
from rpc.tool.error_code import CODES, gen
from rpc.tool.dict_mixin import to_dict

from rpc.decorators import bind_context, bind
from api.tool.user_tool import get_user_from_context
from gm_types.gaia import SCAN_TYPE
from api.services.user_face_services import FaceService
from api.tasks.face_task import rectify_client_record_task


@bind_context('api/scan_record/have_scan_face_and_skin', with_context=True)
def have_scan_face_and_skin(ctx, device_id):
    result = {}
    user = get_user_from_context(ctx)
    if user:
        have_scan_face = ScanRecord.objects.filter(user_id=user.id).last()
    else:
        have_scan_face = ScanRecord.objects.filter(device_id=device_id).last()
    result['have_scan_face'] = True if have_scan_face else False
    return result


@bind('api/scan_record/save_scan_face_record')
def save_user_scan_face_record(user_id, device_id, detect='', megvii_analytics='',
                               extra='', scan_type=SCAN_TYPE.SCAN_FACE, image_url=''):

    if not ScanRecord.objects.filter(user_id=user_id or 0, device_id=device_id, image_url=image_url).exists():
        ScanRecord.objects.create(user_id=user_id or 0, device_id=device_id, scan_type=scan_type, detect=detect,
                                  megvii_analytics=megvii_analytics, extra=extra, image_url=image_url)


@bind_context('api/scan_record/get_scan_face_record')
def get_scan_face_record(ctx, device_id):
    user = get_user_from_context(ctx)
    if user:
        obj = ScanRecord.objects.filter(user_id=user.id, scan_type=SCAN_TYPE.SCAN_FACE).last()
    else:
        obj = ScanRecord.objects.filter(device_id=device_id, scan_type=SCAN_TYPE.SCAN_FACE).last()
    data = obj.data if obj else {}
    return {"data": data}


@bind_context('api/had_test_record')
def get_scan_face_record(ctx, device_id):
    user = get_user_from_context(ctx)
    user_id = user.id if user else None

    return FaceService.had_test_record(user_id=user_id, device_id=device_id)


@bind_context('api/rectify_client_record')
def rectify_client_record(ctx, device_id):
    user = get_user_from_context(ctx)
    user_id = user.id if user else None

    return rectify_client_record_task(user_id=user_id, device_id=device_id)


@bind_context('api/tab_kyc/need_scan')
def tab_kyc_need_scan(ctx, device_id):
    user = get_user_from_context(ctx)
    user_id = user.id if user else None

    q = Q(user_id=user_id) | Q(device_id=device_id)
    # 面孔起源(type=4)/模拟整形(type=5)
    face_q = q & Q(scan_type=5)
    scan_record = ScanRecord.objects.filter(face_q).exists()
    if scan_record:
        return {"need_scan": False}

    # 扫脸
    face_record = FaceOneImageV2.objects.filter(q).exists()
    if face_record:
        return {"need_scan": False}

    # 测肤/打卡
    skin_record = SkinResult.objects.filter(q).exists()
    if skin_record:
        return {"need_scan": False}
    return {"need_scan": True}


@bind_context('api/kyc_record/get')
def kyc_record(ctx, device_id):
    user = get_user_from_context(ctx)
    user_id = user.id if user else None

    # todo 方便调试，把device_id去了，上线前加上
    # q = Q(device_id=device_id) | Q(user_id=user_id)
    q = Q(user_id=user_id)
    key_record = AiFeedKyc.objects.filter(q).last()
    if not key_record:
        return gen(CODES.NEED_KYC_ERROR)

    return {"has_kyc_record": to_dict(key_record)}


@bind_context('api/kyc_record/save')
def kyc_record(ctx, scan_image, face_shape_ids, face_feature_ids, face_style_ids, device_id):

    user = get_user_from_context(ctx)
    user_id = user.id if user else None

    kyc = AiFeedKyc()
    kyc.device_id = device_id
    kyc.user_id = user_id
    kyc.image_url = scan_image
    kyc.face_feature = face_feature_ids
    kyc.face_shape = face_shape_ids
    kyc.face_style = face_style_ids
    kyc.save()
    return {"kyc_id": kyc.id}


