# -*- coding: utf-8 -*-
from gm_types.error import ERROR

from api.models.user_face_image import FaceOneImageV2
from api.services.beauty_service import BeautyService
from api.services.word_service import WordService
from api.tool.user_tool import get_user_from_context
from rpc.decorators import bind_context, bind
from rpc.tool.error_code import gen

from api.services.user_face_services import (
    FaceService,
    FacialService,
    FacialFeatureService
)
from rpc.tool.dict_mixin import to_dict
from api.services.tag_service import list_tags_by_polymer_ids


@bind_context('api/user/face_image')
def get_user_face_image(ctx, face_id=None):
    """
    获取用户扫脸图片
    :param ctx:
    :param face_id:
    :return:
    """
    user = get_user_from_context(ctx)
    if not user:
        return gen(ERROR.LOGIN_REQUIRED)

    ret = FaceService.get_one_image(user_id=user.id, face_id=face_id)
    if not ret:
        return gen(ERROR.FACE_IMAGE_NOT_FOUND)

    return ret


@bind_context('api/face_image/add')
def add_one_image(ctx, face_id, face_url, beauty_url):
    user = get_user_from_context(ctx)
    if not user:
        return gen(ERROR.LOGIN_REQUIRED)

    res = {"face_id": face_id}

    ret = FaceService.create_one_image(face_id=face_id, user_id=user.id, face_url=face_url)
    if not ret:
        return gen(ERROR.FACE_IMAGE_CREATE_FAILURE)

    data = BeautyService.get_beauty_info(image_url=beauty_url)
    if not data:
        return res

    beauty_photo_id = data.get('id')

    if not beauty_photo_id:
        return res

    word_parent_info = WordService.get_word_by_beauty_photo(beauty_photo_id=beauty_photo_id)

    data['word_parent_info'] = word_parent_info
    res.update(data)

    return res


@bind_context('api/user_face/share')
def beauty_info(ctx, image_url):
    """获取美图分享信息"""

    user = get_user_from_context(ctx)
    if not user:
        return gen(ERROR.LOGIN_REQUIRED)

    data = BeautyService.get_beauty_info(image_url=image_url)

    word_parent_info = WordService.get_word_by_beauty_photo(url=image_url)

    data['word_parent_info'] = word_parent_info

    return data


@bind("api/one_image/create")
def create_one_image_info(face_id, face_url="", device_id="", user_id=0):
    """
    新版记录用户的扫脸数据
    :param face_id:
    :param face_url:
    :param device_id:
    :param user_id:
    :return:
    """
    return FaceService.create_one_image_v2(
        face_id=face_id,
        face_url=face_url,
        device_id=device_id,
        user_id=user_id
    )


@bind("api/one_image/get_by_face_id")
def get_face_url_by_face_id(face_id):
    """
    通过face_id 获取 对应扫脸图片
    :param face_id:
    :return:
    """
    return FaceService.get_one_image_v2(face_id)


@bind("api/one_image/get_facial_feature")
def get_facial_feature(feature=None):
    """
    获取五官分类信息
    :return:
    """
    _infos = FacialFeatureService.get_online_info(feature)

    return _infos


@bind("api/one_image/facial_display")
def get_facial_display(contour, chin, eyebrow, eye, nose, lip):
    """
    获取面部部位对应的文案
    :param contour:
    :param chin:
    :param eyebrow:
    :param eye:
    :param nose:
    :param lip:
    :return:
    """

    return FacialService.get_desc_by_classification(
        contour=contour,
        chin=chin,
        eyebrow=eyebrow,
        eye=eye,
        nose=nose,
        lip=lip
    )


@bind("api/one_image/get_face_shape")
def get_face_shape(filters):
    """
    通过筛选，获取脸型及文案
    :param filters: {"contour":"yuanlian", "chin":"jianxiaba"}  筛选条件
    :return:
    """
    return FacialService.get_face_name_and_desc(filters=filters)



@bind("api/one_image/create_tractate_record")
def create_tractate_record(face_id, tractate_id):
    """
    保存测颜值发帖记录
    :param face_id:
    :param tractate_id:
    :return:
    """
    try:
        obj = FaceOneImageV2.objects.get(face_id=face_id)
    except:
        obj = None

    if not obj:
        gen(ERROR.APPEARANCE_RES_NOT_FOUND)

    if obj.tractate_id:
        gen(ERROR.APPEARANCE_TRACTATE_EXIT)

    obj.tractate_id = tractate_id
    obj.save()

    return obj.id


@bind("api/one_image/get_tractate_record_by_face_id")
def get_tractate_record_by_face_id(face_id):
    """
    获取测颜值发帖记录
    :param face_id:
    :return:
    """
    try:
        obj = FaceOneImageV2.objects.get(face_id=face_id)
    except:
        obj = None

    if obj:
        return to_dict(obj)
    else:
        return {}


@bind('api/one_image/get_send_msg_record')
def get_send_msg_record(face_id):
    """
    获取测颜值结果是否发送私信
    :param face_id: 测颜值结果id
    :return: {send_msg: 是否发送过私信}
    """
    try:
        obj = FaceOneImageV2.objects.get(face_id=face_id)
    except:
        gen(ERROR.APPEARANCE_RES_NOT_FOUND)

    res = {"send_msg": obj.send_msg}

    return res


@bind('api/one_image/save_send_msg_record')
def save_send_msg_record(face_id):
    """
    保存测颜值结果 发送私信记录
    :param face_id: 测颜值结果id
    :return:
    """
    try:
        obj = FaceOneImageV2.objects.get(face_id=face_id)
    except:
        gen(ERROR.APPEARANCE_RES_NOT_FOUND)

    obj.send_msg = True
    obj.save()

    return obj.face_id
