# coding=utf8

from api.models.tag import Tag
from django.db.models import Model, CharField, IntegerField, TextField,DateTimeField, BooleanField
from gm_upload import IMG_TYPE, ImgUrlField
from cached_property import cached_property
from gm_types.gaia import TOP_CONTENT_RELATED_TYPE, INDEX_CARD_TYPE, GROUP_TOPIC_TYPE


class Group(Model):
    class Meta:
        verbose_name = '内容聚合'
        db_table = 'api_group'

    name = CharField('标题', max_length=32)
    introduction = TextField('描述简介')
    header_image = ImgUrlField(img_type=IMG_TYPE.POLYMER, max_length=255, verbose_name=u'背景图')
    bg_image = ImgUrlField(img_type=IMG_TYPE.POLYMER, max_length=255, verbose_name=u'背景图')
    sm_head_image = ImgUrlField(img_type=IMG_TYPE.POLYMER, max_length=255, verbose_name=u'小头图')

    top_card_type = CharField(verbose_name=u'置顶卡片类型', max_length=3, null=True)
    top_card_id = IntegerField(verbose_name=u'置顶卡片ID', null=True)

    punch_entrance = TextField(verbose_name=u'打入入口相关信息（json）', null=True)

    group_type = IntegerField(verbose_name=u'小组类型', choices=GROUP_TOPIC_TYPE, default=GROUP_TOPIC_TYPE.COMMON)
    is_single_feed = BooleanField(verbose_name=u'是否是单列Feed', default=False)
    show_service_tab = BooleanField(verbose_name=u'是否展示美购tab', default=False)
    show_video_tab = BooleanField(verbose_name=u'是否展示视频tab', default=False)

    is_online = BooleanField('是否上线', default=True)
    is_show_index = BooleanField('是否在首页显示', default=False)
    create_time = DateTimeField(auto_now_add=True)
    update_time = DateTimeField(verbose_name=u'更新时间', auto_now=True)

    @cached_property
    def tag_list(self):
        tag_ids = GroupTag.objects.filter(group_id=self.id).values_list("tag_id", flat=True)
        tags = Tag.objects.filter(pk__in=tag_ids) if tag_ids else []
        return [{"id": tag.id, "name": tag.name, "tag_type": tag.tag_type}for tag in tags if tag.is_online]

    @cached_property
    def topics(self):
        topics = list(GroupTopicRelation.objects.filter(group_id=self.id, is_online=True).order_by("rank").values("topic_id", "rank"))
        topics = sorted(topics, key=lambda item: (item["rank"] or 99999, item["topic_id"]))
        return topics

    @cached_property
    def top_contents(self):
        top_contents = list(
            TopContent.objects.filter(
                related_id=self.id,
                related_type=TOP_CONTENT_RELATED_TYPE.GROUP,
                is_online=True,
            ).values("id", "theme", "title", "content_type", "content_id")
        )
        return top_contents


class GroupTag(Model):
    """小组标签"""

    class Meta:
        verbose_name = u'小组关注'
        db_table = 'api_group_tag'

    is_online = BooleanField(verbose_name=u"是否有效", default=True)
    create_time = DateTimeField(verbose_name=u'创建时间', auto_now_add=True)
    update_time = DateTimeField(verbose_name=u'更新时间', auto_now=True)
    tag_id = IntegerField(verbose_name=u'标签', db_index=True)
    group_id = IntegerField(verbose_name=u'小组id', db_index=True)


class GroupTopicRelation(Model):
    """小组话题关联标签"""

    class Meta:
        verbose_name = u'小组关注'
        db_table = 'api_group_topic_relation'
        unique_together = ('group_id', 'topic_id')

    group_id = IntegerField(verbose_name=u'小组id')
    topic_id = IntegerField(verbose_name=u'话题id', db_index=True)
    rank = IntegerField(verbose_name=u'排序', default=None)

    is_online = BooleanField(verbose_name=u"是否有效", default=True)
    create_time = DateTimeField(verbose_name=u'创建时间', auto_now_add=True)
    update_time = DateTimeField(verbose_name=u'更新时间', auto_now=True)


class GroupTagV3(Model):
    """小组标签"""

    class Meta:
        verbose_name = u'小组关注关联标签'
        db_table = 'api_group_tag_v3'

    group_id = IntegerField(verbose_name=u'小组id', db_index=True)
    tag_v3_id = IntegerField(verbose_name=u'新标签3.0', db_index=True)
    is_online = BooleanField(verbose_name=u"是否有效", default=True)
    create_time = DateTimeField(verbose_name=u'创建时间', auto_now_add=True)
    update_time = DateTimeField(verbose_name=u'更新时间', auto_now=True)


class GroupFollow(Model):
    """小组关注"""

    class Meta:
        verbose_name = u'小组关注'
        db_table = 'api_group_follow'

    is_online = BooleanField(verbose_name=u"是否有效", default=True)
    create_time = DateTimeField(verbose_name=u'创建时间', auto_now_add=True)
    update_time = DateTimeField(verbose_name=u'更新时间', auto_now=True)
    user_id = IntegerField(verbose_name=u'用户ID', db_index=True)
    group_id = IntegerField(verbose_name=u'小组ID', db_index=True)


class GroupContent(Model):
    """小组发布内容"""

    class Meta:
        verbose_name = u'小组关注'
        db_table = 'api_group_content'

    is_online = BooleanField(verbose_name=u"是否有效", default=True)
    create_time = DateTimeField(verbose_name=u'创建时间', auto_now_add=True)
    update_time = DateTimeField(verbose_name=u'更新时间', auto_now=True)
    content_type = IntegerField(verbose_name=u'内容类型')
    content_id = IntegerField(verbose_name=u'内容id', db_index=True)
    group_id = IntegerField(verbose_name=u'小组id', db_index=True)


class TopContent(Model):
    """小组话题置顶内容"""
    class Meta:
        verbose_name = u'小组话题置顶内容'
        db_table = 'api_top_content'

    theme = CharField(verbose_name=u'主题', max_length=32, default='')
    title = CharField(verbose_name=u'标题', max_length=128, default='')
    content_type = CharField(verbose_name=u'置顶内容类型', max_length=3, choices=INDEX_CARD_TYPE, default=INDEX_CARD_TYPE.TRACTATE)
    content_id = IntegerField(verbose_name=u'置顶内容ID', default=0)
    related_type = IntegerField(
        verbose_name=u'关联的类型',
        choices=TOP_CONTENT_RELATED_TYPE,
        default=TOP_CONTENT_RELATED_TYPE.GROUP,
    )
    related_id = IntegerField(verbose_name=u'group表或topic表的ID', default=0)

    is_online = BooleanField(verbose_name=u"是否有效", default=True)
    create_time = DateTimeField(verbose_name=u'创建时间', auto_now_add=True)
    update_time = DateTimeField(verbose_name=u'更新时间', auto_now=True)