# coding=utf8
from gm_types.gaia import GROUP_TOPIC_TYPE, TOP_CONTENT_RELATED_TYPE

from api.models.tag import Tag
from django.db.models import Model, CharField, IntegerField, TextField,DateTimeField, BooleanField
from gm_upload import IMG_TYPE, ImgUrlField
from cached_property import cached_property

from group.models import TopContent


class Topic(Model):
    class Meta:
        verbose_name = '小组话题'
        db_table = 'api_group_topic'

    name = CharField('标题', max_length=32)
    introduction = TextField('描述简介')
    header_image = ImgUrlField(img_type=IMG_TYPE.POLYMER, max_length=255, verbose_name=u'背景图')
    bg_image = ImgUrlField(img_type=IMG_TYPE.POLYMER, max_length=255, verbose_name=u'背景图')
    sm_head_image = ImgUrlField(img_type=IMG_TYPE.POLYMER, max_length=255, verbose_name=u'小头图')

    top_card_type = CharField(verbose_name=u'置顶卡片类型', max_length=2, null=True)
    top_card_id = IntegerField(verbose_name=u'置顶卡片ID', null=True)
    punch_entrance = TextField(verbose_name=u'打入入口相关信息（json）', null=True)

    display_hit_rank = BooleanField(u'是否展示榜单tab', default=True)
    hit_rank_start_time = DateTimeField(u'榜单开始时间', null=True)
    hit_rank_end_time = DateTimeField(u'榜单结束时间', null=True)

    is_show_index = BooleanField('是否在首页显示', default=False)
    group_topic_type = IntegerField(verbose_name=u'小组话题类型', choices=GROUP_TOPIC_TYPE, default=GROUP_TOPIC_TYPE.COMMON)
    is_single_feed = BooleanField(verbose_name=u'是否是单列Feed', default=False)
    show_service_tab = BooleanField(verbose_name=u'是否展示美购tab', default=False)
    show_video_tab = BooleanField(verbose_name=u'是否展示视频tab', default=False)

    is_online = BooleanField('是否上线', default=True)
    create_time = DateTimeField(auto_now_add=True)
    update_time = DateTimeField(verbose_name=u'更新时间', auto_now=True)

    @cached_property
    def tag_list(self):
        tag_ids = TopicTag.objects.filter(topic_id=self.id).values_list("tag_id", flat=True)
        tags = Tag.objects.filter(pk__in=tag_ids) if tag_ids else []
        return [{"id": tag.id, "name": tag.name, "tag_type": tag.tag_type}for tag in tags if tag.is_online]

    @cached_property
    def top_contents(self):
        top_contents = list(
            TopContent.objects.filter(
                related_id=self.id,
                related_type=TOP_CONTENT_RELATED_TYPE.TOPIC,
                is_online=True,
            ).values("id", "theme", "title", "content_type", "content_id")
        )
        return top_contents


class TopicTag(Model):
    """话题标签1.0"""

    class Meta:
        verbose_name = u'话题关注'
        db_table = 'api_group_topic_tag'

    tag_id = IntegerField(verbose_name=u'标签', db_index=True)
    topic_id = IntegerField(verbose_name=u'话题id', db_index=True)

    is_online = BooleanField(verbose_name=u"是否有效", default=True)
    create_time = DateTimeField(verbose_name=u'创建时间', auto_now_add=True)
    update_time = DateTimeField(verbose_name=u'更新时间', auto_now=True)


class TopicTagV3(Model):
    """话题标签3.0"""

    class Meta:
        verbose_name = u'话题关注关联标签'
        db_table = 'api_group_topic_tag_v3'

    topic_id = IntegerField(verbose_name=u'话题id', db_index=True)
    tag_v3_id = IntegerField(verbose_name=u'新标签3.0', db_index=True)
    is_online = BooleanField(verbose_name=u"是否有效", default=True)
    create_time = DateTimeField(verbose_name=u'创建时间', auto_now_add=True)
    update_time = DateTimeField(verbose_name=u'更新时间', auto_now=True)


class TopicFollow(Model):
    """话题关注"""

    class Meta:
        verbose_name = u'话题关注'
        db_table = 'api_group_topic_follow'

    is_online = BooleanField(verbose_name=u"是否有效", default=True)
    create_time = DateTimeField(verbose_name=u'创建时间', auto_now_add=True)
    update_time = DateTimeField(verbose_name=u'更新时间', auto_now=True)
    user_id = IntegerField(verbose_name=u'用户ID', db_index=True)
    topic_id = IntegerField(verbose_name=u'话题ID', db_index=True)


class TopicContent(Model):
    """话题发布内容"""

    class Meta:
        verbose_name = u'话题关注'
        db_table = 'api_group_topic_content'

    is_online = BooleanField(verbose_name=u"是否有效", default=True)
    create_time = DateTimeField(verbose_name=u'创建时间', auto_now_add=True)
    update_time = DateTimeField(verbose_name=u'更新时间', auto_now=True)
    content_type = IntegerField(verbose_name=u'内容类型')
    content_id = IntegerField(verbose_name=u'内容id', db_index=True)
    topic_id = IntegerField(verbose_name=u'话题id', db_index=True)
