# -*- coding: utf-8 -*-
import json
from cached_property import cached_property

from django.db import models
from django.utils import timezone

from gm_types.gaia import (
    AI_QA_QUESTION_TYPE,
    AI_QA_NEXT_TYPE,
    AI_QA_MESSAGE_TYPE,
    AI_QA_ANSWER_TYPE,
    AI_QA_POPUP_TYPE,
)
from gm_upload import ImgUrlField, IMG_TYPE


class Theme(models.Model):

    class Meta:
        verbose_name = u'主题'
        db_table = 'api_ai_qa_theme'

    title = models.TextField(verbose_name=u'主题', null=False, default="")
    ass_uid = models.IntegerField(verbose_name=u'小助手用户ID')
    is_online = models.BooleanField(verbose_name=u'是否在线', default=False)
    first_welcome_contents = models.TextField(verbose_name=u'首次欢迎语')
    redo_welcome_contents = models.TextField(verbose_name=u'重新开始欢迎语')
    conclusion = models.TextField(verbose_name=u'意外退出结束语', default="")
    entrance_question_id = models.IntegerField(verbose_name=u'开始的问题ID', db_index=True)
    create_time = models.DateTimeField(u"创建时间", auto_now_add=True)
    update_time = models.DateTimeField(u'更新时间', auto_now=True)
    last_modified = models.DateTimeField(verbose_name=u'最后修改时间', default=timezone.now)


class RecommendCard(models.Model):

    class Meta:
        verbose_name = u'推荐卡片'
        db_table = 'api_ai_qa_recommend'

    title = models.CharField(max_length=128, null=False, verbose_name=u'标题')
    desc = models.CharField(max_length=1024, null=False, verbose_name=u'描述')
    url = models.CharField(max_length=256, null=False, verbose_name=u'跳转协议')
    scheme_popup = models.IntegerField(verbose_name=u'方案弹窗', choices=AI_QA_POPUP_TYPE, default=0)
    is_online = models.BooleanField(verbose_name=u'是否在线', default=False)
    create_time = models.DateTimeField(u"创建时间", auto_now_add=True)
    update_time = models.DateTimeField(u'更新时间', auto_now=True)
    last_modified = models.DateTimeField(verbose_name=u'最后修改时间', default=timezone.now)


class AIQuestion(models.Model):

    class Meta:
        verbose_name = u'问题'
        db_table = 'api_ai_qa_question'

    title = models.CharField(max_length=128, null=False, verbose_name=u'标题')
    question_type = models.IntegerField(verbose_name=u'类型', default=AI_QA_QUESTION_TYPE.BUTTON, choices=AI_QA_QUESTION_TYPE)


class AIAnswer(models.Model):

    class Meta:
        verbose_name = u'回答'
        db_table = 'api_ai_qa_answer'

    title = models.CharField(max_length=1024, null=False, verbose_name=u'标题')
    image_url = ImgUrlField(img_type=IMG_TYPE.NOWATERMARK, verbose_name=u'图片', max_length=256)
    color = models.CharField(max_length=20, null=False, verbose_name=u'颜色值')
    is_online = models.BooleanField(verbose_name=u'在线状态', default=True)
    answer_type = models.IntegerField(verbose_name=u'回答类型', choices=AI_QA_ANSWER_TYPE, default=AI_QA_ANSWER_TYPE.NORMAL)


class AIQuestion2Answer(models.Model):

    class Meta:
        verbose_name = u'问题 回答 关联关系'
        db_table = 'api_ai_qa_question_answer'

    ai_question_id = models.IntegerField(verbose_name=u"问题id", db_index=True)
    ai_answer_id = models.IntegerField(verbose_name=u"回答id", db_index=True)
    rank = models.IntegerField(verbose_name=u"排序")
    next_type = models.IntegerField(verbose_name=u"下一个关联类型", default=AI_QA_NEXT_TYPE.AI_QUESTION, choices=AI_QA_NEXT_TYPE)
    relation_data = models.TextField(verbose_name=u"关联数据", null=False, default="")
    # 关联数据样例 {'reco_desc': '推荐文案','follow_group_id': 1,'reco_card_ids': [1, 2, 3], 'reco_popup_type'【弹窗类型】:0：无；1方案；2商品}


class AIAnswerTag(models.Model):

    class Meta:
        verbose_name = u'回答标签关联关系'
        db_table = 'api_ai_qa_answer_tag'

    ai_answer_id = models.IntegerField(verbose_name=u"回答id", db_index=True)
    tag_id = models.IntegerField(verbose_name=u"标签id", db_index=True)


class FollowQuestionGroup(models.Model):

    class Meta:
        verbose_name = u'关注问题组'
        db_table = 'api_ai_qa_follow_question_group'

    title = models.CharField(max_length=512, null=False, verbose_name=u'引导语')
    content = models.TextField(verbose_name=u'问题组([{"image_url": "xx", "question_id": 1},])', null=False, default="")
    is_online = models.BooleanField(verbose_name=u'是否在线', default=False)
    create_time = models.DateTimeField(u"创建时间", auto_now_add=True)
    update_time = models.DateTimeField(u'更新时间', auto_now=True)
    last_modified = models.DateTimeField(verbose_name=u'最后修改时间', default=timezone.now)

    @cached_property
    def contents(self):
        try:
            return json.loads(self.content)
        except:
            return []


class AIMessageGroup(models.Model):

    class Meta:
        verbose_name = u'消息组（每个问答流程为一组）'
        db_table = 'api_ai_qa_message_group'

    group_id = models.CharField(max_length=128, verbose_name=u'每次问答流程的主ID(UUID)', db_index=True)
    user_id = models.IntegerField(verbose_name=u"用户id", db_index=True)
    ass_uid = models.IntegerField(verbose_name=u"客服用户id", db_index=True)
    is_online = models.BooleanField(verbose_name=u'是否在线', default=False)
    finished = models.BooleanField(verbose_name=u'是否完成', default=False)

    is_read = models.BooleanField(verbose_name=u'已读状态', default=False)

    create_time = models.DateTimeField(u"创建时间", auto_now_add=True)
    update_time = models.DateTimeField(u'更新时间', auto_now=True)
    last_modified = models.DateTimeField(verbose_name=u'最后修改时间', default=timezone.now)


class AIMessage(models.Model):

    class Meta:
        verbose_name = u'消息'
        db_table = 'api_ai_qa_message'

    group_id = models.CharField(max_length=128, verbose_name=u'每次问答流程的主ID', db_index=True)
    user_id = models.IntegerField(verbose_name=u"发消息人ID", db_index=True)
    ass_uid = models.IntegerField(verbose_name=u"收消息人ID", db_index=True)
    msg = models.CharField(max_length=1024, null=False, verbose_name=u'消息内容')
    type = models.IntegerField(verbose_name=u'类型', default=AI_QA_MESSAGE_TYPE.BUTTON, choices=AI_QA_MESSAGE_TYPE)
    ident = models.CharField(max_length=128, verbose_name=u'回答或者回答组ID')
    data = models.CharField(max_length=1024, verbose_name=u'冗余信息')
    is_online = models.BooleanField(verbose_name=u'是否在线', default=False)
    create_time = models.DateTimeField(u"创建时间", auto_now_add=True)
    update_time = models.DateTimeField(u'更新时间', auto_now=True)
    last_modified = models.DateTimeField(verbose_name=u'最后修改时间', default=timezone.now)


class AIConsultInfo(models.Model):

    class Meta:
        verbose_name = u'AI咨询信息'
        db_table = 'api_ai_qa_consult_info'

    title = models.CharField(max_length=1024, verbose_name=u'咨询文案')
    is_online = models.BooleanField(verbose_name=u'是否在线', default=True)
    create_time = models.DateTimeField(u"创建时间", auto_now_add=True)
    update_time = models.DateTimeField(u'更新时间', auto_now=True)
