# coding:utf-8

from __future__ import unicode_literals

import dateutil.parser as parser
import time
import datetime

from api.models import Merchant, Coupon, GiftToCoupon

from django.core.management import BaseCommand
from rpc.tool.error_code import CODES

from search.utils.conversation import search_conversation_from_es
from services.unread.stat import UserUnread

from api.models.types import MESSAGE_TYPE
from hippo.models import MerchantRelevance, Doctor
from gm_types.gaia import DOCTOR_TYPE, COUPON_TYPES, COUPON_DISTRIBUTION_STATUS
from gm_types.msg import CONVERSATION_ORDER, CONVERSATION_TYPE
from gm_types.trade import COUPON_TIME_TYPE
from message.views.message import internal_message_send
from api.models.message import ConversationUserStatus

from rpc.exceptions import RPCPermanentError


def get_conversations_v3(user_ids, offset, size, order_by=CONVERSATION_ORDER.LAST_REPLY_TIME,
                         last_reply_time_start=None, last_reply_time_end=None, reply_status=None,
                         is_star=None, user_id=None, user_last_name=None, comment=None):
    """
    获取会话列表， 与v2不同的是全部走es获取conversation_ids
    :param user_ids: LIST[USER_ID]
    :param offset: INT 偏移量
    :param size: INT 每页数量
    :param order_by: 排序规则
    :param last_reply_time_start: str 最后回复开始时间筛选
    :param last_reply_time_end: str 最后回复结束时间筛选
    :param reply_status: bool 回复状态（新消息、旧消息）
    :param is_star: bool
    :param user_id: int 精确搜索用户ID
    :param user_last_name: int 搜索用户昵称
    :param comment: int 搜索备注
    :return:
    """
    if not isinstance(user_ids, list):
        raise RPCPermanentError
    if not isinstance(size, int):
        raise RPCPermanentError
    if size <= 0 or size >= 50:
        size = 50
    es_filters = {'user_ids': user_ids}
    if is_star is not None and isinstance(is_star, bool):
        es_filters['is_star'] = is_star
    if last_reply_time_start is not None:
        es_filters['last_reply_time_start_gte'] = parser.parse('{}+0800'.format(last_reply_time_start)).isoformat()
    if last_reply_time_end is not None:
        es_filters['last_reply_time_end_lte'] = parser.parse('{}+0800'.format(last_reply_time_end)).isoformat()
    es_query = dict()
    # if search_content is not None:
    #     es_query = {
    #         'comment': search_content,
    #         'user_last_name': search_content,
    #     }
    if user_id is not None:
        es_query['user_id'] = user_id
    if user_last_name:
        es_query['user_last_name'] = user_last_name
    if comment:
        es_query['comment'] = comment
    es_sort_type = order_by
    # 全部
    es_result_total = search_conversation_from_es(offset=offset, size=size, filters=es_filters,
                                                  query=es_query, sort_type=es_sort_type)
    # 已回复
    es_filters['multi_filter_status'] = {
        'status': True,
        'user_ids': user_ids,
    }
    es_result_reply = search_conversation_from_es(offset=offset, size=size, filters=es_filters,
                                                  query=es_query, sort_type=es_sort_type)
    # 未回复
    es_filters['multi_filter_status'] = {
        'status': False,
        'user_ids': user_ids,
    }
    es_result_not_reply = search_conversation_from_es(offset=offset, size=size, filters=es_filters,
                                                      query=es_query, sort_type=es_sort_type)
    # es_result = search_conversation_from_es(offset=offset, size=size, filters=es_filters,
    #                                         query=es_query, sort_type=es_sort_type)
    if reply_status == None:
        es_result = es_result_total
    else:
        if reply_status:
            es_result = es_result_reply
        else:
            es_result = es_result_not_reply
    _conversation_ids = es_result['conversation_ids']
    cus_ids = []
    for conversation_id in _conversation_ids:
        sub_ids = ConversationUserStatus.objects.filter(
            user_id__in=user_ids, conversation_id=conversation_id
        ).values_list('id', flat=True)
        cus_ids.extend(list(sub_ids))
    conversation_user_status = ConversationUserStatus.objects.filter(id__in=cus_ids).order_by('-last_reply_time')

    conversation_info_list = [cs.conversation.conversation_info_v3(cs, user_ids) for cs in conversation_user_status]
    conversation_id_to_user_id = {cs.conversation_id: cs.user_id for cs in conversation_user_status}
    for c in conversation_info_list:
        default = 1 if c['is_new'] else 0
        c['unread_num'] = UserUnread(conversation_id_to_user_id[c['id']]
                                     ).get_conversation_unread(c['id'], default=default)
    return {
        'conversation_list': conversation_info_list,
        'total_count': es_result_total['total_count'],
        'reply_total': es_result_reply['total_count'],
        'not_reply_total': es_result_not_reply['total_count']
    }


class Command(BaseCommand):
    DOCTOR_BUSINESS_CHANNAL_ID_ONLINE = 35

    # only check merchant id can use
    availableMerchantIDList = []

    # check merchant id and coupon can use
    availableMerchantIDListWithCoupon = []

    onlineTestuid = 33703995

    # content = [["area", "merchant_name", 6477328345687146383, 533, 0, "福建最大的整形机构在线独家补贴，领取红包为美减压，爱美就到爱美尔"]]

    c = a = [['华东', '武汉五洲整形美容医院', 6329571025797364148, 60793, 0, '宝宝您好，七夕快乐！感谢关注五洲整形，送您一张无门槛优惠券，有任何不明白的可以随时联系我们哦！'],
             ['华东', '武汉中翰整形外科医院', 6424920729401199852, 60792, 60794,
              '您好，七夕快乐！中翰医学美容，看您有浏览我们医院的项目，亲亲是想了解哪方面呢？有什么疑虑也可以提出来，平台用户还有专享红包，请记得领取哦'],
             ['华东', '武汉韩辰医疗美容门诊', 6328834522641169708, 60804, 60805,
              '您好，七夕快乐！韩辰医学美容，看您有浏览我们医院的项目，亲亲是想了解哪方面呢？有什么疑虑也可以提出来，平台用户还有专享红包，请记得领取哦'],
             ['华东', '武汉诠美医疗美容门诊部', 6328834462117380177, 60830, 60828,
              '您好，武汉诠美祝您七夕快乐。1.十余年本地口碑机构放心变美2.私信领尾款红包下单更优惠3.外地顾客报销路费4.预约到院即有专家面诊'],
             ['华东', '武汉奥黛丽医疗美容', 6329570573982637925, 60821, 0,
              '奥黛丽整形医院（武汉总院）#祝您气息快乐#，①两人同行任意消费赠送水光针，②主动私信领取皮肤管理项目，有任何疑问可以私信了解哦~'],
             ['华东', '武汉美基元医疗美容医院', 6328834476751308492, 60820, 0, '美基元23周年庆， 欢迎到院享受福利，超大额无门槛红包，快来领取吧~祝您七夕快乐'],
             ['华东', '华中科技大学同济医学院医院', 6335684117300101907, 60823, 60824,
              '您好，同济医学院医美科祝您七夕快乐，看您有浏览我们医院的项目，亲亲是想了解哪方面呢？有什么疑虑也可以提出来，平台用户还有专享红包，请记得领取哦'],
             ['华东', '武汉华美整形外科医院总院（中整协修复基地）', 6329570416843092905, 60831, 0,
              '宝宝您好，感谢关注华美整形，祝您七夕快乐，送您一张无门槛优惠券，有任何不明白的可以随时联系我们哦！'],
             ['华东', '武汉美莱医疗美容医院', 6328834423693338627, 60791, 0, '宝宝您好，七夕快乐！感谢关注美莱整形，送您一张优惠券，有任何不明白的可以随时联系我们哦！'],
             ['华东', '长沙中山医院', 6671703063285281559, 60859, 0, '宝宝您好，七夕快乐！感谢关注中山私密整形，送您一张无门槛优惠券，有任何不明白的可以随时联系我们哦！'],
             ['华东', '长沙三和整形美容', 6329570405292002435, 60237, 60238, '宝宝您好，七夕快乐！感谢关注三和整形，送您一张优惠券，有任何不明白的可以随时联系我们哦！'],
             ['华东', '武汉艺星医疗美容医院（汉口院）', 6445571571951100091, 60360, 0,
              '七夕快乐! 您有一份红包还没领取哟！这里是艺星整形。亲是想了解哪方面呢？有什么疑虑也可以提出来。请记得领取哦期待您的回复！'],
             ['华东', '武汉艺星医疗美容（眼鼻旗舰店）', 6379868908551212219, 60359, 0,
              '宝宝，七夕快乐！您有一份红包还没领取哟！这里是艺星整形。亲是想了解哪方面呢？有什么疑虑也可以提出来。请记得领取哦期待您的回复！'],
             ['华东', '武汉美都医疗美容门诊部', 6328834448271997674, 60865, 60866,
              '【武汉美都整形】宝宝您好，您的专享无门槛券已到账！看您有浏览我们医院的项目，亲亲是想了解哪方面呢？有任何不明白的可以随时联系我们哦！'],
             ['华东', '武汉新至美医疗美容', 6328834434883766693, 58033, 0,
              '亲爱的小主，七夕快乐！欢迎来到武汉新至美，看到您有关注咋们家的项目，是想了解什么想呢呢？很高兴能够为您解答相关的疑惑哟！咱们为平台用户还准备了专享红包，请记得领取哦"'],
             ['华东', '武汉壹加壹医疗美容医院', 6374536413001558630, 60800, 0, '宝宝您好，七夕快乐！感谢关注壹加壹整形，送您30元无门槛优惠券，有任何不明白的可以随时联系我们哦！'],
             ['华东', '上海薇琳医疗美容', 6328834454294911932, 60343, 60342,
              '#欢迎咨询上海薇琳中美韩（国际）连锁整形医院#暑期超值价①水光补水2ml特价秒杀288元②"瘦身季"美国酷塑溶脂首点2980元③黄金微雕吸脂下巴仅6800元④买一送二"热玛吉面+眼"仅需16800元⑤买一送一"全脸超皮秒"2980元⑥进口除皱瘦脸白菜价880元⑦眼两项“初恋眼+内眼角”仅需4980元更多“颧骨颌面，鼻修复、眼修复、其它整形等咨询客服小姐姐鸭~”'],
             ['华东', '上海百达丽医疗美容门诊部', 6328834501472520725, 60328, 0,
              '\r终于等到你 欢迎关注上海百达丽。 \r①原九院医生团队\r②日记均为本院真实用户体验\r③中整协指定脂肪移植基地\r④2019年度十大鼻部名院\r⑤主打眼鼻脂肪私密外科及皮肤针剂项目，告诉我们你最感兴趣的项目，给您私发项目红包！！！\r\r上海百达丽8月特惠倒计时~\r低至五折 必买推荐：\r①切开双眼皮630元\r②眼部两项1830元\r③祛眼袋1680元\r④双眼皮修复4640元起\r⑤沈国雄院长鼻中隔鼻综合38400元'],
             ['华东', '上海首尔丽格医疗美容医院', 6328834457289693664, 61235, 61234,
              '长沙“减肥女孩”已经可爱到热搜了你的变美计划还不安排起来。①下颌骨切除or颧骨缩小 活动价29800还有1万元颧骨固定费，5千元下巴补助金，直接当场抵扣②双眼皮（切开+祛皮+祛脂+翘睫）  活动价3980颜值小改变 气质新体现③硅胶隆鼻＋耳软骨＋鼻中隔 活动价价19800④金柱院长全脸脂肪填充 活动价7800⑤面部下垂矫正 活动价6800⑥酷塑小点位2980不限购更有800元轮廓修整面诊费，500元CT补助金，只等你来，想要预约的小仙女们快快浏览医院主页下方项目，联系客服预约吧'],
             ['华东', '上海铂曼医疗美容门诊部 ', 6328834432585202327, 61096, 0,
              '七夕热卖推荐①双美胶原蛋白 全国御用讲师手打价5580元②乔雅登玻尿酸 乔雅登风云讲师手打价5980元③酷塑冷冻溶脂 艾尔建讲师评估-首点2980元④悦升线苹果肌复位+快翎线中下面部   ###抗衰CP线&打包价16500元###⑤祛眼袋2980元 眼综合4项5800元             ###本院热玛吉###⑥#支持四维验真,单人单头,足量发数#'],
             ['华东', '上海艺星医疗美容医院', 6329570408508957823, 61066, 0, '空瓶焕肤&七夕情人节持任意化妆品空瓶，满880减300留言客服领取红包活动时间：8月16日-8月31日'],
             ['华东', '上海愉悦美联臣医疗美容医院有限公司', 6328834438998297377, 61229, 61230,
              '上海愉悦.美联臣8月愉悦轮廓节 基础下颌角 18800元起 免术前拍片 mini颧骨 18800元起 免术前拍片 (颌面优惠价格 每月限前5名颌面缴费者，详情请联系客服） 国产除皱20u 199元 进口除皱20u 666元 伊婉c冰点价 699元限购一次 伊婉v冰点价 980元限购一次 （新客及未注射过的客户可下单） 买瘦腿/瘦肩1600元 赠一次唇毛/腋毛脱毛 买瑞蓝2020元 赠一次舒敏之星 买婕尔3980元 赠一次双重瓷白 买馨妍2ML4800元 赠一次双重瓷白 买乔雅登雅致5299元 赠一次调Q激光 买乔雅登极致7999元 赠一次热拉提PLUS 买瑞蓝丽缇8800元 赠一次超皮秒 四代热玛吉 眼周/全面部足量 5800元 四代热玛吉 全面部足量 两人团购价7999元 更有超多美肤项目低价到没朋友~ 快来下单囤货吧??~错过等一年！ 宝宝快快下单预约吧~~'],
             ['华东', '上海大麦医疗美容门诊部', 6328834429917647624, 59895, 0,
              '大麦微针植发.特惠活动"植男""植女"变形记来啦下单预约可享受 到院术后深层理疗2次免费试种200单位 ！更有10000-500的红包可以使用哦'],
             ['华东', '杭州大麦整形外科门诊部', 6417270138088734617, 61092, 0,
              '大麦微针植发.特惠活动"植男""植女"变形记来啦下单预约可享受 到院术后深层理疗2次免费试种200单位 ！更有10000-500的红包可以使用哦'],
             ['华东', '南京科发源医疗美容有限公司', 6541984561818037227, 61093, 0, ''],
             ['华东', '上海天大医疗美容医院', 6328834467477712267, 61091, 0,
              '上海天大整形，8月欢乐购①无针水光（8月特惠）51元②深蓝射频眼周         299元③热拉提PLUS           999元④芭比眼综合案例享      4800元⑤进口假体隆鼻案例享    3888元⑥双美胶原蛋白0.5ml     2680元⑦热玛吉闺密双人行       7900元/人     更多活动详情请咨询客服宝宝~~'],
             ['华东', '上海华美医疗美容医院', 6328834441472921182, 61236, 0,
              '文案：暑期变美倒计时...充值满减活动充多少抵多少最高立减5000元注射瘦脸低至3折皮肤美容低至1折双眼皮3天拆线哦悄悄变美就趁现在做新校园最靓的仔'],
             ['华东', '杭州珈禾医疗美容医院', 6433496798190426143, 61237, 59459,
              '【七夕来袭 珈禾买单】约会之前#先变美 ★到院报销路费+送价值1980元皮肤项目（3选1）★眼鼻吸脂案例招募，变美无忧★舒敏蜜语199元/次★保妥适瘦脸针1880元/次★高光热玛吉：2980（拼团300发）★鼻综合3项：6800★眼综合4项：4800★手臂吸脂：3980★半肋鼻综合6项目：19800赶快@出你心仪的产品，珈禾医生创立成立医院17年，所有产品均支持检验'],
             ['华东', '上海时光整形外科医院', 6328834558070481766, 55488, 55485,
              '宝宝您好，七夕快乐！感谢关注上海时光整形，送您一张满减优惠券，有任何感兴趣的项目可以随时联系我们哦！'],
             ['华东', '上海光博士医疗美容门诊部', 6328834508275693538, 60591, 60590,
              '七夕特惠，全场项目领取红包立减80元，更有特惠项目全场抢购1.国产除皱20单位 198元 2.光子嫩肤闲时卡 199元 3.皮秒嫩肤+C6祛黄 399元4.M22光子嫩肤399元 5.私密漂红 380元 6.水光2次+C62次 666元 7.切开双眼皮 980元 8.基础水光年卡1080元 9.蜂巢皮秒全面部 1200元 10.菲蜜丽私密紧致 1280元11.伊婉V玻尿酸 1200元 12.热拉提全面部 1980元 13.自体脂肪隆胸 4900元 14.乔雅登雅致 3980元 15.热玛吉全面部 5999元送热拉提一次七夕限定，别错过哦！'],
             ['华东', '上海奉浦医疗美容', 6328834461181935897, 59127, 59135, '宝宝您好，七夕快乐！感谢关注上海奉浦医美，送您一张满减优惠券，有任何不明白的可以随时联系我们哦！'],
             ['华东', '郑州艺德雅医疗整形', 6340741263578422625, 59180, 59228,
              '亲爱的 七夕来临，现艺德雅针对本周双休日推出优惠小活动。优惠如下：①凡本月到院者，每位宝宝到院可享9元脱腋毛十次。若多人同行（≤2人）一人收费，同行者皆免单。让您自信整个夏天，对尴尬说拜拜~~②到院客户即送无菌医用补水面膜一盒10贴，让您整个夏天皮肤水润嫩滑'],
             ['华东', '郑州善雅医疗门诊部', 6670976491028744546, 58447, 58446, '正在为女神派发红包～善雅女神专属红包可以直接抵尾款哦，现在咨询更有机会预约院长亲自手术哟！'],
             ['华东', '河南汇星整形医疗美容', 6328834445352728936, 61242, 0, '浪漫七夕 关注河南汇星整形 秒杀不止5折！惊喜不断！'],
             ['华东', '河南中医药大学第一附院医院', 6341949409768536792, 61256, 44487,
              '七夕收到这份神秘礼物，一定会让你惊喜╰(*°▽°*)╯活动时间：8月15日~9月15日满1000减50满10000减1000私信客服即可领取红包~留下你的变美愿望，我们来帮您实现!!!'],
             ['华东', '郑东安琪儿医疗美容门诊部', 6328834450230692023, 61262, 61263,
              '【安琪儿七夕专场】         ①50元宠粉卡---防晒喷雾一瓶，医用面膜一盒，补水净肤年卡1张，修眉年卡，50元抵用券和1000元抵扣券各一张。【另送水光针或品牌玻尿酸一支】    ②199元星耀卡---12盒保湿医用面膜，10倍膨胀金，1元换购除皱                         ③全身吸脂仅需7600元--【来院送玻尿酸一支】      ④2980元眼综合学生专享价    ⑤全面部热玛吉闺蜜同行6800元/人     更多活动详情请咨询客服宝宝~~'],
             ['华东', '巨星医疗整形连锁（紫荆山分院）', 6494489572083337095, 61260, 61264,
              '不“七”而遇，惺惺相“夕”“脂”为遇见你，从“填”开始，让爱不一“斑”A.充值5800，账得7800【另送小海豚音响+会员日1次】B.充值12800，账得18800【另送满天星投影灯+会员日2次】C.充值22800，账得35800【另送萌宠屋氛围灯+会员日3次】D.充值35800，账得60000【另送网红化妆品收纳盒+会员日4次】E.充值55800，账得110000【另送多功能果蔬消毒机+会员日5次】'],
             ['华东', '河南gy广运整形（美博士）医疗美容', 6537149559146021770, 57919, 57920,
              '1、七夕热卖推荐A、0元双眼皮B、小切口拉皮2680C、私密紧致1188D、无骨鼻5666C、一针降颧骨39882、浪漫七夕预付777到院消费可抵1314（外科手术）'],
             ['华东', '郑州美莱医疗美容医院', 6382856553178943526, 22107, 61271,
              '郑州美莱七夕特价热卖花7.7元抵100元无门槛现金使用券①瘦脸针（100单位） 599元②益术康水光针 128元③脱唇毛/腋毛年卡 69元④超皮秒祛斑 999元⑤热玛吉面部 5999元⑥眼综合 3679元⑦鼻综合 4999元'],
             ['华东', '上海正璞医疗美容门诊部', 6688752474595008350, 61246, 61248,
              '感谢集美的关注 \r上海正璞医疗周年庆携手更美 【独家补贴】上线啦~ \r下单预约可享受 玻尿酸补水一次 \r国产除皱单部位 199元/单部位 \r超光子嫩肤666元\r国产瘦脸 780元/100u \r艾莉薇 1880元起\r本店所有项目 私聊可享超值心动红包 \r切开双眼皮 980元 \r眼综合（双眼皮+内眼角）2980元\r假体隆鼻 1980元\r全面部自体脂肪填充 5800元 \r手臂吸脂 2980元 \r更有超多美肤项目低价到没朋友~ \r快来下单囤货吧??~错过等一年！ \r集美们快快下单预约吧~~'],
             ['华东', '安徽维多利亚整形外科医院', 6481430998646024191, '58808', '58807',
              '\u3000\u3000安徽维多利亚美颜季专场特惠，先别着急退出哦~耐心看完，了解秒杀的惊喜优惠!\u3000\u30001、超微小气泡特惠19.8元!\u3000\u30002、鱼尾纹除皱特惠99元!\u3000\u30003、美白补水微针特惠299元!\u3000\u30004、瘦脸针100U特惠680元!\u3000\u30005、爱芙莱玻尿酸两支特惠999元!\u3000\u30006、艾莉薇专场特惠3支6666元!\u3000\u3000回复对应数字详细给你介绍哦~目前咨询到院即送水润生肌补水一次!'],
             ['华东', '南京柠檬', 6556456340485616955, 61270, 61275, 'ᐅ整形产品超大券：满10000减1000七夕专享礼 定制你的美#私信get更多福利~#'],
             ['华东', '广州建国医院', 6653595066751288394, 61265, 61269,
              'ᐅ浪漫七夕·给你宠爱1.韩式完美处女膜修复：880元2.微创紧缩修复术：980元3.小阴唇双侧肥大修复：3400元了解更多活动详情可私信联系我们~（外地顾客私信到院还送车费报销礼哦）'],
             ['华东', '合肥西美医疗美容', 6516220829414481101, 60504, 0,
              'ᐅ七夕福利 超低价纹眉多种眉型可选 无隐性消费医学纹绣一口价 仅需368元ᐅ充值美惠 畅享全院项目即日起至8月31日，西美开启充值特惠❶充3000得4000    ❷充5000得7000❸充10000得15000 ❹充20000得30000还有超多除皱、补水年卡、闺蜜卡免费送炙热八月 爱上合肥西美'],
             ['华东', '江苏施尔美', 6328834446480980730, 60600, 0, '七夕抄底 钜惠给你，二十周年庆典，足量正品可验瘦脸针仅售380元！眼综合2680元！'],
             ['华东', '郑州华领医疗美容医院', 6329571039579834406, 61252, 0,
              '七夕超强力度 钜惠来袭限时购来了！①国产瘦脸针100U：520元②内吸祛眼袋（无痕）：520元③520元女神卡（五选三）：单部位国产除皱/法国牛奶酸/焕颜水光/瑞可丽医用面膜一盒/GSD深层补水月卡④M22全面部嫩肤+法国牛奶酸+焕颜水光：777元⑤吸脂单部位（负压吸脂）：777元⑥定制鼻综合三项到手价7777元⑦韩式精吸手臂加上背：7777元⑧自体脂肪填充单部位（结构式填充） ：777元变美又省钱 专享特惠放心美另外宝宝现在新客到院可免费领取医用面膜一盒和玫瑰花一支哦，宝宝对项目有什么不清楚的可以咨询华小领哦'],
             ['华东', '河南缔莱美整形医院', 6582094348182189887, '61267', 61261,
              '七夕来袭 小仙女享受超低价 限时抢购开始啦①国产瘦脸针100U：399元②韩式双眼皮（无痕）：2388元③picoway超皮秒：580元④热玛吉全面部：5999元⑤吸脂单部位：580元⑥定制鼻综合三项到手价4800元⑦韩式精吸大腿环吸：5999元⑧自体脂肪填充单部位 ：520元想变美 缔莱美 放心做整形另外宝宝现在到院可免费领取医用面膜一盒宝宝对项目有什么不清楚的欢迎咨询哦~'],
             ['华东', '美国容丽妍南京分院（中国区旗舰院）', 6581857546452205154, 61273, 61274,
              '欢迎咨询美国容丽妍南京分院~~~\r抗衰精品，热浪来袭\r热玛吉5代强势登陆\r体验尝鲜价1980元，皮肤套餐1580元\r私信客服小姐姐即可获得“上门礼遇”\r韩国小气泡、无针水光、脱唇毛/腋毛，任选其一'],
             ['华东', '南京新秀植发', 6526279491545778039, '61284', 0, '七夕不想‘’光‘’，来新秀植发，让爱 更‘’密‘’。'],
             ['华东', '郑州虎珂整形', 6574840536383913572, 57006, 57005,
              '【虎珂整形】七夕鹊桥惠：520元四选三：国产瘦脸针（50U）1次进口果酸医学美肤 2次Q弹桃花肌1次水光奕彩3次1314元五选三Botox眼尾除皱2次小白瓶微针1次荣耀光子瓷白肌美国悍马激光祛斑1次维密盆底肌修复 3次'],
             ['华东', '青岛华颜美', 6417268770800266963, 60131, 60129,
              '1.华颜美院庆盛典活动中,新客预约到院消费即可领取mini粉红豹一只或者星巴克杯一个2.预约到院消费任意金额即可享受韩国水美人一次，还可领取医用面膜一盒3.单部位除皱仅需199，足量瘦脸针仅需499！4.暑期特惠眼鼻综合，小眼综合仅需2966，小鼻综合也仅需2966！双人同行还可减1000元5.全新一代热玛吉（第五代）强势来袭！对抗地心引力，紧致提升，年轻十岁6.有任何问题都可直接私信客服！'],
             ['华东', '青岛华韩整形美容医院', 6451001379719868551, 58873, 54463,
              '青岛华韩正在为女神派发红包～华韩女神专属红包原价的基础上可以直接抵尾款哦，现在咨询更多超低优惠套餐等您来约！'],
             ['华东', '青岛诺美德医疗美容', 6329571820945023082, 60585, 61288,
              '您好，看您这边对咱们医院的项目比较感兴趣，因为每个人的自身基础不一样，适合的方式也是不一样的，您这边可以简单的说一下您的基本情况，我给您针对性的建议哦，会为您提供一对一专属服务，现在医院正是活动火爆期，价格特别优惠哦。'],
             ['华东', '济南吉芙医疗美容门诊部', 6328834436649571169, 49987, 47225, '正在为女神派发红包～吉芙女神专属红包可以直接抵尾款哦，现在咨询更有机会预约院长亲自手术哟！'],
             ['华东', '济南艺星医疗美容医院\t', 6361823477556696944, 61113, 0,
              '济南艺星  眼部、鼻部、脂肪填充体验官火热招募中···爆款项目师生特惠8.5折起。手术项目每满2000送500！单部位除皱398元，大分子玻尿酸低至1280元/支 ！更多疑问欢迎在线咨询'],
             ['华东', '合肥台美丽格医疗美容 ', 6566596510962754122, 61268, 61272,
              '浪漫七夕你男朋友给不了你的，我们给你啊~~贵气小姐姐必选套餐①鼻综合8项8999元 ②肋骨鼻综合秒杀价14980！③天生尤物必备款：假体丰胸15800！④自体丰胸5800！案例模特眼综合1000元，鼻综合2000元在线招募中!即日起手术项目来院专车接待，到院免费领取价值198元皮肤项目！'],
             ['华东', '原韩整形', 6432129803196862117, 61281, 61285,
              '亲爱的宝宝，原韩夏日变美计划火热进行中❗现推出以下活动 ❗ 1.仅9⃣9⃣8⃣0⃣即可购买皮肤微整年卡，享10种爆款产品全年畅打❗ 2.仅9⃣9⃣8⃣0⃣起即可购买全面部改造王牌项目套餐❗ 2.凭学生证/教师证，到院所有产品项目费用专享9⃣折优惠❗ 3.老顾客带新顾客成功消费可以金额的1⃣0⃣%，免费享受任何项目❗ 凡到院顾客，任意消费即可免费体验等离子嫩肤一次或韩国小气泡一次❗'],
             ['华东', '济南海峡美容整形医院', 6355971291014606078, 61289, 0,
              '“爱在七夕 美在海峡”终于等待你~上门即送价值198元医用面膜一盒+价值198元水光裸肌补水1次超级大补贴充5000送5000、充10000送10000院长免费亲诊20年经验副主任医师领衔塑美诚意如何？随时等你来撩~'],
             ['华东', '南京米尚恩', 6329570856301265250, 61291, 61290,
              '终于等到你 小仙女欢迎关注米尚恩（中韩联合抗衰机构）1.品牌水光2ML补水2882.瘦脸针秒杀498（效果好）3.法国LPG明星瘦身管理498（单部位体验）2.法国进口中胚层菲洛嘉水光3ML9803.2020最火热玛吉四代全面部49804.初恋少女眼（切开+去皮+去脂+内眼角）58005.双心脸填充 3999'],
             ['华东', '南京美贝尔', 6328834490479193411, 61276, 61278,
              '"༺ۣۖ甜蜜七夕，给你宠爱༻美贝尔七夕疯狂钜惠预约来院，价值1680元活动大礼包免费送：1.单部位除皱1次2.焕氧美肤1次3.visa皮肤检测1次七夕有爱，甜宠升级，美贝尔抄底秒杀，福利送不停，等你来抢哦！"\r'],
             ['华东', '郑州东方整形美容医院有限公司', 6328834454051651566, 61250, 61255,
              'ᐅ浪漫七夕·东方给你专宠                            唇毛/腋毛包年卡23元              私密 SPA+比基尼美雕+6项检测  23元                                   彩光嫩肤 1 次230元             东国水光针 1 次 230元          国产除皱针 1 次（单部位，额纹除外）230元                        瘦肩针/瘦腿针（国产）1880         清新美眼综合：（普通全切双眼皮+去皮去脂+开内眼角）2300        不开刀祛眼袋：（内路微创去除）2300元                         极速动感复合丰胸：（曼托毛面圆盘+双平面技术+独家极速无创剥离技 术+肉毒素胸大肌放松） 23000 到院大礼包：到院并分享院庆图至朋友圈，即可领取 3980 元                    到院大礼包： 院庆定制网红咖啡杯 1 套或 398 元医用修复原液喷雾 1 瓶+3000 元现金券（1500 元闺蜜现金券+1500 元本人专用现金券，美外/皮肤/微整各 500 元；                                           '],
             ['华东', '南京康美', 6328834424624479013, 61280, 61283,
              '终于等到你   ▎康美整形 ▎18周年庆✿180元=进口玻尿酸✚神仙水年卡✚1000元现金券☛50倍增值9000元★名额仅剩39位 先到先得哦★'],
             ['华东', '青岛艾菲医疗美容 ', 6589753748900393876, 51349, 0,
              '“爱在七夕 美在艾菲”终于等待你~玻尿酸、瘦脸针、水光针..下单咨询客服有惊喜~ 闺蜜同行线上预约赠送冬季温暖礼包，惊喜多多，优惠多多，详情咨询客服小姐姐'],
             ['华东', '青岛伊美尔国宾整形外科医院', 6474155179343091800, 61057, 0,
              '【青岛伊美尔七夕奢宠周】甜蜜七夕♥给你宠爱★ 7赞有礼小气泡升级版 77元埋线重睑 777元吸脂单部位 777元★ 秒杀来袭切开双眼皮 1777元假体隆鼻 1777元黄金微雕 1777元自体脂肪填充 1777元皮秒全面部美肤 1777元法思丽玻尿酸 1777元★ 消费立返一次性预存/消费每满1万，即享777元返现，上不封顶咨询在线客服小姐姐，解锁更多特惠~'],
             ['华东', '南京医科大学友谊整形外科医院', 6411890656733565465, 56073, 0, ''],
             ['华东', '济南韩氏整形外科医院', 6456699846714840470, 61302, 61304, '专属于小仙女的红包到啦，快快签收哦~上帝欠您的，韩氏还给您！'],
             ['华东', '济南美辰医疗美容整形', 6495550423368267548, 61306, 0,
              'Hello宝宝好呀感谢关注新氧口碑机构济南美辰医疗美容，新氧甜蜜七夕活动开始啦。惊喜项目超值优惠，大额红包领到手软，护肤项目低至一元，回复私信我来独宠你！'],
             ['华东', '济南妇儿医院', 6447387965231542525, 61305, 0,
              '私密呵护，开启您的性福之窗。亲爱的您好！我是济南妇儿医院的客服，很乐意为您提供私密整形咨询和预约服务！现在预约即享受同城接送，到院赠送9项妇科检查～亲亲想要了解什么项目呢？'],
             ['华东', '杭州美莱医疗美容医院', 6328834497466989650, 61254, 0,
              '亲亲，欢迎您进入杭州美莱整形医院。 【周年庆限量回馈】玻尿酸¥388， 除皱针¥499，光子嫩肤¥520，进口瘦脸针¥1680，菲蜜丽¥2980，双眼皮¥1280，鼻综合¥6600，全身吸脂（任选四部位）¥6999，热玛吉¥7500，5G复合隆胸¥9800。在线预约到院，可享参加抽奖及专车接送~'],
             ['华东', '南京凤凰岛整形外科', 6340464445805873076, 61294, 61298,
              '您好，感谢关注凤凰岛整形外科！①我获得最具影响力美容整形机构、南京市玄武区示范诊所、明星医院等各种殊荣！②专注眼部整形十四年之久的胡国强院长亲诊！③限时菲洛嘉688，润百颜白紫288、伊婉v厂家回馈1280限购一支、英诺小棕瓶1180送皮秒④凤凰岛8月抗衰季精品套餐，打造您的面部年轻化，送您10年前的面容！凤凰岛承诺：所售产品及项目均正品保障，当面可验。'],
             ['华东', '杭州丽脂医疗美容', 6379635575154122552, 61253, 61257, '七夕甜蜜暴击，多重满减活动'],
             ['华东', '艾尔莎·奥拉克医疗美容门诊部', 6476030498811491675, 54976, 60692, '点击领取您的无门槛红包'],
             ['华东', '合肥新地崔劲松医疗美容', 6328834430857228931, 61238, 59552,
              'Hello 小仙女，七夕快乐！不止七夕，新地崔劲松爱你朝朝夕夕今日私信客服免费领取价值 198 元皮肤巨补水项目哟，回复数字[1]了解具体详情~'],
             ['华东', '合肥华美整形外科医院', 6328834490659525492, 60483, 60484,
              '亲亲，您好。感谢关注华美整形，合肥华美整形外科医院是目前安徽省规模最大的医美整形外科医院，有任何问题可以咨询我哦，亲有空可以拍下想改善的部位照片给我们，这边给您分析下改善方案呢！~'],
             ['华东', '合肥恒美整形外科医院', 6328834488050755184, 59220, 61266,
              '欢迎咨询合肥恒美整形医院#年中大促超值价 ①水光补水2ml特价秒杀199元 ②"瘦身季"衡力瘦腿针1780元 ③全国热销"埋线提升"仅需4999元 ④全国热销"热玛吉"仅需15800元 ⑤全国热销祛斑"全脸皮秒"1980元 ⑥国产瘦脸白菜价488元 ⑦国产除皱75元'],
             ['华东', '合肥艺星医疗美容医院', 6342312590370150012, 60003, 0,
              '您好，欢迎选择合肥艺星，15年19城23院连锁，针剂保真品牌全，引进热玛吉等正版进口仪器适应症广泛，由副高级专家技术领衔，超20年经验资深麻醉医师配台，每一位医护人员持证上岗，确保医疗行为安全规范。我们承诺只推荐适合您的项目，绝不为成交而夸大效果，选艺星选安心！'],
             ['华东', '合肥白领安琪儿医疗美容', 6330588843795906116, 61279, 61282,
              '亲亲你好，看到您在浏览我们的页面，是想了解什么项目呢？有任何问题都可以咨询我们的客服小姐姐哈，很希望可以帮到您'],
             ['华东', '杭州华山连天美医疗美容医院', 6328834550013200855, 60516, 0, '7夕活动开始啦，宝宝们有心仪的项目可以领券下单了哦！'],
             ['华东', '杭州健丽', 6375622511257798276, 61241, 0, ''],
             ['华东', '杭州天鑫', 6679646006415478064, 61313, 0,
              '亲爱的顾客，天鑫七夕狂欢季，等待您的到来~特价项目：1、艾薇蓝水光针￥1992、衡力瘦肩/瘦腿针￥11993、全面部热玛吉600发 限时拼团活动￥5999更多项目优惠，记得私信我们客服小姐姐哦~'],
             ['华东', '浙江艾博医疗美容', 6549130697758602613, 60636, 60635,
              '七夕追爱季艾博＆甜蜜攻略；您的美丽👉我来守护，六重礼遇甜蜜助攻收割男神！上门有礼：海量医用级面膜免费送！红包派送：最高立减1314元！消费即赠：纳米美白太空舱、超皮秒、进口除皱包年、私密少女嫩红等速美项目消费满额即赠!超低加购：进口玻尿酸、超皮秒、进口瘦脸针、菲洛嘉、纳米美白太空舱、逆龄提升术、少女私密粉嫩……加购价仅需￥520起~姐妹有礼：闺蜜同行各送百元抵用券！'],
             ['华东', '南京芬迪医美', 6328834530547386754, 61310, 61308,
              '芬迪医美七夕特惠♥♥☞☞满10元立减10元☚☚☞☞满500立减50元☚☚☞☞满1000立减120元☚☚☞☞满2000立减288元☚☚爱你！！让你变美！！啾咪~~'],
             ['华东', '深圳弘美医疗美容门诊部', 6328834436033005523, 60848, 60849,
              '宝宝您好，感谢关注弘美整形，送您两个无门槛优惠券，有任何不明白的可以随时联系我们哦！现在咨询即可领取礼物哟：1、水氧活肤；2、脱毛 1次；3、小气泡深度清洁补水；4、光子嫩肤 5、专家面诊整形形象设计一次；以上项目5选一，直接回复礼物数字登记预约即可，豪礼多多，无套路完全免费,外地用户还可以享受路费减免'],
             ['华东', '重庆华美整形美容医院', 6329571006155434729, 59714, 59713,
              '重庆华美整形医院感谢您关注本院整形项目，送您一个红包可叠加更美红包一起使用，做项目更优惠哟！有项目上想咨询了解的请随时联系'],
             ['华东', '成都润美玉之光医疗美容门诊部', 6372323510777303498, 60626, 0,
              '亲爱的，欢迎咨询成都品牌润美玉之光，西南区最受肯定的医美品牌，数十年来受过央视（CCTV-1）专访，拥有14项国家专利技术，博士领衔专家团队。 我是您的私人美学顾问，七夕有惊喜活动哦，请问您想咨询什么项目呢？'],
             ['华东', '成都军大整形美容医院', 6328834424494408282, 61176, 0,
              '您好，“成都军大医院”欢迎您！ 我院为中国人民解放军、西部战区空军医院技术指导医院 • 中整协分会指定西南教学示范基地 • 并由中国平安;中国人保承保西南地区医美保险官方指定首家合作机构，我是专属您的客服，接下来我将为您一对一提供咨询服务！'],
             ['华东', '成都中医大银海医美', 6456090571630042287, 53015, 52170,
              '银海中国眼，让美更经典，欢迎来到国家三级医院，眼整形引领者，成都中医大银海医美。银海美颜团，邀你七夕成团，眼鼻整形、埋线提升5折起，热拉提年卡2999、光电年卡2999、玻尿酸年卡3999、菲洛嘉年卡8980。'],
             ['华东', '四川省人民医院医疗集团友谊医院', 6382427920501107262, 60330, 60338, '2020欠你的情人节，七夕都补给你红包抵扣现金，就要偏爱您'],
             ['华东', '\n四川美莱医学美容医院\n', 6328834446543884174, 0, 0, 'hi，小仙女，美莱暑期狂欢季，优惠多多，私信即可领取面膜一盒，赶紧联系客服小姐姐吧'],
             ['华东', '成都武侯格莱丽医疗美容门诊部', 6418755372676294081, 0, 0, '亲亲，格莱丽暑期狂欢优惠多多，还可报名参加案例招募活动，医生面诊后入选案例模特可享0元整形活动哦~'],
             ['华东', '刘晓伟', 6328834883443497610, 60872, 60870,
              '您好，看亲有关注我们刘晓伟院长，刘院是第三军医大新桥医院整形外科博士，毕业于北京协和医学院暨清华大学医学院，在国内外的整形外科杂志上都发表过学术文章近十篇，拥有四项整形方面专利，做的是医生的口碑技术，很多做过手术的亲带自己的亲戚朋友过来做，效果都非常满意。 你是想改善哪个部位呢？'],
             ['华东', '重庆时光', 6328834442743859892, 61214, 60547, '你好，看你有浏览我们医院的项目，亲亲是想了解哪方面呢？有什么疑虑也可以提出来，我都可以帮你解答的[眨眼]'],
             ['华东', '重庆珂琳雅', 6328834459202328430, 61215, 61216, ''],
             ['华东', '重庆美仑美奂整形医院', 6328834448334827353, 60535, 60536, '亲亲 您好，领取的红包可以直接抵扣项目费用哦，有什么能帮您？'],
             ['华东', '重庆光博士医疗美容门诊部', 6329571006646101183, 59467, 59466,
              '你好，亲亲，七夕快乐，最近在了解什么项目呢？现在院内活动期间，优惠力度很大，领红包还可以抵扣项目费用哦~'],
             ['华东', '重庆军美医疗美容医院', 6329153871742549047, 60779, 61073,
              '亲亲，看您有浏览我们医院的项目，亲亲是想了解哪方面呢？有什么疑虑也可以提出来，平台用户还有专享红包，请记得领取哦'],
             ['华东', '重庆军科', 6373818469783304494, 60679, 61228,
              '您好，你最近有关注咱们医院的项目，目前我院正在进行周年庆大促活动，全项目年度超低价秒杀，更有58元享7大特权，有关活动和项目可以随时联系我，我是你的美丽顾问小美~'],
             ['华东', '重庆军美医疗美容医院', 6329153871742549047, 60779, 60780,
              '亲亲，看您有浏览我们医院的项目，亲亲是想了解哪方面呢？有什么疑虑也可以提出来，平台用户还有专享红包，请记得领取哦'],
             ['华东', '重庆华美整形美容医院', 6329571006155434729, 60781, 59714,
              '重庆华美整形感谢您关注本院变美项目，送您一个红包可叠加平台红包共享优惠，有想咨询了解的请随时联系！'],
             ['华东', '重庆军科医院医疗有限公司', 6373818469783304494, 60784, 60783,
              '看到亲有关注我们医院的变美项目，是想了解哪方面呢？送您一个手术无门槛直减红包，有管项目的问题可以随时联系我们哦！'],
             ['华东', '成都鹏爱悦己医疗美容', 6509689155646209261, 44903, 0, '爱美不止夏季，我要您每天都美美哒'],
             ['华东', '成都圣丹福整形美容医院', 6419732530840338339, 56770, 56769,
              '成都圣丹福感谢您关注本院变美项目，送您一个红包可叠加平台红包共享优惠，有想咨询了解的请随时联系！'],
             ['华东', '成都赵善军博士整形美容', 6329570492894207924, 59666, 59665,
              '成都赵博士整形美容感谢您关注本院变美项目，送您一个红包可叠加平台红包共享优惠，有想咨询了解的请随时联系！'],
             ['华东', '成都恒博医疗美容', 6328834489678147584, 56896, 56895,
              '成都恒博医疗美容感谢您关注本院变美项目，送您一个红包可叠加平台红包共享优惠，有想咨询了解的请随时联系！'],
             ['华东', '重庆当代整形美容医院', 6328834449660238121, 54358, 54356, '与时尚接轨、与潮流同步，变美从重庆当代开始，期待您的咨询！'],
             ['华东', '重庆联合丽格美容医院', 6328834429254943862, 58448, 58449, '号外号外号外，重庆联合丽格8月优惠，红包多多，活动多多，快行动起来吧'],
             ['华东', '昆明艺星医疗美容医院', 6328834427111659985, 54300, 52967, '订制属于您专属的美丽，昆明艺星随时欢迎您的来访'],
             ['华东', '成都军大整形美容医院', 6328834424494408282, 60818, 0,
              '您好，“成都军大医院”欢迎您！ 我院为中国人民解放军、西部战区空军医院技术指导医院 • 中整协分会指定西南教学示范基地 • 并由中国平安;中国人保承保西南地区医美保险官方指定首家合作机构，我是专属您的客服，接下来我将为您一对一提供咨询服务！'],
             ['华东', '成都中医大银海医美', 6456090571630042287, 53015, 52170,
              '您有一份红包还没领取哟！这里是国家级三级医院，成都中医药大学附属医院银海眼科医院。关于变美方面小姐姐有任何问题都可以私信我哟！期待您的回复！'],
             ['华东', '四川省人民医院医疗集团友谊医院', 6382427920501107262, 60330, 60340, '专享轮廓红包1千到院抵2000，手术项目满一万立减1000千'],
             ['华东', '西安画美医疗美容医院', 6436029762384264000, 41214, 60641, '帮助每一位爱美的朋友安全无忧的达成美丽心愿，西安画美为您的人生画出更美的色彩！'],
             ['华东', '西安美莱医学美容医院', 6419735170521299284, 60617, 0, '亚洲实力雄厚的专业医疗美容连锁品牌，来美莱更放心，更有优惠活动哦'],
             ['华东', '四川娇点医学美容医院', 6328834489829137966, 60808, 0, '您的美丽，我来守候，期待您的咨询与到访。'],
             ['华东', '成都棕南医院', 6400946260789962410, 60683, 60682, '丰富的技术沉淀、专业的医美服务团队与雄厚的硬件基础，为每一位用户提供贴心、私密、个性化的美丽蜕变方案'],
             ['华东', '成都武侯格莱丽医疗美容门诊部', 6418755372676294081, 60706, 60705,
              '亲亲，您终于来啦，我们有国际化的医美技术、追求卓越的艺术审美、无微不至的人文服务，有什么问题随时来咨询我们哦。'],
             ['华东', '重庆美仑美奂医疗整形美容医院', 6328834448334827353, 60535, 0,
              '终于等到你，欢迎关注西安壹加壹。您的专享抵用券已到账，现我院暑期招募，活动非常优惠，红包直接抵现，有问题随时联系我们哦'],
             ['华东', '西安壹加壹医疗美容医院（西北旗舰店）', 6497039796123247593, 60822, 49072,
              '以”健康塑美，德行天下”为企业使命。医院集多种科室于一体，西安壹加壹是西北地区专业医美领航机构，欢迎您咨询，我们24小时在线为您服务。本月我们医院暑期特惠活动很优惠的'],
             ['华东', '西安米兰柏羽医疗美容医院（原西安美立方）', 6328834443008031526, 60825, 0,
              '嗨~小主~咱们家是重庆首批中国5A整形美容医院之一，更美明星人气机构，服务和品质受到权威官方和众多小姐姐的一致认可~小姐姐是想了解什么项目呢'],
             ['华东', '西安现代医美中心', 6328834635807602879, 60802, 31205, '亲亲，看您有浏览我们项目，我们已经恭候多时了，领取的红包可以直接抵扣项目费用哦，有什么能帮您？'],
             ['华东', '重庆新瑞俪医疗美容', 6328834488243667252, 60029, 56648,
              '亲爱的小主，欢迎来到重庆新瑞俪，看您有关注我院的项目，小主是想了解哪方面呢？有什么疑虑也可以提出来，平台用户还有专享红包，请记得领取哦"'],
             ['华东', '重庆珂琳雅医疗美容门诊部', 6328834459202328430, 60803, 60729, '抓住暑期的尾巴，活动不停，抓紧变美吧'],
             ['华东', '昆明美伊莱医疗美容', 6328834616790646964, 43295, 0,
              '终于等到你，还好我没放弃，小仙女，有什么需要了解的项目吗，昆明美伊莱暑期活动，项目大放价，随时欢迎咨询哦！'],
             ['华东', '深圳弘美医疗美容门诊部', 6328834436033005523, 60848, 60849,
              '宝宝您好，感谢关注弘美整形，送您两个无门槛优惠券，有任何不明白的可以随时联系我们哦！现在咨询即可领取礼物哟：1、水氧活肤；2、脱毛 1次；3、小气泡深度清洁补水；4、光子嫩肤 5、专家面诊整形形象设计一次；以上项目5选一，直接回复礼物数字登记预约即可，豪礼多多，无套路完全免费。\t'],
             ['华东', '广州可玫尔医疗美容', 6547065780838599993, 60850, 60853,
              '宝宝您好，感谢关注广州可玫尔，送您两个无门槛优惠券，有任何不明白的可以随时联系我们哦！现在咨询即可领取礼物哟：【1.水氧活肤 2.脱腋毛/唇毛 1次；3.小气泡深度清洁补水 4.专家面诊整形形象设计一次 】以上项目四选一，直接回复礼物数字登记预约即可，豪礼多多，等你来撩 ~'],
             ['华东', '广州曙光医疗美容医院', 6328834426579004502, 60595, 60597,
              '亲，您好，小曙为您送福利红包啦！咨询来院送水光针1次，充值10000抵20000，师生证还可以享受8.8折的优惠哦，有任何问题都可以联系我们哦~'],
             ['华东', '广州艺美医疗美容门诊部', 6329138755466811434, 60860, 60861, '艺美整形基金拍了拍你，变美无需走弯路。赶紧来咨询预约吧，更多优惠尽在小艺咨询间'],
             ['华东', '深圳阳光整形美容医院', 6352044935939662564, 54528, 60867,
              '亲亲您好呀，感谢关注阳光整形，送您两个8月福利无门槛红包，有任何不明白的联系我们哟！眼鼻整形学生双人行 7折/人，有任何问题可以联系我们呢，我们期待你的联系呢~'],
             ['华东', '厦门欧菲医疗美容', 6329571756508027738, 60868, 60869, '宝宝您好，感谢关注厦门欧菲整形，送您两个8月福利红包，有任何不明白的可以随时联系我们~'],
             ['华东', '深圳美莱医疗美容医院', 6328834599929570773, 60778, 60777, '你好，亲，看亲在浏览咱们家的项目，有什么想要了解的呢？给亲一个变美红包，记得领取哦！'],
             ['华东', '刘晓伟', 6328834883443497610, 60870, 60872,
              '您好，看亲有关注我们刘晓伟院长，刘院是第三军医大新桥医院整形外科博士，毕业于北京协和医学院暨清华大学医学院，在国内外的整形外科杂志上都发表过学术文章近十篇，拥有四项整形方面专利，做的是医生的口碑技术，很多做过手术的亲带自己的亲戚朋友过来做，效果都非常满意。 你是想改善哪个部位呢？'],
             ['华东', '四川美莱医学美容医院', 6328834446543884174, 60325, 60326,
              '你的暑假余额已充足! 请及时享乐！ 美莱暑期特惠大放送 ！下单即享无门槛20元代金券 ，满3000减300元超值优惠 来美莱，开启你的高光时刻'],
             ['华东', '四川省人民医院医疗集团友谊医院', 6382427920501107262, 60330, 60341, '国内轮廓品牌机构平台独家补贴， 领取红包为美减压吧！'],
             ['华东', '西安画美医疗美容医院', 6436029762384264000, 39243, 41214, '西北最大整形医院在线独家补贴，即刻领取20红包，更有多重折扣，欢迎在线询问！'],
             ['华东', '重庆华美整形美容医院', 6329571006155434729, 59714, 60322,
              '西南地区唯一一家五大轮廓名院送您20专属积分，积分膨胀10倍，最高抵扣2000元！ 更有全项目3期免息，12期贴息政策让您轻松变美！'],
             ['华东', '重庆万家燕整形外科医院', 6329570617217561931, 60293, 60323,
              '感谢关注重庆万家燕整形外科医院，本院是西南地区首家三级整形外科医院，四级高难度手术合法资质，硕导专家团队&金燕体系&路费报销，最高抵扣2500，期待您的来访！'],
             ['华东', '广州紫馨', 6328834427262649614, 60315, 60309, '感谢亲亲关注紫馨医院，送您一个变美红包。如对项目有任何问题，欢迎随时沟通！'],
             ['华东', '深圳广和门诊部', 6328834422183305150, 60311, 60314, '感谢亲亲关注广和医疗美容，送您一个变美红包。如对项目有任何问题，欢迎随时沟通！'],
             ['华东', '广东省韩妃整形医院（华南旗舰院）', 6496297469436051261, 60254, 60307, '感谢亲亲关注韩妃医疗美容，送您一个变美红包。如对项目有任何问题，欢迎随时沟通！'],
             ['华东', '福州爱美尔医疗美容', 6328834445709231124, 60308, 60306, '感谢亲亲关注福州爱美尔医院，送您一个变美红包。如对项目有任何问题，欢迎随时沟通！'],
             ['华东', '深圳谦达医疗美容', 6362216957520990826, 60413, 60412,
              '小仙女，感谢您一直关注我们，本月到院的新客人，有夏日护肤、除皱针等免费体验，还有红包直接抵现哦，别犹豫啦，即刻私信我吧！'],
             ['华东', '深圳艺星医疗美容医院', 6456040881702122545, 58172, 58171,
              '暑期上门礼：来院送医用面膜1片/玻尿酸次抛原液1支交通补贴: 全城滴滴免费接送（整形项目消费满5000元）'],
             ['华东', '深圳鹏爱秀琪医疗美容医院', 6358965722089031521, 60075, 0,
              '衡力瘦脸针仅需560元100单位，瘦腿瘦肩针200单位仅需1580，除皱单部位低至99元起，玻尿酸更有特惠，中分子399起~ 进口伊婉599起，更有保妥适除皱680元单部位，保妥适瘦脸针1980元~师生季，凭证明，来院最高可以直接抵扣最高1000元！！ 与客服小姐姐沟通了解更多吧~'],
             ['华东', '广州曙光医学美容医院', 6366207821284979921, 61105, 0,
              '【曙光23周年庆超级生日惠，一年仅此一次】 人气爆款【230元】任选GO，享专属福利特权 福利一：送230元=2300元 福利二：46项热销爆款，每天限购1项。 福利三：来院领取驼绒被一套或者4盒医用械字号面膜 具体详情请咨询我们的客服小姐姐哟~'],
             ['华东', '广州华美医疗美容医院', 6328834434288073264, 60855, 0, '广州华美4D模拟设计，先模拟术后效果再决定变美。赶紧来咨询预约吧，更多优惠尽在广州华美。'],
             ['华东', '广州美莱医疗美容', 6328834463652437620, 56730, 0,
              '感谢关注广州美莱华南旗舰院！亚洲22年36院连锁品牌，196名国际名医，19层独栋五星环境，百级层流手术室，解决您所有美丽烦恼。'],
             ['华东', '广州爱丽雅思医疗美容门诊有限公司', 6527467879267344889, 61132, 0,
              '感谢关注广州爱丽雅思医疗美容，为您提供专业的医学美容治疗和满意的服务一直是我们不断改进和奋斗的目标！国庆中秋不打烊，直减红包、到店即赠，活动多多，等您来哦'],
             ['华东', '广州市康华清医学美容医院', 6440131960944793740, 58731, 0, '感谢关注广州市康华清医学美容医院，欢迎咨询预约，领取红包获取更多优惠哟！'],
             ['华东', '广州健丽', 6328834433763774687, 61140, 0,
              '您好呀，这里是专注眼部25年广州健丽医疗美容医院，8月手术整形硬核特惠，欢迎来撩~ 不开刀去眼袋：5440元 不开刀双眼皮：5440元，下单预约送一次全面皮秒/水光针1支！全脸热玛吉900发仅需8800元，眼周热玛吉6800元，各赠送射频紧肤一次！ 另有199元六选三个项目：水氧活肤/激光点痣/脱毛1次/去黑头/小气泡/眼部BIO紧致'],
             ['华东', '厦门欧菲医疗美容医院', 6329571756508027738, 60868, 0, '变美无需多走弯路，有任何疑问或想了解优惠活动可以私信客服了解~'],
             ['华东', '福州市鼓楼区美贝尔医疗美容门诊部有限公司', 6328834426813895975, 60655, 0,
              '美贝尔是全国连锁机构，福州美贝尔全新升级是中国区旗舰医院，拥有6大整形专利技术，目前在整形行业比较领先的，您有任何的变美诉求都可在线问诊我们，我们将会给您带来最好的变美方案！'],
             ['华东', '福州爱美尔医疗美容', 6328834445709231124, 60308, 0, '福建最大的整形机构在线独家补贴，领取红包为美减压，爱美就到爱美尔'],
             ['华东', '广州可玫尔医疗美容', 6547065780838599993, 60852, 0,
              '姐妹们！年度白菜娇宠卡福利来啦~258元即可选玻尿酸/水光针/光一次，再送冰点脱毛年卡、水氧嫩肤2次、手术基金1000元代金券~~仅需258元4项好礼抱回家'],
             ['华东', '广东韩妃', 6496297469436051261, 60307, 0, '下单即享无门槛50元代金券', '2020-7-27 至 2020-10-29']]

    def handle(self, *args, **options):
        self.checkMerchant()
        self.checkCoupon()

        self.sendCoupon()

    def sendCoupon(self):

        resMerchantFailedSend = {}
        resMerchantSucceedSend = {}
        resConversationMap = {}

        sendReqNum = 0
        timeStart = time.time()

        for line in self.availableMerchantIDListWithCoupon:

            failedSendUserList = []
            succeedSendUserList = []
            allConversationID = []

            doctorIDList = []
            doctorUIDList = []
            sendCouponAdminUID = 0
            targetUserSet = set()
            sendCouponList = []
            conversationList = []

            merchantID = line[2]

            print()
            print()
            print("start send coupon of merchant id: {merchantID}".format(merchantID=merchantID))
            print("merchant id: ", merchantID)
            couponID1 = line[3]
            print("coupon id 1: ", couponID1)
            couponID2 = line[4]
            print("coupon id 2: ", couponID2)
            sendCouponList.append(couponID1)
            sendCouponList.append(couponID2)

            sendGiftList = self.getGiftID(sendCouponList)
            print "gift list: "
            print sendGiftList
            msgContent = line[5]
            print("msg content: ", msgContent)

            # get merchant info
            merchant = Merchant.objects.filter(id=merchantID).first()
            if not merchant:
                continue

            # get admin doctor id
            adminDoctorID = merchant.doctor.id

            # add admin doctor id to doctorIDList
            doctorIDList.append(adminDoctorID)

            # get other doctors
            doctors = MerchantRelevance.objects.filter(merchant_id=merchantID, doctor__doctor_type=DOCTOR_TYPE.DOCTOR)
            doctorCount = doctors.count()
            ds = doctors[0: doctorCount]
            otherDoctorList = [d.doctor.id for d in ds]

            # add other doctor id to doctorIDList
            doctorIDList.extend(otherDoctorList)

            # get all doctor info
            doctors = Doctor.objects.filter(pk__in=doctorIDList)

            for doctor in doctors:

                ddid = doctor.id
                duid = doctor.user_id

                # add every doctor uid to doctorUIDList
                doctorUIDList.append(duid)
                if ddid == adminDoctorID:
                    sendCouponAdminUID = duid

            # get message target user' user_id list at 2020/4/1 - 2020/7/31
            offsetNumber = 0
            while True:

                # get conversations between 2020/4/1 -- 2020/6/30 of merchant
                conversationResList = get_conversations_v3(
                    user_ids=doctorUIDList,
                    offset=offsetNumber,
                    size=50,
                    is_star=0,
                    last_reply_time_start="2020-01-01T00:00:00",
                    last_reply_time_end="2020-08-23T00:00:00",
                )

                # print("msg res: ")
                # pprint.pprint(conversationResList)
                resCount = conversationResList["total_count"]

                # no conversation
                if resCount <= 0:
                    break

                conversationListOne = conversationResList["conversation_list"]

                conversationList.extend(conversationListOne)

                if len(conversationListOne) < 50:
                    break

                offsetNumber += 50

            conversationNum = len(conversationList)
            print("conversation number of merchant: {merchantID} in 2020-1-1 and 2020-8-23 is ".format(
                merchantID=merchantID), conversationNum)

            # no conversation, handle next merchant
            if conversationNum == 0:
                continue

            # get target user id set
            # 因为发送者和接收者，顺序不定，所以都取出来去判断
            for conversationRow in conversationList:

                targetUserID0 = conversationRow["user_ids"][0]
                targetUserID1 = conversationRow["user_ids"][1]
                conversationID = conversationRow["id"]

                allConversationID.append(conversationID)

                if targetUserID0 not in doctorUIDList:
                    targetUserSet.add(targetUserID0)

                if targetUserID1 not in doctorUIDList:
                    targetUserSet.add(targetUserID1)

            # print("target user id set: ", targetUserSet)
            targetUserNum = len(targetUserSet)
            print("target user number of merchant id:{merchantID} is ".format(merchantID=merchantID), targetUserNum)

            targetUserSet.add(self.onlineTestuid)

            for targetUserOne in targetUserSet:

                for giftID in sendGiftList:

                    resSendCoupon = internal_message_send(
                        sender_user_id=sendCouponAdminUID,
                        target_user_id=targetUserOne,
                        msg_type=MESSAGE_TYPE.GIFT,
                        conversation_type=CONVERSATION_TYPE.MESSAGE,
                        content={"gift_id": giftID, "channel_id": self.DOCTOR_BUSINESS_CHANNAL_ID_ONLINE}
                    )

                    sendReqNum += 1

                    errSendGift = resSendCoupon["error"]
                    # send failed
                    if errSendGift == CODES.MESSAGE_CONTENT_FILTERED_OUT:
                        # print(
                        #     "Failed to send merchant:{merchantID}'s coupon: {couponGiftID} to user: {targetIDOne}".format(
                        #         merchantID=merchantID, couponGiftID=giftID, targetIDOne=targetUserOne))

                        failedGiftInfo = str(giftID) + "_" + str(targetUserOne)
                        failedSendUserList.append(failedGiftInfo)

                    # send success
                    # if errSendGift == CODES.SUCCESS:
                    #     print(
                    #         "Succeed to send merchant:{merchantID}'s coupon: {couponGiftID} to user: {targetIDOne}".format(
                    #             merchantID=merchantID, couponGiftID=giftID, targetIDOne=targetUserOne))

                # send msg
                resSendMsg = internal_message_send(
                    sender_user_id=sendCouponAdminUID,
                    target_user_id=targetUserOne,
                    msg_type=MESSAGE_TYPE.TEXT,
                    conversation_type=CONVERSATION_TYPE.MESSAGE,
                    content={"text": msgContent}
                )

                sendReqNum += 1

                errSendMsg = resSendMsg["error"]
                # if errSendMsg == CODES.MESSAGE_CONTENT_FILTERED_OUT:
                #     print(
                #         "Failed to send merchant:{merchantID}'s msg to user: {targetIDOne}".format(
                #             merchantID=merchantID, targetIDOne=targetUserOne))

                if errSendMsg == CODES.SUCCESS:
                    # print(
                    #     "Succeed to send merchant:{merchantID}'s msg to user: {targetIDOne}".format(
                    #         merchantID=merchantID, targetIDOne=targetUserOne))
                    succeedSendUserList.append(targetUserOne)

            resMerchantFailedSend[merchantID] = failedSendUserList
            resConversationMap[merchantID] = allConversationID
            resMerchantSucceedSend[merchantID] = succeedSendUserList

            print("send coupon complete of merchant: {merchantID}".format(merchantID=merchantID))

        print("res failed to send: ")
        print(resMerchantFailedSend)

        print("res succeed to send: ")
        print(resMerchantSucceedSend)

        print("conversation id list: ")
        print(resConversationMap)

        print("total send request number: ", sendReqNum)

        timeEnd = time.time()
        timeUse = timeEnd - timeStart
        print("time use: ", timeUse)

    def checkMerchant(self):

        unavailableMerchantList = []

        print "===== start to check merchant id ====="

        for line in self.c:
            try:

                merchantID = line[2]

                m = Merchant.objects.filter(id=merchantID).first()
                if not m:
                    unavailableMerchantList.append(line)
                else:
                    self.availableMerchantIDList.append(line)
            except Exception as e:
                print e
                continue


        print "=== result of check merchant id ==="
        print "num of unavailable line with merchant:"
        print len(unavailableMerchantList)
        print "content of unavailable line with merchant:"
        print unavailableMerchantList
        print "===== end to check merchant id ====="

    def checkCoupon(self):

        unavailableMerchantInfoSet = set()

        print "===== start to check coupon id ====="

        for line in self.availableMerchantIDList:

            doctorIDList = []
            couponList = []

            merchantID = line[2]
            couponID1 = line[3]
            couponID2 = line[4]
            couponList.append(couponID1)
            couponList.append(couponID2)

            # get merchant obj
            m = Merchant.objects.filter(id=merchantID).first()

            # get admin doctor
            adminDoctorID = m.doctor.id
            print "admin doctor", adminDoctorID

            # add admin doctor id to doctorIDList
            doctorIDList.append(adminDoctorID)

            # get other doctors
            doctors = MerchantRelevance.objects.filter(merchant_id=merchantID, doctor__doctor_type=DOCTOR_TYPE.DOCTOR)
            doctorCount = doctors.count()
            ds = doctors[0: doctorCount]
            otherDoctorList = [d.doctor.id for d in ds]

            doctorIDList.extend(otherDoctorList)
            print "doctor list: ", doctorIDList

            # get doctor of every coupon, find it is exist in doctor list
            for couponID in couponList:
                if couponID == 0:
                    continue

                c = Coupon.objects.filter(id=couponID).first()
                if not c:
                    unavailableMerchantInfoSet.add(merchantID)
                    continue

                createDoctor = c.doctor
                if not createDoctor:
                    unavailableMerchantInfoSet.add(merchantID)
                    continue

                createDoctorID = createDoctor.id

                # coupon's create doctor is not belong to merchant,
                # add to unavailableMerchantInfoSet
                if createDoctorID not in doctorIDList:
                    unavailableMerchantInfoSet.add(merchantID)
                    continue

                couponType = c.coupon_type

                # coupon's type is not COUPON_TYPES.DOCTOR,
                # add to unavailableMerchantInfoSet
                if couponType != COUPON_TYPES.DOCTOR:
                    unavailableMerchantInfoSet.add(merchantID)
                    continue

                distributionStatus = c.distribution_status

                # coupon's distribution status is not COUPON_DISTRIBUTION_STATUS.OPEN,
                # add to unavailableMerchantInfoSet
                if distributionStatus != COUPON_DISTRIBUTION_STATUS.OPEN:
                    unavailableMerchantInfoSet.add(merchantID)
                    continue

                timeType = c.time_type

                # if time type is COUPON_TIME_TYPE.START_END,
                # check start_time <= now <= end_time,
                # if incompatible, add to unavailableMerchantInfoSet
                if timeType == COUPON_TIME_TYPE.START_END:
                    startDataTime = c.start_time
                    endDataTime = c.end_time

                    nowTime = datetime.datetime.now()

                    if nowTime < startDataTime or nowTime > endDataTime:
                        unavailableMerchantInfoSet.add(merchantID)
                        continue

                # if can't find GiftToCoupon obj where coupon.id=couponID,
                # add to unavailableMerchantInfoSet
                couponGift = GiftToCoupon.objects.filter(coupon_id=couponID).first()
                if not couponGift:
                    unavailableMerchantInfoSet.add(merchantID)
                    continue

                # if coupon is not enable, add to unavailableMerchantInfoSet
                isEnable = couponGift.coupon_enable
                if not isEnable:
                    unavailableMerchantInfoSet.add(merchantID)
                    continue

        print "===== result of check coupon id ====="
        print "num of unavailable line with merchant and coupon:"
        print len(unavailableMerchantInfoSet)
        print "content of unavailable line with merchant and coupon:"
        print unavailableMerchantInfoSet
        print "===== end to check coupon id ====="

        for line in self.c:

            mid = line[2]

            if mid in unavailableMerchantInfoSet:
                continue

            self.availableMerchantIDListWithCoupon.append(line)

        print "===== available line with merchant and coupon ====="
        print self.availableMerchantIDListWithCoupon
        print "================ END ======================"

    @staticmethod
    def getGiftID(couponList):

        giftList = []

        for couponID in couponList:
            if couponID == 0:
                continue

            couponGift = GiftToCoupon.objects.filter(coupon_id=couponID).first()
            giftID = couponGift.coupon_gift.id
            giftList.append(giftID)

        return giftList
