# coding=utf8

from __future__ import unicode_literals, absolute_import, print_function

from django.db import models
from django.db.models import ForeignKey, IntegerField, CharField, TextField
from django.db.models import ManyToManyField, BooleanField, DateTimeField

from gm_upload import ImgUrlField, IMG_TYPE
from gm_types.wiki import PRODUCT_TYPE, PRODUCT_IMAGE_TYPE, ATTESTATION_TYPE
from gm_types.wiki import PRODUCT_PARAMS, ANESTHESIA_TYPE, TREATMENT_TYPE
from gm_types.wiki import ACHE_LEVEL_NEW, NEW_WIKI_WORDREL_CATEGORY

from api.models.wordrel import WordRel
from wiki.models.brand import Brand


class Product(models.Model):
    class Meta:
        verbose_name = u'物品百科'
        db_table = 'wiki_product'

    soyoung_id = IntegerField(u'新氧id', null=True)
    name = CharField(u'名字', max_length=50)
    icon = ImgUrlField('图标', img_type=IMG_TYPE.ITEMWIKI, max_length=128)
    other_name = CharField(u'别名', max_length=100)
    product_type = CharField(u'类型', choices=PRODUCT_TYPE, max_length=4)
    brand = ForeignKey(Brand, null=True, related_name='products')
    origin = CharField('产地', max_length=48)
    description = TextField(u'介绍')
    special = TextField(u'特色')
    theory = TextField(u'原理')
    effect = CharField(u'功效', max_length=128)
    application_range = CharField(u'适用范围', max_length=128)
    advantage = TextField(u'优点')
    disadvantage = TextField(u'缺点')
    after_know = TextField(u'术后提示')
    safe_tips = TextField(u'安全提示')
    verify_url = CharField(u'防伪链接', max_length=256, default='')
    clinical = CharField(u'临床时间', max_length=50, default='')
    ingredient = TextField(u'原料原理', default='')
    treatment_method = CharField(u'治疗方式', max_length=8, choices=TREATMENT_TYPE)
    anesthesia_method = CharField(u'麻醉方法', max_length=8, choices=ANESTHESIA_TYPE)
    ache_level = CharField(u'疼痛感', max_length=8, choices=ACHE_LEVEL_NEW, default=ACHE_LEVEL_NEW.ZERO)
    price = CharField(u'参考价格', max_length=50)
    hospitalized = BooleanField(u'是否住院', default=False)
    treatment_time = CharField(u'操作时长', max_length=50)
    maintain_time = CharField(u'维持时长', max_length=50)
    recover_time = CharField(u'恢复周期', max_length=50)
    treatment_num = CharField(u'治疗周期', max_length=50)
    treatment_doctor = CharField(u'治疗人员要求资质', max_length=50)
    similar_product = ManyToManyField('self')
    is_hot = BooleanField(u'是否热门', default=False)
    is_online = BooleanField(u'是否上线', default=True)
    create_time = DateTimeField(u'创建时间', auto_now_add=True)
    similar_sy_product = CharField(u'新氧类似项目', max_length=500, default='[]')

    def detail_info(self):
        fields = [
            'id', 'name', 'other_name', 'product_type', 'origin', 'description',
            'special', 'theory', 'effect', 'application_range', 'advantage',
            'disadvantage', 'after_know', 'safe_tips', 'verify_url', 'is_hot',
            'ache_level', 'ingredient', 'price'
        ]
        detail_info = {}
        for _f in fields:
            detail_info[_f] = getattr(self, _f)
        return detail_info

    def get_similar_products(self):
        similar_products = self.similar_product.filter(is_online=True).exclude(pk=self.id)
        return [
            {
                "id": x.id,
                "name": x.name,
            }for x in similar_products
        ]


class ProductParams(models.Model):
    class Meta:
        verbose_name = '物品参数'
        db_table = 'wiki_productpararms'

    product = ForeignKey(Product, related_name='params')
    params_type = CharField(u'参数名称', max_length=8, choices=PRODUCT_PARAMS)
    params_message = CharField(u'参数内容', max_length=256, default='')


class ProductImage(models.Model):
    class Meta:
        verbose_name = u'物品图片'
        db_table = u'wiki_productimage'

    product = ForeignKey(Product, related_name='images')
    info = CharField(u'图片介绍', max_length=256, default='')
    image_type = CharField(u'图片类型', max_length=8, choices=PRODUCT_IMAGE_TYPE)
    image_url = ImgUrlField(u'图片url', max_length=128, null=True, img_type=IMG_TYPE.ITEMWIKI)
    is_online = BooleanField('是否上线', default=True)


class ProductAttestation(models.Model):
    class Meta:
        verbose_name = '物品认证'
        db_table = 'wiki_productattestation'

    product = ForeignKey(Product, related_name='attestations')
    attestation_type = CharField(u'认证类型', max_length=8, choices=ATTESTATION_TYPE)
    attestation_image = ImgUrlField('认证图片', img_type=IMG_TYPE.ITEMWIKI, null=True, max_length=128)
    is_online = BooleanField('是否上线', default=True)


class ProductRelationKeyWords(models.Model):
    class Meta:
        verbose_name = '药品、材料、仪器百科 关联的关键词'
        db_table = 'wiki_product_keywords'

    product = ForeignKey(Product, related_name='words')
    wordrel = ForeignKey(WordRel, verbose_name=u'关键字')
    category = CharField(verbose_name=u'分类', max_length=5, choices=NEW_WIKI_WORDREL_CATEGORY)
    is_effect = BooleanField(verbose_name=u'是否有效', default=True)
    created_time = DateTimeField(verbose_name=u'创建时间', auto_now_add=True)
    updated_time = DateTimeField(verbose_name=u'最后更新时间', auto_now=True)


class ProductTagV3(models.Model):
    class Meta:
        verbose_name = u'物品(仪器、材料、药品)百科关联新标签'
        db_table = 'wiki_product_tag_v3'

    product_id = IntegerField(verbose_name=u'物品(仪器、材料、药品)百科关联新标签', db_index=True)
    tag_v3_id = IntegerField(verbose_name=u'标签V3', db_index=True)
    create_time = DateTimeField(verbose_name=u'创建时间', auto_now_add=True)
    update_time = DateTimeField(verbose_name=u'更新时间', auto_now=True)
