#! /usr/bin/env python
# -*- coding: utf-8 -*-
# __author__ = "chenwei"
# Date: 2018/5/16

import json

from gm_dataquery.db import DB
from django.db.models import Q
from qa.models import TouTiaoTag, ToutiaoRelation
from gm_dataquery.dataquery import DataBuilder, DataSQLQuery
from django.core.cache import cache

from ..views.common import get_tag_info_by_id, get_tag_id_by_name

def cache_tag_info(id, tag_id_list):
    key = 'tag:{}:fronted_related_tag'.format(id)
    tags_info = cache.get(key)
    if not tags_info:
        tags_info = get_tag_info_by_id(list(tag_id_list))
        cache.set(key, tags_info, 1 * 60 * 60)
    return tags_info


class TouTiaoTagDB(DataBuilder):

    def getval_related_tag(self, obj):
        tag_id_list = list(obj.relationtags.all().values_list('tag', flat=True))
        return tag_id_list if tag_id_list else ''

    def getval_fronted_related_tag(self, obj):
        tag_name_list = []
        tag_id_list = list(obj.relationtags.all().values_list('tag', flat=True))
        result = cache_tag_info(obj.id, list(tag_id_list))
        for _data in result:
            tag_name_list.append("{}:{}".format(_data['id'],_data['name']))
        return ','.join(tag_name_list) if tag_name_list else ''

@DB
class TouTiaoTagDQ(DataSQLQuery):
    model = TouTiaoTag
    data_model = TouTiaoTagDB

    def filter_fronted_related_tag(self,srch_key, srch_val, regex=False):
        tag_ids = get_tag_id_by_name(srch_val)
        return Q(id__in=ToutiaoRelation.objects.filter(tag__in=tag_ids).values_list('toutiaotag_id', flat=True))

    def filter_related_tag(self,srch_key, srch_val, regex=False):
        return Q(id__in=ToutiaoRelation.objects.filter(tag=srch_val).values_list('toutiaotag_id', flat=True))


    @classmethod
    def create(cls, **kwargs):
        tags = kwargs.pop('related_tag', None)
        name = kwargs.pop('toutiao', None)
        obj,created = TouTiaoTag.objects.get_or_create(toutiao=name, defaults=kwargs)
        if created and tags:
            for tag_id in tags:
                ToutiaoRelation.objects.get_or_create(toutiaotag=obj, tag=tag_id)
            obj.is_related = True
            obj.save()
        return {'id':obj.id}

    def update(self, updates, **kwargs):
        if 'related_tag' in updates:
            is_related = True if updates.get('related_tag') else False
            tags = updates.pop('related_tag')
            obj = self.model.objects.get(**kwargs)
            obj.is_related = is_related
            has_insert = obj.relationtags.values_list('tag', flat=True)
            wait_to_insert_tag_list = set(tags) - set(has_insert)
            need_to_delete_tag_list = set(has_insert) - set(tags)
            for tag in wait_to_insert_tag_list:
                ToutiaoRelation.objects.get_or_create(toutiaotag_id=kwargs.get('id'), tag=tag)

            for tag in need_to_delete_tag_list:
                ToutiaoRelation.objects.filter(toutiaotag_id=kwargs.get('id'), tag=tag).delete()
            obj.save()
        return  super().update(updates, **kwargs)