#!/usr/bin/env python
# -*- coding: utf-8 -*-

from gm_types.antispam import DocType
from gm_types.error import ERROR
from gm_types.gaia import (
    DOCTOR_PROCESS_STATUS,
)
from gm_types.mimas import (
    PUSH_ACTION_TYPE,
)
from gm_types.user_hierarchy import EventType

from communal.tools import PushService
from qa.models import (
    Answer,
    Question,
)
from qa.serializers import AnswerReplySerializer
from qa.tasks import check_spam
from talos.portal import (
    get_doctor_by_user_id,
)
from talos.models.doctor import DoctorMessageList
from user_hierarchy.portal import process_task
from utils.rpc import (
    gen,
    logging_exception,
)
from utils.stat_log import SocialStatLogForUserAction


class AnswerReplyCreateMethod(object):

    @staticmethod
    def create_answer_reply(create_info):
        """
        创建回答评论
        :param create_info:
        :return:
        """
        serializer = AnswerReplySerializer(data=create_info)
        if not serializer.is_valid(raise_exception=False):
            gen(ERROR.CONTENT_CREATE_ERROR)

        answer_reply = serializer.save()

        result = {
            "answer_reply": answer_reply,
            "reply_data": answer_reply.reply_data_after_create(),
            "comment_data": answer_reply.comment_data_after_create(),
        }

        return result

    @staticmethod
    def various_logic_after_answer_reply_create(answer_reply, user, question_id):
        """
        回答创建完毕，触发各个规则
        :param answer_reply:
        :param user:
        :return:
        """


        try:
            doctor = get_doctor_by_user_id(user.id)
        except IndexError:
            doctor = None
        if doctor:
            DoctorMessageList.objects.filter(
                question_id=question_id,
                doctor_id=doctor.id
            ).update(process_status=DOCTOR_PROCESS_STATUS.processed)

        # 用户行为埋点 评论相关
        try:
            SocialStatLogForUserAction.stat_log_for_reply(
                content_type=SocialStatLogForUserAction.CONTENT_TYPE.question,
                user_id=user.id,
                content_id=question_id
            )
        except:
            logging_exception()

        # 成长值相关
        event_data = process_task(
            user_id=user.id,
            event_type=EventType.COMMENT_OTHER,
            related_item=answer_reply.id
        )
        _data = {
            "growth_value": event_data['growth_value'],
            "point_value": event_data['point_value'],
            "submit_count": event_data['submit_count'],
        }

        return _data

    @staticmethod
    def trigger_tasks_after_answer_reply_create(answer_reply_id, author_user_id, push_user_id, **kwargs):
        """
        回答创建完毕，触发各个异步任务
        :param answer_reply_id:
        :param author_user_id:
        :param push_user_id:
        :return:
        """
        # 易盾反垃圾
        # check_spam.delay(answer_reply_id, ['content'], DocType.COMMENT)

        # push
        content = kwargs.get("content", "")
        valid_user_ids = PushService.filter_valid_user_ids([push_user_id], filter_user_id=author_user_id)
        PushService.push(
            user_ids=valid_user_ids,
            action_type=PUSH_ACTION_TYPE.ANSWER_RECV_REPLY,
            content=content,
            nick_name=kwargs.get("nick_name", ""),
            answer_id=kwargs.get("answer_id", 0),
            question_id=kwargs.get("question_id", 0),
            comment_id=answer_reply_id,
            is_commented_reply=kwargs.get("is_commented_reply", False),
            is_timely_push=kwargs.get("is_timely_push", True),
        )

    @classmethod
    def answer_reply_create(cls, user, content, answer_id, answer_reply_id=None, **kwargs):
        """
        回答评论创建
        :param user:
        :param content:
        :param answer_id:
        :param answer_reply_id:
        :param kwargs:
        :return:
        """
        create_data = {
            "user": user.id,
            "content": content,
            "answer": answer_id,
            "commented_reply": answer_reply_id,
            "is_fake": kwargs.pop("is_fake", False),
        }

        _answer = Answer.objects.filter(pk=answer_id).only("question_id").first()
        _question_id = _answer.question_id
        try:
            _data = cls.create_answer_reply(create_data)

            answer_reply = _data.pop("answer_reply", None)
            _other_data = cls.various_logic_after_answer_reply_create(answer_reply, user, _question_id)
            _data.update(_other_data)

            is_commented_reply = True if answer_reply_id else False
            if is_commented_reply:
                push_user_id = _data["comment_data"].get("at_user_id")
            else:
                push_user_id = _data["reply_data"].get("reply_user_id")

            cls.trigger_tasks_after_answer_reply_create(
                answer_reply_id=answer_reply.id,
                author_user_id=user.id,
                push_user_id=push_user_id,
                content=content,
                answer_id=answer_id,
                question_id=_question_id,
                nick_name=getattr(user, "nick_name", "") or getattr(user, "nickname", ""),
                is_commented_reply=is_commented_reply,
                is_timely_push=kwargs.get("is_timely_push", True)
            )
            _data.update({
                "answer_reply_id": answer_reply.id,
            })
            return _data

        except:
            logging_exception()
            return gen(ERROR.CONTENT_CREATE_ERROR)
