from django.conf import settings
from gm_rpcd.all import bind
from gm_types.doris import CARD_TYPE
from gm_types.mimas import GRABBING_PLATFORM

from qa.models import Answer, AnswerTag, AnswerTagV3
from talos.models.tractate import Tractate, TractateTag, TractateTagV3
from talos.services.tractate import TractateService


def get_author_id_by_business_id(content_id, content_type=CARD_TYPE.USERTOPIC):
    """
    根据内容，获取作者
    :param content_id:
    :param content_type:
    :return: user_id,  related_tag_ids
    """
    if not content_id:
        return None, [], []
    if content_type == CARD_TYPE.USERTOPIC:
        try:
            content = Tractate.objects.get(id=content_id)
        except Tractate.DoesNotExist:
            return None
    elif content_type == CARD_TYPE.ANSWER:
        try:
            content = Answer.objects.get(id=content_id)
        except Tractate.DoesNotExist:
            return None

    return content.user_id


def get_answers_by_author_id(author_id, start_num=0, count=10):
    """
    获取作者的回答
    :param author_id:
    :param start_num:
    :param count:
    :return:
    """
    kyc_answers = Answer.objects.using(settings.SLAVE_DB_NAME).filter(
        user=author_id, is_online=True, platform=GRABBING_PLATFORM.KYC, question__is_online=True,
    ).order_by('-create_time')
    if count:
        kyc_answers = kyc_answers[start_num: start_num + count]
    answer_ids = [answer.id for answer in kyc_answers]

    return {
        'ids': answer_ids,
    }


@bind('mimas/get_content_ids')
def get_content_ids(business_id, content_type, start_num=0, count=10, answer_start_num=0, only_kyc_answers=False):
    """
    获取内容id和类型
    :param business_id:
    :param content_type:
    :param start_num:
    :param answer_start_num:
    :param count:
    :param only_kyc_answers:
    :return:
    """
    result = {
        "tractate_ids": [],
        'answer_ids': [],
        'author_id': None,
    }
    if not all([business_id, content_type]):
        return result
    business_id = int(business_id)

    author_id = get_author_id_by_business_id(business_id, content_type)
    if not author_id:
        return result
    result['author_id'] = author_id
    if start_num == 0:
        if content_type == CARD_TYPE.USERTOPIC:
            result['tractate_ids'].insert(0, business_id)
        elif content_type == CARD_TYPE.ANSWER:
            result['answer_ids'].insert(0, business_id)

    if only_kyc_answers:
        kyc_answers = get_answers_by_author_id(
            author_id, count=0,
        )
        result['answer_ids'].extend(kyc_answers.get('ids', []))
        return result

    tractates = TractateService.get_mark_tractate_ids(
        author_id, business_id, start_num=start_num, count=count
    )
    tractate_ids = tractates.get('ids', [])
    result['tractate_ids'].extend(tractate_ids)
    if len(result['tractate_ids']) < count:
        kyc_answers = get_answers_by_author_id(
            author_id, count=count-len(result['tractate_ids']), start_num=answer_start_num
        )
        result['answer_ids'].extend(kyc_answers.get('ids', []))

    return result

