# coding=utf8

from __future__ import unicode_literals, absolute_import, print_function

from django.conf import settings
from django.db import models
from django.utils import timezone

from gm_types.gaia import VIDEO_CODE_STATUS
from gm_types.mimas.qa import VIDEO_SOURCE_TYPE

from talos.libs.image_utils import get_full_path

from .topic import Problem


class Video(models.Model):
    class Meta:
        db_table = 'api_topic_video'
        app_label = 'talos'

    topic = models.OneToOneField(Problem, default=None, null=True, related_name="video")
    video_url = models.CharField(max_length=128, verbose_name=u'视屏地址')
    video_pic = models.CharField(max_length=128, verbose_name=u'视屏封面')
    persistentId = models.CharField(max_length=128, verbose_name=u'七牛视频处理id')
    persistent_status = models.IntegerField(default=VIDEO_CODE_STATUS.NOSTART, verbose_name=u"七牛状态", db_index=True)
    water_url = models.CharField(max_length=128, verbose_name=u'水印视屏地址')
    persistent_clip_id = models.CharField(max_length=128, verbose_name=u'七牛视频截取处理id')
    persistent_clip_status = models.IntegerField(default=VIDEO_CODE_STATUS.NOSTART, verbose_name=u"七牛截取状态", db_index=True)
    intercept_video_url = models.CharField(max_length=128, verbose_name=u"截取的视频地址", default="")
    webp_url = models.CharField(max_length=128, verbose_name=u'webp图片地址')

    def get_video_info(self):
        if self.video_pic:
            video_pic = get_full_path(self.video_pic, '-w')
        else:
            video_pic = settings.VIDEO_HOST + self.video_url + settings.VIDEO_PIC_URL

        if self.water_url:
            video_url = self.water_url
        else:
            video_url = self.video_url

        if self.persistent_clip_status == VIDEO_CODE_STATUS.SUCCESS:
            video_webp = self.webp_url
        else:
            video_webp = ''

        return {
            'video_url': settings.VIDEO_HOST + video_url,
            'video_pic': video_pic,
            'short_video_url': video_webp
        }

    @classmethod
    def cleaned_video_url(cls, video_url):
        return video_url.replace(settings.VIDEO_HOST, '')


class VideoCover(models.Model):
    class Meta:
        db_table = 'api_video_cover'
        app_label = 'talos'
        index_together = [
            ['video_url'],
        ]

    source_id = models.IntegerField(verbose_name=u'视频来源ID', null=False)
    source_type = models.SmallIntegerField(verbose_name=u'视频来源类型', choices=VIDEO_SOURCE_TYPE)
    video_url = models.CharField(max_length=128, verbose_name=u'视屏地址')
    video_pic = models.CharField(max_length=128, verbose_name=u'视屏封面')
    width = models.IntegerField(verbose_name="图片宽度", default=0)
    height = models.IntegerField(verbose_name="图片高度", default=0)

    persistent_clip_id = models.CharField(max_length=128, verbose_name=u'七牛视频截取处理id', default="")
    persistent_clip_status = models.IntegerField(default=VIDEO_CODE_STATUS.NOSTART, verbose_name=u"七牛截取状态")
    intercept_video_url = models.CharField(max_length=128, verbose_name=u"截取的视频地址", default="")
    webp_url = models.CharField(max_length=128, verbose_name=u'webp图片地址', default="")


class VideoLibrary(models.Model):
    """用于处理问答、专栏的视频打上图标水印。记录原始视频地址及处理之后的数据"""
    class Meta:
        verbose_name = "视频库"
        db_table = 'api_community_video_library'
        app_label = 'talos'

    source_id = models.IntegerField(verbose_name=u"关联id", db_index=True)
    raw_video_url = models.CharField(max_length=128, verbose_name=u"视频地址")
    water_video_url = models.CharField(max_length=128, verbose_name=u"带图片水印的视频地址", default="")
    persistent_id = models.CharField(max_length=128, verbose_name=u'七牛视频处理id', default='')
    persistent_status = models.IntegerField(verbose_name=u"七牛状态", choices=VIDEO_CODE_STATUS, default=VIDEO_CODE_STATUS.NOSTART)
    video_type = models.IntegerField(verbose_name=u"视频类型", choices=VIDEO_SOURCE_TYPE)
    create_time = models.DateTimeField(verbose_name=u'创建时间', default=timezone.now)
    update_time = models.DateTimeField(verbose_name=u'更新时间', auto_now=True)
