#!/usr/bin/env python
# -*- coding: utf-8 -*-

from __future__ import unicode_literals, absolute_import, print_function

from gm_types.gaia import (
    DOCTOR_TYPE,
)

from talos.services import (
    UserService,
    DoctorService,
)

from utils.gevent_jobs import Tasks


def filter_ids(ids):
    return list(map(int, filter(None, ids)))


class UserConvertService(object):
    """
    用户信息转换，需要知道当前这个用户的医生信息
    """

    @classmethod
    def __get_doctor_info_by_user_ids(cls, user_ids):
        doctor_dic = {}

        if not user_ids:
            return doctor_dic

        for doctor in DoctorService.get_doctor_from_user_ids_v1(user_ids):
            _user_id = doctor.user_id  # Doctor user_id 数据类型判断
            if isinstance(_user_id, str):
                _doctor_user_id = int(_user_id) if _user_id.isdigit() else 0
            else:
                _doctor_user_id = int(_user_id)

            if doctor.is_online and _doctor_user_id:
                hospital = doctor.hospital

                doctor_dic[_doctor_user_id] = {
                    "user_id": _doctor_user_id,
                    "doctor_id": doctor.id,
                    "doctor_name": doctor.name,
                    "doctor_portrait": doctor.portrait,
                    "doctor_type": doctor.doctor_type,
                    "hospital_id": doctor.hospital_id,
                    "hospital_name": hospital.name,
                }

        return doctor_dic

    @classmethod
    def __get_user_info_by_user_ids(cls, user_ids, simple=True):

        if not user_ids:
            return {}

        _user_info_dic = UserService.get_users_by_user_ids(user_ids)
        return cls.__get_users_info_by_users(_user_info_dic, simple=simple)

    @classmethod
    def get_base_users_info_by_users(cls, _user_info_dic):

        user_dic = {}
        for user_id, user in _user_info_dic.items():
            _data = cls.__get_user_info(user, simple=True)
            user_dic[user_id] = _data

        return user_dic

    @classmethod
    def __get_users_info_by_users(cls, _user_info_dic, simple=True):

        user_dic = {}
        for user_id, user in _user_info_dic.items():
            _data = cls.__get_user_info(user, simple=simple)
            user_dic[user_id] = _data

        return user_dic

    @classmethod
    def __get_user_info(cls, user, simple=True):
        _data = {
            'user_id': user.id,
            'person_id': user.person_id,
            'user_name': user.nickname,
            'membership_level': user.membership_level,
            'portrait': user.portrait,
            'user_level': {
                'membership_icon': user.membership_icon,
                'level_icon': user.level_icon,
                'constellation_icon': user.constellation_icon,
            },
        }

        if not simple:
            _data.update({
                "vote_count": user.vote_count,
                "topic_count": user.topic_count,
                "city_id": user.city_id,
                "city_name": user.city_name,
            })
        return _data

    @classmethod
    def __get_users_info_by_persons(cls, person_info_dic, simple=True):

        person_dict = {}
        for person_id, user in person_info_dic.items():
            _data = cls.__get_user_info(user, simple=simple)
            person_dict[person_id] = _data

        return person_dict

    @classmethod
    def __get_users_info_by_person_ids(cls, person_ids, simple=True):

        if not person_ids:
            return {}

        _user_info_dic = UserService._get_user_data_by_person_ids(person_ids=person_ids)
        person_info_dict = {
            person_id: _user_info_dic[person_id]
            for person_id in person_ids
            if person_id in _user_info_dic
        }
        return cls.__get_users_info_by_persons(person_info_dict, simple=simple)

    # @classmethod
    # def get_doctor_info_by_user_ids(cls, user_ids):
    #     """针对只需要获取医生信息数据的接口"""
    #     return cls.__get_doctor_info_by_user_ids(filter_ids(user_ids))

    @classmethod
    def get_user_info_by_user_ids(cls, user_ids, simple=True):
        """
        通过 user_id 获取用户相关的基本信息。
        包括医生，医院，及医生类型
        :param user_ids:
        :return:
        """
        results = {}
        user_ids = filter_ids(user_ids)

        if not user_ids:
            return results

        # user_info_dic = cls.__get_user_info_by_user_ids(user_ids, simple=simple)
        # doctor_info_dic = cls.__get_doctor_info_by_user_ids(user_ids)
        tasks = Tasks()
        tasks.add('user_info_dic', cls.__get_user_info_by_user_ids, user_ids, simple=simple)
        tasks.add('doctor_info_dic', cls.__get_doctor_info_by_user_ids, user_ids)
        user_info_dic = tasks.jobs['user_info_dic'] or {}
        doctor_info_dic = tasks.jobs['doctor_info_dic'] or {}

        for user_id, user_info in user_info_dic.items():
            doctor_info = doctor_info_dic.get(user_id, {})
            user_info.update({
                "doctor_id": doctor_info.get("doctor_id", ""),
                "doctor_type": doctor_info.get("doctor_type", DOCTOR_TYPE.DOCTOR),
                "hospital_id": doctor_info.get("hospital_id", ""),
            })
            results[user_id] = user_info

        return results

    @classmethod
    def get_user_info_by_person_ids(cls, person_ids, simple=True):
        """
        通过 person_id 获取用户相关的基本信息。
        包括医生，医院，及医生类型
        :param person_ids:
        :return:
        """
        results = {}
        if not person_ids:
            return results

        person_info_dic = cls.__get_users_info_by_person_ids(person_ids, simple=simple)

        user_ids = [item["user_id"] for _, item in person_info_dic.items()]
        doctor_info_dic = cls.__get_doctor_info_by_user_ids(user_ids)

        for person_id, user_info in person_info_dic.items():
            user_id = user_info["user_id"]
            doctor_info = doctor_info_dic.get(user_id, {})
            user_info.update({
                "doctor_id": doctor_info.get("doctor_id", ""),
                "doctor_type": doctor_info.get("doctor_type", DOCTOR_TYPE.DOCTOR),
                "hospital_id": doctor_info.get("hospital_id", ""),
            })
            results[person_id] = user_info

        return results

    @classmethod
    def get_user_info_by_user_id(cls, user_id, simple=True):
        if not user_id:
            return {}
        user_info = cls.get_user_info_by_user_ids([user_id], simple=simple)
        return user_info.get(user_id, {})

    @classmethod
    def format_doctor_analogy_user_info_by_doctor_ids(cls, doctor_ids):
        """
        类比用户格式化医生信息
        :param doctor_ids:
        :return:
        """
        results = {}
        if not doctor_ids:
            return results

        doctors = DoctorService.get_doctor_from_doctor_ids(doctor_ids)
        for doctor_info in doctors:
            if doctor_info.is_online:
                results[doctor_info.id] = {
                    "user_id": int(doctor_info.user_id),
                    "user_name": doctor_info.name,
                    "membership_level": "",
                    "portrait": doctor_info.portrait,
                    'user_level': {
                        'membership_icon': "",
                        'level_icon': "",
                        'constellation_icon': "",
                    },
                    "doctor_id": doctor_info.id,
                    "doctor_type": doctor_info.doctor_type,
                    "hospital_id": doctor_info.hospital_id,
                }

        return results

    @classmethod
    def format_doctor_analogy_user_info_by_doctor_id(cls, doctor_id):
        if not doctor_id:
            return {}
        doctor_infos = cls.format_doctor_analogy_user_info_by_doctor_ids([doctor_id])
        return doctor_infos.get(doctor_id, {})
