# coding=utf8

from __future__ import unicode_literals, absolute_import, print_function

from django.conf import settings
from gm_types.gaia import USER_RIGHTS_LEVEL, USER_TYPE
from helios.rpc import RPCFaultException

from talos.libs.image_utils import get_thumb_path
from talos.libs.utils import convert_to_type_or
from talos.rpc import RPCMixin
from talos.services.models.base import ServiceModelBase


class SimpleUser(RPCMixin, ServiceModelBase):
    """SimpleUser object."""

    __slots__ = [
        'id', 'person_id', 'nickname', 'portrait',
        'current_city_id', 'city_id', 'is_puppet',
        'user_type', 'doctor_id', 'hospital_id', 'doctor_name', 'doctor_portrait',
        'constellation', 'user_rights_level', 'membership_level',
    ]

    def __init__(self, id, person_id, nickname, portrait,
                 topic_count=0, vote_count=0, membership_level=None,
                 current_city_id=None, city_id=None,
                 user_rights_level=USER_RIGHTS_LEVEL.V1,
                 constellation=None, user_level=None,
                 is_puppet=None, user_type=USER_TYPE.NORMAL, doctor_id=None, doctor_type=None,
                 doctor_name=None, doctor_portrait=None, hospital_id=None
        ):
        self.id = convert_to_type_or(id, int, None)
        self.person_id = person_id
        self.nickname = nickname
        self.portrait = get_thumb_path(portrait)

        self.topic_count = convert_to_type_or(topic_count, int, 0)
        self.vote_count = convert_to_type_or(vote_count, int, 0)

        self.membership_level = membership_level
        self.current_city_id = current_city_id
        self.city_id = city_id
        self.user_rights_level = user_rights_level
        self.constellation = constellation
        self.membership_icon = user_level and user_level["membership_icon"]
        self.level_icon = user_level and  user_level["level_icon"]
        self.constellation_icon = user_level and user_level["constellation_icon"]
        self.is_puppet = is_puppet
        self.user_type = user_type
        self.doctor_id = doctor_id
        self.doctor_type = doctor_type
        self.doctor_name = doctor_name
        self.doctor_portrait = doctor_portrait
        self.hospital_id = hospital_id

    def __bool__(self):
        return self.id is not None


class User(RPCMixin, ServiceModelBase):
    """user object.

    properties:
        @id int
        @person_id str
        @nickname str
        @portrait(with -thumb) str

        @topic_count: online topic count int
        @vote_count: votes count received int

        @membership_level str
        @city_id str
        @city_name str
        @city_tag_id int
        @province_tag_id int
        @country_tag_id int
    """

    __slots__ = [
        'id', 'person_id', 'nickname', 'portrait',
        'topic_count', 'vote_count', 'membership_level',
        'current_city_id', 'city_id', 'city_name', 'city_tag_id',
        'province_tag_id', 'country_tag_id', 'user_rights_level', 'constellation', 'membership_icon',
        'level_icon', 'constellation_icon', 'is_puppet',
    ]

    def __init__(self, id, person_id, nickname, portrait,
                 topic_count=0, vote_count=0, membership_level=None,
                 current_city_id=None, city_id=None, city_name=None, city_tag_id=None,
                 province_tag_id=None, country_tag_id=None, user_rights_level=USER_RIGHTS_LEVEL.V1,
                 constellation=None, membership_icon=None, level_icon=None, constellation_icon=None,
                 is_puppet=None, user_type=USER_TYPE.NORMAL):
        self.id = convert_to_type_or(id, int, None)
        self.person_id = person_id
        self.nickname = nickname
        self.portrait = get_thumb_path(portrait)

        self.topic_count = convert_to_type_or(topic_count, int, 0)
        self.vote_count = convert_to_type_or(vote_count, int, 0)

        self.membership_level = membership_level
        self.current_city_id = current_city_id
        self.city_id = city_id
        self.city_name = city_name
        self.city_tag_id = convert_to_type_or(city_tag_id, int, None)
        self.province_tag_id = convert_to_type_or(province_tag_id, int, None)
        self.country_tag_id = convert_to_type_or(country_tag_id, int, None)
        self.user_rights_level = user_rights_level
        self.constellation = constellation
        self.membership_icon = membership_icon
        self.level_icon = level_icon
        self.constellation_icon = constellation_icon
        self.is_puppet = is_puppet
        self.user_type = user_type

    def __bool__(self):
        return self.id is not None

    __nonzero__ = __bool__

    @classmethod
    def get_anonymous_user(cls):
        return cls(None, None, None, None)

    def incr_vote_count(self):
        try:
            self.call_rpc('api/person/incr_vote', user_id=self.id)
            return True

        except RPCFaultException:
            return False

    def decr_vote_count(self):
        try:
            self.call_rpc('api/person/decr_vote', user_id=self.id)
            return True

        except RPCFaultException:
            return False

    def incr_topic_count(self):
        try:
            self.call_rpc('api/person/incr_topic_count', user_id=self.id)
            return True

        except RPCFaultException:
            return False

    def decr_topic_count(self):
        try:
            self.call_rpc('api/person/decr_topic_count', user_id=self.id)
            return True

        except RPCFaultException:
            return False

    @property
    def can_create_video(self):
        from talos.models.topic.videowhitelist import VideoWhiteList

        # 过滤直播黑名单
        try:
            result = self.call_rpc('api/user/is_user_in_blacklist', user_id=self.id)
            if result:
                return False
        except RPCFaultException:
            return False

        if VideoWhiteList.objects.filter(user_id=self.id).exists():
            return True

        # added 7.2.5
        # user rights level > =3,can create video
        if int(self.user_rights_level) >= int(settings.CAN_CREATE_VIDEO_LEVEL):
            return True

        return False


