# coding:utf-8

import os
import os.path
import json
import random
from urllib.parse import urljoin

from django.conf import settings
from django.core.management import BaseCommand, CommandError

from gm_types.mimas import TRACTATE_PLATFORM
from gm_upload.utils.image_utils import Picture

from qa.utils.image import get_image_base_info
from talos.models.tractate import Tractate, TractateImages, TractateVideo
from talos.libs.image_utils import fetch_picture_and_save_to_qiniu_v2
from talos.tasks.tractate import (
    set_tractate_video_water_mark_url,
)
from utils.common import get_data_from_rich_text
from utils.execel import ExcelReader
from utils.rpc import logging_exception

query_map = {
    1: ["下巴", "下颌角", "玻尿酸丰下巴", "假体下巴"],
    2: ["纹眉"],
    3: ["厚唇改薄术", "丰唇"],
    4: ["玻尿酸", "自体脂肪填充", "艾莉薇", "伊婉V", "自体脂肪", "伊婉C"],
    5: ["热玛吉", "超声刀", "热拉提", "除皱", "法令纹", "黄金微针", "嗨体", "埋线"],
    6: ["水光针", "光子嫩肤", "美白针", "果酸焕肤", "微针", "小气泡", "无针水光", "收缩毛孔", "黑脸娃娃", "童颜针"],
    7: ["脱毛", "植发", "激光脱毛", "植发际线"],
    8: ["牙齿矫正", "牙齿美白", "洗牙"],
    9: ["面部吸脂"],
    10: ["瘦腿针", "瘦肩针", "吸脂瘦大腿", "溶脂", "光纤溶脂", "腰腹吸脂", "瘦腿", "溶解酶"],
    11: ["双眼皮修复", "祛眼袋", "埋线双眼皮", "黑眼圈", "切开双眼皮", "泪沟", "下眼睑下至", "开外眼角"],
    12: ["祛斑", "皮秒", "激光祛斑", "蜂巢皮秒"],
    13: ["祛疤"],
    14: ["点阵激光", "祛痘", "微针祛痘坑", "祛痘祛痘印"],
    15: ["点痣", "激光点痣"],
    16: ["私密", "女性私密紧致", "妊娠纹"],
    17: ["美白", "白瓷娃娃", "m22"],
    18: ["隆胸", "丰胸（隆胸）"],
    19: ["手术面部提升", "自体脂肪填充面部", "瘦脸", "溶脂针", "颧骨", "保妥适", "磨骨", "肉毒素", "面部提升", "面部填充", "酒窝", "脸型"],
    20: ["鼻翼缩小", "鼻翼缩小", "鼻基底", "玻尿酸隆鼻", "肋骨鼻", "人中缩短", "埋线隆鼻"],
}

users_map = {
    1: [32778396, 32778395, 32778394, 32778393, 32778376, 32778404, 32778403, 32778401, 32778400, 32778399],
    2: [32778412, 32778411, 32778408, 32778407, 32778406, 32778469, 32778468, 32778466, 32778464, 32778460],
    3: [32778426, 32778425, 32778423, 32778422, 32778421, 32778420, 32778419, 32778417, 32778415, 32778414],
    4: [32778509, 32778497, 32778496, 32778495, 32778494, 32778493, 32778490, 32778488, 32778486, 32778485],
    5: [32778528, 32778526, 32778524, 32778523, 32778521, 32778519, 32778518, 32778516, 32778515, 32778514],
    6: [32778553, 32778550, 32778549, 32778547, 32778545, 32778542, 32778541, 32778539, 32778537, 32778534],
    7: [32778575, 32778574, 32778572, 32778569, 32778568, 32778566, 32778564, 32778562, 32778561, 32778558],
    8: [32778597, 32778593, 32778591, 32778590, 32778588, 32778587, 32778586, 32778583, 32778582, 32778581],
    9: [32778645, 32778644, 32778643, 32778641, 32778640, 32778639, 32778637, 32778636, 32778634, 32778632],
    10: [32778659, 32778658, 32778657, 32778656, 32778654, 32778653, 32778650, 32778649, 32778648, 32778647],
    11: [32778676, 32778675, 32778673, 32778672, 32778667, 32778666, 32778664, 32778662, 32778661, 32778660],
    12: [32778696, 32778694, 32778693, 32778692, 32778689, 32778687, 32778682, 32778680, 32778679, 32778678],
    13: [32778717, 32778716, 32778714, 32778713, 32778712, 32778708, 32778705, 32778703, 32778702, 32778701],
    14: [32778731, 32778729, 32778728, 32778727, 32778726, 32778725, 32778724, 32778723, 32778721, 32778719],
    15: [32778750, 32778748, 32778747, 32778743, 32778740, 32778738, 32778735, 32778734, 32778733, 32778732],
    16: [32778773, 32778765, 32778763, 32778761, 32778759, 32778757, 32778756, 32778754, 32778753, 32778751],
    17: [32778808, 32778807, 32778806, 32778805, 32778803, 32778802, 32778801, 32778800, 32778798, 32778797],
    18: [32778820, 32778818, 32778816, 32778815, 32778814, 32778813, 32778812, 32778811, 32778810, 32778809],
    19: [32778838, 32778835, 32778834, 32778832, 32778829, 32778828, 32778827, 32778826, 32778825, 32778821],
    20: [32778848, 32778847, 32778846, 32778845, 32778844, 32778843, 32778842, 32778841, 32778840, 32778839],
}

class Command(BaseCommand):

    def add_arguments(self, parser):
        parser.add_argument('-p', '--path', dest='path', type=str)

    def get_user_id_by_query(self, query):
        """根据运营给到得query随机对应到特定的用户列表中"""

        idx = None
        for i, qus in query_map.items():
            if query in qus:
                idx = i
                break

        if idx is None:
            raise CommandError("没有找到对应的映射用户")

        return random.choice(users_map[idx])

    def get_tractate_by_platform_id(self, platform_id):

        if not platform_id:
            return None

        try:
            return Tractate.objects.using(settings.SLAVE_DB_NAME).get(platform_id=platform_id)
        except:
            return None

    def create_tractate(self, tractate_info):

        platform_id = tractate_info["platform_id"]
        if self.get_tractate_by_platform_id(platform_id):
            tractate_info.pop("content", None)
            print("数据库已经存在: ", tractate_info)
            return

        image_urls = tractate_info.pop('image_urls', [])
        video_url = tractate_info.pop("video_url", "")
        tractate_obj = Tractate.objects.create(**tractate_info)
        _tractate_id = tractate_obj.id

        images = []
        for url in image_urls:
            if not url.strip(): continue
            info = {
                "tractate_id": _tractate_id,
                "image_url": url,
                "width": 0,
                "height": 0,
            }
            _base_data = get_image_base_info(urljoin(settings.QINIU_CDN_HOST, url))
            info.update(_base_data)
            images.append(TractateImages(**info))

        if images:
            TractateImages.objects.bulk_create(images)

        if video_url:
            _video_cover_url = "{url}{params}".format(url=urljoin(settings.VIDEO_HOST, video_url), params=settings.VIDEO_PIC_URL)
            new_video_caver_url = fetch_picture_and_save_to_qiniu_v2(_video_cover_url)
            info = {
                "tractate_id": _tractate_id,
                "raw_video_url": video_url,
                "video_cover_url": new_video_caver_url,
                "is_online": True,
                "width": 0,
                "height": 0,
            }
            _base_data = Picture.get_image_base_info(new_video_caver_url)
            info.update(_base_data)

            video_obj = TractateVideo.objects.create(**info)
            set_tractate_video_water_mark_url.delay(video_obj.id)

        return _tractate_id

    def find_data_file(self, path):

        result = []
        for root, _, files in os.walk(path):
            for i in files:
                if i.endswith(".xlsx"):
                    result.append(os.path.join(root, i))

        return result

    def get_data(self, file):

        reader = ExcelReader(file)
        datas = reader.read_rows(min_row=2)  # 规定第二列为用户ID

        # col: platform_id, vote_cnt, images_raw, video_url, video_pic, content
        tractates_info = []
        for item in datas:

            content = item[5]
            if not content: continue
            content = content.replace("\\r\\n", "\r\n").strip()
            if not content: continue

            try:
                info = {
                    "platform_id": item[0],
                    "image_urls": item[2].split('|'),
                    "video_url": item[3],
                    "content": content,
                    "is_online": False,
                    "platform": TRACTATE_PLATFORM.SOYOUNG,
                }
            except:
                logging_exception()
                continue
            tractates_info.append(info)

        return reader._current_sheet.title, tractates_info

    def handle(self, *args, **options):

        path = options['path']
        if not path:
            raise CommandError("请指定数据文件目录")

        for f in self.find_data_file(path):

            try:
                query, tractates = self.get_data(f)
            except:
                print("文件异常: ", f)
                continue

            for tractate in tractates:

                user_id = self.get_user_id_by_query(query)
                tractate["user_id"] = user_id
                err = None
                tractate_id = self.create_tractate(tractate)
                if not tractate_id:
                    err = True
                if err:
                    print("创建帖子失败: ", tractate)
