from django.conf import settings
from django.utils import timezone
from gm_types.error import ERROR

from utils.rpc import gen
from .base import BaseService

from talos.models.tractate import TractateFavor, Tractate


class TractateFavorService(BaseService):

    model = TractateFavor

    @classmethod
    def get_tractate_favord_info_for_user(cls, tractate_ids, user_id):
        """
        获取用户对帖子的已收藏的数据
        :param tractate_ids:
        :param user_id:
        :return:
        """
        result = dict()
        if not tractate_ids or not user_id:
            return result

        tfs = list(cls.model.objects.filter(
            tractate_id__in=tractate_ids,
            user_id=user_id,
            is_deleted=False
        ).values_list("tractate_id", flat=True))
        if not tfs:
            return result

        result = {tv: True for tv in tfs}
        return result

    @classmethod
    def create(cls, tractate, user_id):
        """
        新帖子收藏
        :param tractate_id:
        :param user_id:
        :return:
        """

        tf, created = cls.model.objects.get_or_create(
            user_id=user_id, tractate_id=tractate.id,
        )

        filter_bool = True if created or tf.is_deleted else False
        if tf.is_deleted:
            tf.is_deleted = False
            tf.is_read = False
            tf.save()

        return tf, filter_bool

    @classmethod
    def cancel(cls, user_id, tractate):
        """
        新帖子取消收藏
        :param user_id:
        :param tractate:
        :return:
        """
        try:
            tf = cls.model.objects.get(user_id=user_id, tractate_id=tractate.id)
        except TractateFavor.DoesNotExist:
            return gen(ERROR.TRACTATE_UNFAVOR_ERROR)

        cancel_status = True if not tf.is_deleted else False
        if not tf.is_deleted:
            tf.is_deleted = True
            tf.is_read = True
            tf.save()

        return tf, cancel_status

    @classmethod
    def get_unread_favorite_num(cls, user_id):
        tractate_ids = Tractate.objects.filter(user_id=user_id).values_list('id', flat=True)
        if not tractate_ids:
            return 0

        return cls.model.objects.filter(tractate_id__in=tractate_ids, is_read=False, is_deleted=False).count()

    @classmethod
    def tractate_favorite_read(cls, user_id):
        tractate_ids = Tractate.objects.filter(user_id=user_id).values_list('id', flat=True)
        cls.model.objects.filter(tractate_id__in=tractate_ids, is_read=False).update(is_read=True)

    @classmethod
    def tractate_favorite_list(cls, user_id, start_num, count):
        """
        获取新版本帖子ID list
        :param user_id:
        :param start_num:
        :param count:
        :return:
        """
        tractate_ids = list(cls.model.objects.filter(
            user_id=user_id, is_deleted=False
        ).order_by('-id').values_list('tractate_id', flat=True)[start_num: start_num + count])

        return tractate_ids

    @classmethod
    def get_favor_count(cls, user_id):
        tra_ids = list(cls.model.objects.using(settings.SLAVE_DB_NAME).filter(
            user_id=user_id, is_deleted=False).values_list('tractate_id', flat=True))
        return Tractate.objects.using(settings.SLAVE_DB_NAME).filter(id__in=tra_ids, user_del=False, is_online=True).count()
