#!/usr/bin/env python
# -*- coding: utf-8 -*-

from gm_types.gaia import REPLYOBJECT
from django.utils import timezone

from talos.libs.datetime_utils import get_timestamp_or_none


class ReplyTool(object):
    """reply tool for user。"""

    # 日记贴回答 + 新帖子评论 + 问题回答 + 回答评论

    def __init__(self, redis_c, user_id):
        self.user_id = user_id
        self.redis_c = redis_c

        self._topic_reply_key_prefix = 't:'
        self._problem_reply_key_prefix = 't:'
        self._tractate_reply_key_prefix = 'tv:'
        self._answer_key_prefix = 'q:'  # 问题回答
        self._answer_reply_key_prefix = 'a:'  # 回答评论
        self._soft_article_reply_key_prefix = 'srr'  # 医生后台帖子评论 回复
        self._quality_question_key_prefix = 'qua_q:'    # 问答追问
        self._quality_answer_key_prefix = 'qua_a:'  # 问答追问回复
        self._quality_reply_key_prefix = 'qua_r:'  # 问答追问评论

        self._reply_key_prefix_format = 'user:{user_id}:replies:received'
        self._k = self._reply_key_prefix_format.format(user_id=user_id)

    def _cache_this(self, k):
        now = timezone.now()
        score = get_timestamp_or_none(now)
        self.redis_c.zadd(self._k, score, k)

    def receive_topic_reply(self, topicreply_id):
        v = self._topic_reply_key_prefix + str(topicreply_id)
        self._cache_this(v)

    def remove_topic_reply(self, topicreply_id):
        v = self._topic_reply_key_prefix + str(topicreply_id)
        self._del_cache(v)

    def receive_problem_reply(self, problemreply_id):
        v = self._problem_reply_key_prefix + str(problemreply_id)
        self._cache_this(v)

    def remove_problem_reply(self, problemreply_id):
        v = self._problem_reply_key_prefix + str(problemreply_id)
        self._del_cache(v)

    def receive_tractate_reply(self, tractate_reply_id):
        v = self._tractate_reply_key_prefix + str(tractate_reply_id)
        self._cache_this(v)

    def remove_tractate_reply(self, tractate_reply_id):
        v = self._tractate_reply_key_prefix + str(tractate_reply_id)
        self._del_cache(v)

    def receive_answer(self, answer_id):
        v = self._answer_key_prefix + str(answer_id)
        self._cache_this(v)

    def remove_answer(self, answer_id):
        v = self._answer_key_prefix + str(answer_id)
        self._del_cache(v)

    def receive_answer_reply(self, answer_id):
        v = self._answer_reply_key_prefix + str(answer_id)
        self._cache_this(v)

    def remove_answer_reply(self, answer_id):
        v = self._answer_reply_key_prefix + str(answer_id)
        self._del_cache(v)

    def receive_soft_article_reply(self, soft_article_reply_id):
        v = self._soft_article_reply_key_prefix + str(soft_article_reply_id)
        self._del_cache(v)

    def receive_quality_question(self, quality_question_id):
        v = self._quality_question_key_prefix + str(quality_question_id)
        self._cache_this(v)

    def remove_quality_question(self, tractate_reply_id):
        v = self._quality_question_key_prefix + str(tractate_reply_id)
        self._del_cache(v)

    def receive_quality_answer(self, quality_answer_id):
        v = self._quality_answer_key_prefix + str(quality_answer_id)
        self._cache_this(v)

    def remove_quality_answer(self, quality_answer_id):
        v = self._quality_answer_key_prefix + str(quality_answer_id)
        self._del_cache(v)

    def receive_quality_reply(self, quality_reply_id):
        v = self._quality_reply_key_prefix + str(quality_reply_id)
        self._cache_this(v)

    def remove_quality_reply(self, quality_reply_id):
        v = self._quality_reply_key_prefix + str(quality_reply_id)
        self._del_cache(v)

    def get_replies_received(self, start_num=0, count=10):
        """return list of tuple as [(id, type), ]"""
        result = self.redis_c.zrevrange(self._k, start_num, start_num + count - 1)
        if not result:
            return []

        prefix2type = {
            self._topic_reply_key_prefix: REPLYOBJECT.TOPIC,
            self._tractate_reply_key_prefix: REPLYOBJECT.TRACTATE,
            self._answer_key_prefix: REPLYOBJECT.ANSWER,
            self._answer_reply_key_prefix: REPLYOBJECT.ANSWER_REPLY,
            self._soft_article_reply_key_prefix: REPLYOBJECT.SOFT_ARTICLE,
            self._quality_question_key_prefix: REPLYOBJECT.QUALITY_QUESTION,
            self._quality_answer_key_prefix: REPLYOBJECT.QUALITY_ANSWER,
            self._quality_reply_key_prefix: REPLYOBJECT.QUALITY_REPLY,
        }

        res = []
        for r in result:
            prefix, v = r.decode("utf-8").split(':')   # 数据类型，byte -> str
            prefix += ':'
            _type = prefix2type.get(prefix)
            if _type:
                res.append((v, _type))

        return res

    def get_reply_count(self):
        return self.redis_c.zcard(self._k)

    def _del_cache(self, v):
        self.redis_c.zrem(self._k, v)
