# -*- coding: utf-8 -*-
from __future__ import unicode_literals, absolute_import, print_function

from django.db import models
from django.utils import timezone
from gm_types.gaia import HOSPITAL_TYPE
from gm_types.doctor import DOCTOR_TITLE_TYPE, HOSPITAL_LEVEL
from gm_upload import ImgUrlField, IMG_TYPE

from .answer import Question

from qa.utils.time import get_timestamp_epoch

class BDAnswer(models.Model):
    class Meta:
        app_label = 'qa'
        db_table = 'bd_answer'
        verbose_name = u'百度熊掌号回答'

    original_id = models.IntegerField(verbose_name=u'APP内部原回答id')
    question = models.ForeignKey(Question, verbose_name=u"所属问题", related_name='bd_answers')
    user_id = models.IntegerField(verbose_name=u'用户ID')
    name = models.CharField(max_length=100, null=True, verbose_name='医生昵称', blank=True, default='')
    portrait = ImgUrlField(img_type=IMG_TYPE.DOCTOR, verbose_name='医生头像', max_length=200, blank=True, default='')
    doctor_id = models.CharField(max_length=100, verbose_name=u'医生ID')
    doctor_title = models.CharField(max_length=2, verbose_name=u"医生职称", choices=DOCTOR_TITLE_TYPE,
                             default=DOCTOR_TITLE_TYPE.EMPTY)
    doctor_department = models.CharField(max_length=50, default="", verbose_name=u'医院科室')
    hospital_type = models.CharField(max_length=1, verbose_name=u'医院类型', null=False, blank=False,
                                     default=HOSPITAL_TYPE.PUBLIC, choices=HOSPITAL_TYPE)
    hospital_name = models.CharField(max_length=100, verbose_name="医院名称")
    hospital_level = models.IntegerField(verbose_name=u'医院等级', choices=HOSPITAL_LEVEL, default=HOSPITAL_LEVEL.NONE)
    content = models.TextField(verbose_name='回答', null=False)
    is_recommend = models.BooleanField(verbose_name='是否推荐', default=False)
    like_num = models.IntegerField(verbose_name='点赞数量', default=0)
    is_submit = models.BooleanField(verbose_name='是否已提交', default=False)
    create_time = models.DateTimeField(verbose_name=u'创建时间', default=timezone.now)
    update_time = models.DateTimeField(verbose_name=u'更新时间', auto_now=True)

    def data_for_list(self):
        data = {
            "id": self.id,
            "answer_id": self.original_id,  # 原回答id
            "question_id": self.question_id,  # 对应问题id
            "raw_content": self.content,  # 兼容
            "create_time": get_timestamp_epoch(self.create_time),  # 时间戳，回答的创建时间
            "doctor_info": {
                "user_id": self.user_id,
                "id": self.doctor_id,  # 医生id 兼容处理
                "doctor_id": self.doctor_id,
                "name": self.name,  # 医生名字
                "portrait": self.portrait,  # 医生头像
                "title": self.doctor_title,  # 职称
                "department": self.doctor_department,  # 科室
                "hospital_name": self.hospital_name,  # 医院名
                "hospital_level": self.hospital_level,  # 医院等级
                # 兼容处理，上线以后删除
                "hospital_id": "",
                "hospital": {"name": ""},
            }
        }
        return data