from django.db import models

from gm_upload import ImgUrlField, IMG_TYPE
from gm_types.mimas import QUALITY_QUESTION_CLASSIFY_TYPE


class BaseModel(models.Model):

    class Meta:
        abstract = True

    create_time = models.DateTimeField(verbose_name='创建时间', auto_now_add=True)
    update_time = models.DateTimeField(verbose_name='更新时间', auto_now=True)


class QualityQuestionPool(BaseModel):

    class Meta:
        verbose_name = '问题池'
        db_table = 'api_quality_question_pool'
        app_label = 'qa'

    title = models.CharField(max_length=256, null=False, verbose_name=u'问题标题')
    is_online = models.BooleanField(verbose_name="是否有效", default=True)


class QualityClassfyQuestion(BaseModel):

    class Meta:
        verbose_name = '绑定的问题'
        db_table = 'api_quality_classfy_question'
        app_label = 'qa'

    classify_type = models.SmallIntegerField(verbose_name='问题类型', default=QUALITY_QUESTION_CLASSIFY_TYPE.COMMON, choices=QUALITY_QUESTION_CLASSIFY_TYPE)
    tag_id = models.IntegerField(verbose_name='标签ID', db_index=True)
    tag_group_id = models.IntegerField(verbose_name='标签组ID', db_index=True)
    is_online = models.BooleanField(verbose_name="是否有效", default=True)


class QualityQuestionClassifyRelation(BaseModel):
    class Meta:
        verbose_name = '问题和类别映射'
        db_table = 'api_quality_classify_map'
        app_label = 'qa'

    question_id = models.IntegerField(verbose_name='标签ID', db_index=True)
    quality_classify_id = models.IntegerField('绑定的类别id', db_index=True)


class QualityQuestion(BaseModel):

    class Meta:
        verbose_name = '特殊问答'
        db_table = 'api_quality_question'
        unique_together = ('answer_id', 'question_id')
        app_label = 'qa'

    answer_id = models.IntegerField(verbose_name='作者回答ID', db_index=True)
    question_id = models.IntegerField(verbose_name='用户追问问题ID', db_index=True)
    ask_cnt = models.IntegerField(verbose_name='被提问次数', default=0)
    answer_cnt = models.IntegerField(verbose_name='回答次数', default=0)
    vote_cnt = models.IntegerField(verbose_name='点赞次数', default=0)

    is_online = models.BooleanField(verbose_name="是否在线", default=True)


class QualityUserQuestion(BaseModel):

    class Meta:
        verbose_name = '用户追问问题'
        db_table = 'api_quality_user_question'
        index_together = ('answer_id', 'user_id')
        app_label = 'qa'

    quality_question_id = models.IntegerField(verbose_name='问题下特殊问题ID', db_index=True)
    answer_id = models.IntegerField(verbose_name='作者回答ID', db_index=True)
    question_id = models.IntegerField(verbose_name='冗余添加用户追问问题ID', db_index=True)
    user_id = models.IntegerField(verbose_name='用户ID', db_index=True)

    is_online = models.BooleanField(verbose_name="是否在线", default=True)


class QualityQuestionVote(BaseModel):

    class Meta:
        verbose_name = '特殊问题点赞'
        db_table = 'api_quality_question_vote'
        app_label = 'qa'

    quality_question_id = models.IntegerField(verbose_name='问题下特殊问题ID', db_index=True)
    user_id = models.IntegerField(verbose_name='用户ID', db_index=True)

    is_online = models.BooleanField(verbose_name="是否有效", default=True)


class QualityAuthorAnswer(BaseModel):

    class Meta:
        verbose_name = '特殊问答-作者回答'
        db_table = 'api_quality_author_answer'
        app_label = 'qa'

    quality_question_id = models.IntegerField(verbose_name='问题下特殊问题ID', db_index=True)
    user_id = models.IntegerField(verbose_name='作者用户ID', db_index=True)
    content = models.TextField(verbose_name='回答', null=False)

    is_online = models.BooleanField(verbose_name="是否有效", default=True)


class QualityAuthorAnswerImage(BaseModel):

    class Meta:
        verbose_name = '特殊问答-作者回答图片'
        db_table = 'api_quality_author_answer_image'
        app_label = 'qa'

    quality_answer_id = models.IntegerField(verbose_name='问题下特殊问题ID', db_index=True)
    url = ImgUrlField(img_type=IMG_TYPE.TOPICIMAGE, max_length=300, verbose_name=u'图片地址')
    width = models.IntegerField(verbose_name="图片宽度", default=0)
    height = models.IntegerField(verbose_name="图片高度", default=0)

    is_online = models.BooleanField(verbose_name="是否有效", default=True)


class QualityReply(BaseModel):

    class Meta:
        verbose_name = '评论'
        db_table = 'api_quality_reply'
        app_label = 'qa'

    quality_question_id = models.IntegerField(verbose_name='问题下特殊问题ID', db_index=True)
    quality_answer_id = models.IntegerField(verbose_name='被评论的回答ID', db_index=True)
    top_id = models.IntegerField(verbose_name='评论的一级评论ID', db_index=True)
    replied_id = models.IntegerField(verbose_name='被评论的评论ID', db_index=True)

    user_id = models.IntegerField(verbose_name='用户ID', db_index=True)
    content = models.TextField(verbose_name='回答', null=False)

    is_online = models.BooleanField(verbose_name="是否有效", default=True)


class QualityReplyImage(BaseModel):

    class Meta:
        verbose_name = '特殊问答-作者回答图片'
        db_table = 'api_quality_reply_image'
        app_label = 'qa'

    quality_reply_id = models.IntegerField(verbose_name='评论ID', db_index=True)
    url = ImgUrlField(img_type=IMG_TYPE.TOPICIMAGE, max_length=300, verbose_name=u'图片地址')
    width = models.IntegerField(verbose_name="图片宽度", default=0)
    height = models.IntegerField(verbose_name="图片高度", default=0)

    is_online = models.BooleanField(verbose_name="是否有效", default=True)


class QualityQuestionRead(BaseModel):

    class Meta:
        verbose_name = '用户提问给作者写未读数据'
        db_table = 'api_quality_question_read'
        app_label = 'qa'

    quality_question_id = models.IntegerField(verbose_name='问题下特殊问题ID', db_index=True)
    user_id = models.IntegerField(verbose_name='用户ID', db_index=True)
    is_read = models.BooleanField(verbose_name="是否已读", default=False)


class QualityAnswerRead(BaseModel):

    class Meta:
        verbose_name = '作者回答给用户写未读数据'
        db_table = 'api_quality_answer_read'
        app_label = 'qa'

    quality_question_id = models.IntegerField(verbose_name='问题下特殊问题ID', db_index=True)
    user_id = models.IntegerField(verbose_name='用户ID', db_index=True)
    is_read = models.BooleanField(verbose_name="是否已读", default=False)
