# coding=utf8
from __future__ import unicode_literals, absolute_import, print_function

import json
import datetime
from django.conf import settings
from django.db import models
from django.utils import timezone
from gm_types.gaia import IMAGE_TYPE

from talos.libs.datetime_utils import get_timestamp, get_timestamp_epoch
from talos.models.diary.diary import Diary
from talos.services.tag import TagService
from talos.services.user import UserService


class Draft(models.Model):
    class Meta:
        verbose_name = u'草稿箱'
        verbose_name_plural = u'草稿箱'
        app_label = 'talos'
        db_table = 'api_draft'

    update_time = models.DateTimeField(verbose_name=u'修改时间', auto_now=True)
    is_show = models.BooleanField(default=False, verbose_name=u'是否展示')
    diary = models.ForeignKey(Diary, related_name='drafts', null=True)
    content = models.TextField(default='', blank=True, verbose_name=u'帖子内容')
    tags = models.CharField(max_length=64, null=True, verbose_name=u'tag列表json形式')
    images = models.TextField(default='', blank=True, verbose_name=u'图片列表json形式')
    zone_tag = models.CharField(max_length=6, null=True, verbose_name=u'发帖圈子tag id')
    cover = models.CharField(max_length=128, null=True, verbose_name=u'封面图')
    person_id = models.UUIDField(editable=False, verbose_name=u'用户外键id')
    video = models.CharField(max_length=128, null=True, verbose_name=u'视频地址')
    # v 7.6.75 新加字段 用于记录用户自己选择的术后日期
    operation_date = models.DateTimeField(db_index=True, null=True, verbose_name=u"用户选择的术后日期", default=timezone.now)
    deliver_polymer = models.CharField(max_length=128, null=True, verbose_name=u'投稿到小组')

    def info(self, simple=True):
        draft_info = {
            'id': self.id,
            'diary_id': str(self.diary.id) if self.diary else '',
            'content': self.content,
            'time': get_timestamp(self.update_time),  #草稿的更新时间
            'diary_title':str(self.diary.title) if self.diary else '',
            'images': json.loads(self.images) if self.images else [],
            'video': self.video,
            'deliver_polymer': json.loads(self.deliver_polymer) if self.deliver_polymer else [],
        }
        if not simple:
            if self.images:
                images = json.loads(self.images)
            else:
                images = []
            if self.tags:
                tags = json.loads(self.tags)
            else:
                tags = []
            draft_info['zone_tag_id'] = int(self.zone_tag) if self.zone_tag else None
            draft_info['images'] = []
            draft_info['tags'] = []
            draft_info['operation_date'] = get_timestamp_epoch(self.operation_date)
            for image in images:
                if image:
                    draft_info['images'].append(
                        {
                            'image': image,
                            'is_cover': self.cover == image,
                            'image_type': IMAGE_TYPE.POST_OPERATION_IMAGE,
                        }
                    )
            tag_objects = TagService.get_tags_by_tag_ids(tags)

            for tag in tag_objects:
                draft_info['tags'].append(
                    {
                        'tag_id': tag.id,
                        'name': tag.name,
                    }
                )
            if self.video:
                draft_info['video_url'] = settings.VIDEO_HOST + self.video
                draft_info['video_cover'] = settings.VIDEO_HOST + self.video + settings.VIDEO_PIC_URL
            else:
                draft_info['video_url'] = ''
                draft_info['video_cover'] = ''

            # user_id = UserService.get_user_id_by_person_id(self.person_id.hex)
            user = UserService.get_user_by_person_id(self.person_id.hex)
            if user is None:
                draft_info['can_create_video'] = False

            else:
                draft_info['can_create_video'] = user.can_create_video

        return draft_info

    @classmethod
    def cleaned_video_url(cls, video_url):
        return video_url.replace(settings.VIDEO_HOST, '')
