"""收藏缓存"""

from gm_types.gaia import FAVOROBJECT
from django.utils import timezone

from talos.libs.datetime_utils import get_timestamp_or_none


class FavorTool(object):
    """收藏缓存。"""

    def __init__(self, redis_c, user_id):
        self.user_id = user_id
        self.redis_c = redis_c

        self._diary_favor_key_prefix = 'd:'
        self._problem_favor_key_prefix = 'p:'
        self._topic_favor_key_prefix = 't:'
        self._tractate_favor_key_prefix = 'tf:'
        self._soft_article_favor_key_prefix = 'sf:'
        self._question_favor_key_prefix = 'qf:'          # 问题收藏
        self._answer_favor_key_prefix = 'af:'            # 回答收藏

        self._favor_key_prefix_format = 'user:{user_id}:favors'

        self._k = self._favor_key_prefix_format.format(user_id=user_id)

    def _cache_this(self, k):
        now = timezone.now()
        score = get_timestamp_or_none(now)
        self.redis_c.zadd(self._k, score, k)

    def receive_diary_favor(self, diary_favor_id):

        v = self._diary_favor_key_prefix + str(diary_favor_id)
        self._cache_this(v)

    def receive_problem_favor(self, problem_favor_id):

        v = self._problem_favor_key_prefix + str(problem_favor_id)
        self._cache_this(v)

    def receive_topic_favor(self, topic_favor_id):

        v = self._topic_favor_key_prefix + str(topic_favor_id)
        self._cache_this(v)

    def receive_tractate_favor(self, tractate_favor_id):

        v = self._tractate_favor_key_prefix + str(tractate_favor_id)
        self._cache_this(v)

    def receive_soft_article_favor(self, soft_article_favor_id):
        v = self._soft_article_favor_key_prefix + str(soft_article_favor_id)
        self._cache_this(v)

    def receive_question_favor(self, question_id):
        v = self._question_favor_key_prefix + str(question_id)
        self._cache_this(v)

    def receive_answer_favor(self, answer_id):
        v = self._answer_favor_key_prefix + str(answer_id)
        self._cache_this(v)

    def get_favors_received(self, start_num=0, count=10):

        result = self.redis_c.zrevrange(self._k, start_num, start_num + count - 1)
        if not result:
            return []

        res = []
        for r in result:
            t, v = r.decode("utf-8").split(':')   # 数据类型，byte -> str
            t += ':'
            if t == self._diary_favor_key_prefix:
                res.append((v, FAVOROBJECT.DIARY))

            elif t == self._problem_favor_key_prefix:
                res.append((v, FAVOROBJECT.PROBLEM))

            elif t == self._topic_favor_key_prefix:
                res.append((v, FAVOROBJECT.TOPIC))

            elif t == self._tractate_favor_key_prefix:
                res.append((v, FAVOROBJECT.TRACTATE))
            elif t == self._question_favor_key_prefix:
                res.append((v, FAVOROBJECT.QUESTION))
            elif t == self._answer_favor_key_prefix:
                res.append((v, FAVOROBJECT.ANSWER))

        return res

    def get_favor_count(self):

        return self.redis_c.zcard(self._k)

    def _del_cache(self, v):
        self.redis_c.zrem(self._k, v)

    def remove_diary_favor(self, diary_favor_id):
        v = self._diary_favor_key_prefix + str(diary_favor_id)
        self._del_cache(v)

    def remove_topic_favor(self, topic_favor_id):
        v = self._topic_favor_key_prefix + str(topic_favor_id)
        self._del_cache(v)

    def remove_tractate_favor(self, tractate_favor_id):
        v = self._tractate_favor_key_prefix + str(tractate_favor_id)
        self._del_cache(v)

    def remove_soft_article_favor(self, soft_article_favor_id):
        v = self._soft_article_favor_key_prefix + str(soft_article_favor_id)
        self._del_cache(v)

    def remove_question_favor(self, question_id):
        v = self._question_favor_key_prefix + str(question_id)
        self._del_cache(v)

    def remove_answer_favor(self, answer_id):
        v = self._answer_favor_key_prefix + str(answer_id)
        self._del_cache(v)
