# -*- coding: utf-8 -*-
import abc
import json
import time
import pprint

import yaml
from tencentcloud.common import credential
from tencentcloud.common.exception.tencent_cloud_sdk_exception import \
    TencentCloudSDKException
from tencentcloud.common.profile.client_profile import ClientProfile
from tencentcloud.common.profile.http_profile import HttpProfile
from tencentcloud.monitor.v20180724 import models, monitor_client

from conf import config
from utils.time_tools import timestamp2isoformat

class Tencent(object):
    def __init__(self):
        self.secret_id = config['tencent_settings']['secret_id']
        self.secret_key = config['tencent_settings']['secret_key']
        self.cred = credential.Credential(self.secret_id, self.secret_key)

    @abc.abstractmethod
    def _get_client(self):
        """ get client
        """

    def json_format(self, response):
        if isinstance(response, bytes):
            response = response.decode()
        return json.loads(response)


class Monitor(Tencent):
    def __init__(self, region="ap-beijing", endpoint="monitor.tencentcloudapi.com"):
        super(Monitor, self).__init__()
        self.client = self._get_client(region, endpoint)

    def _get_client(self, region, endpoint):
        httpProfile = HttpProfile()
        httpProfile.endpoint = endpoint
        clientProfile = ClientProfile()
        clientProfile.httpProfile = httpProfile
        client = monitor_client.MonitorClient(self.cred, region, clientProfile)
        return client

    def _describe_base_metrics(self, params_dict):
        assert isinstance(params_dict, dict)
        req = models.DescribeBaseMetricsRequest()
        req.from_json_string(json.dumps(params_dict))
        resp = self.client.DescribeBaseMetrics(req)
        return self.json_format(resp.to_json_string())

    def _get_monitor_data(self, params_dict):
        assert isinstance(params_dict, dict)
        req = models.GetMonitorDataRequest()
        req.from_json_string(json.dumps(params_dict))
        resp = self.client.GetMonitorData(req)
        return self.json_format(resp.to_json_string())

    def get_cdb_metric(self, instance_id_list, metric, duration=60*5):
        """ 批量获取腾讯云mysql实例监控指标
        :param instance_id_list: mysql实例列表
        :type instance_id_list: list
        :param metric: 指标名称，详见腾讯云文档 https://cloud.tencent.com/document/api/248/30386#2.2-.E6.8C.87.E6.A0.87.E5.90.8D.E7.A7.B0
        :type metric: string
        :param duration: 过去多长时间监控数据的时间段时长, 单位(s)
        :type duration: int
        """
        now = int(time.time())
        end_time_str = timestamp2isoformat(now)
        start_time_str = timestamp2isoformat(now - duration)
        params_dict = {
            "Namespace": "QCE/CDB",
            "MetricName": metric,
            "Period": 5,
            "Instances": [
                {
                    "Dimensions": [
                        {
                            "Name": "InstanceId",
                            "Value": instance_id
                        }
                    ]
                } for instance_id in instance_id_list
            ],
            "StartTime": start_time_str,
            "EndTime": end_time_str
        }
        resp = self._get_monitor_data(params_dict)
        return resp
