#!/usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import unicode_literals, absolute_import, print_function

from django.conf import settings
from django.core.management.base import BaseCommand, CommandError
import traceback
import logging
from libs.es import ESPerform
from django.db import models
import datetime

from alpha_types.venus import GRAP_PLATFORM

from .pick_topic import PickTopic
from .tag import TopicTag, Tag
from .user_extra import UserExtra
from .group import Group


class ActionSumAboutTopic(models.Model):
    class Meta:
        verbose_name = u"帖子埋点数据汇总"
        db_table = "action_sum_about_topic"

    partiton_date = models.CharField(verbose_name=u"日期", max_length=20)
    device_id = models.CharField(verbose_name=u"用户设备号", max_length=50)
    topic_id = models.CharField(verbose_name=u"帖子ID", max_length=50)
    user_id = models.CharField(verbose_name=u"用户ID", max_length=50)
    data_type = models.IntegerField(verbose_name=u"动作类型")
    data_value = models.BigIntegerField(verbose_name=u"值")


class TopicImage(models.Model):
    class Meta:
        verbose_name = u'日记图片'
        db_table = 'topic_image'

    id = models.IntegerField(verbose_name='日记图片ID', primary_key=True)
    topic_id = models.IntegerField(verbose_name=u'日记ID')
    url = models.CharField(verbose_name=u'图片URL', max_length=300)
    is_online = models.BooleanField(verbose_name='是否上线')
    is_deleted = models.BooleanField(verbose_name='是否删除')
    create_time = models.DateTimeField(verbose_name=u'创建时间', default=datetime.datetime.fromtimestamp(0))
    update_time = models.DateTimeField(verbose_name=u'更新时间', default=datetime.datetime.fromtimestamp(0))


class ExcellentTopic(models.Model):
    """优质帖子"""

    class Meta:
        verbose_name = "优质帖子"
        db_table = "excellent_topic"

    id = models.IntegerField(verbose_name=u'ID', primary_key=True)
    topic_id = models.IntegerField(verbose_name=u"帖子ID", db_index=True)
    is_online = models.BooleanField(verbose_name=u'是否上线')
    is_deleted = models.BooleanField(verbose_name=u'是否删除')
    excellent_type = models.IntegerField(verbose_name=u"优质类型", db_index=True)
    create_time = models.DateTimeField(verbose_name=u'创建时间', default=datetime.datetime.fromtimestamp(0))
    update_time = models.DateTimeField(verbose_name=u'更新时间', default=datetime.datetime.fromtimestamp(0))


class Topic(models.Model):
    class Meta:
        verbose_name = u'日记'
        db_table = 'topic'

    id = models.IntegerField(verbose_name=u'日记ID', primary_key=True)
    name = models.CharField(verbose_name=u'日记名称', max_length=100)
    # group_id = models.IntegerField(verbose_name='用户所在组ID',default=-1)
    group = models.ForeignKey(
        Group, verbose_name=u"关联的小组", related_name=u"group_topics", null=True, blank=True, default=None,
        on_delete=models.CASCADE)
    user_id = models.IntegerField(verbose_name=u'用户ID')
    has_video = models.BooleanField(verbose_name=u'是否是视频日记')
    drop_score = models.IntegerField(verbose_name=u'人工赋分', default=0)
    description = models.CharField(verbose_name=u'日记本描述', max_length=200)
    content = models.CharField(verbose_name=u'日记本内容', max_length=1000)
    share_num = models.IntegerField(verbose_name='')
    vote_num = models.IntegerField(verbose_name=u'点赞数')
    reply_num = models.IntegerField(verbose_name=u'回复数')
    cover = models.CharField(verbose_name='', max_length=200)
    is_online = models.BooleanField(verbose_name=u'是否上线')
    is_deleted = models.BooleanField(verbose_name=u'是否删除')
    content_level = models.CharField(verbose_name=u'内容等级', max_length=3)
    language_type = models.IntegerField(verbose_name=u'语种类型')
    create_time = models.DateTimeField(verbose_name=u'日记创建时间', default=datetime.datetime.fromtimestamp(0))
    update_time = models.DateTimeField(verbose_name=u'日记更新时间', default=datetime.datetime.fromtimestamp(0))

    is_crawl = models.BooleanField(verbose_name="是否是爬回的数据", default=False)
    has_image = models.BooleanField(verbose_name=u"是否有图片", default=False)
    virtual_content_level = models.CharField(verbose_name=u'模拟内容等级', max_length=100, default='')
    like_num_crawl = models.IntegerField(verbose_name=u'爬取点赞数', default=0)
    comment_num_crawl = models.IntegerField(verbose_name=u'爬取评论数', default=0)
    is_recommend = models.IntegerField(verbose_name=u'是否推荐 0 非推荐 1 推荐', default=0)
    sort_score = models.FloatField(verbose_name='排序分', default=0)
    is_shadow = models.BooleanField(verbose_name="是否是马甲账户", default=False)
    platform = models.IntegerField(verbose_name=u'平台来源', choices=GRAP_PLATFORM, default=GRAP_PLATFORM.ALPHA)
    platform_id = models.BigIntegerField(verbose_name='用平台ID', null=True)

    def get_virtual_vote_num(self):

        try:
            topic_extra = TopicExtra.objects.filter(topic_id=self.id).first()
            virtual_vote_num = topic_extra.virtual_vote_num if topic_extra else 0

            return virtual_vote_num
        except:
            logging.error("catch exception,err_msg:%s" % traceback.format_exc())
            return 0

    def get_pictorial_id(self):
        try:
            pictorial_id_list = []
            logging.info("get pictorial_id:%s" % self.id)

            pictorial_id = PictorialTopic.objects.filter(topic_id=self.id).values_list("pictorial_id", flat=True)
            logging.info("get pictorial_id:%s" % pictorial_id_list)

            for i in pictorial_id:
                pictorial_id_list.append(i)

            logging.info("get pictorial_id:%s" % pictorial_id_list)
            return pictorial_id_list
        except:
            logging.error("catch exception,err_msg:%s" % traceback.format_exc())
            return []

    @property
    def is_complaint(self):
        """是否被举报"""
        try:
            if TopicComplaint.objects.filter(topic_id=self.id, is_online=True).exists():
                return True

            return False
        except:
            logging.error("catch exception,err_msg:%s" % traceback.format_exc())
            return False

    @classmethod
    def get_topic_image_num(cls, topic_id):
        """
        :remark 获取指定帖子的图片数量
        :param topic_id:
        :return:
        """
        try:
            query_list = list(
                TopicImage.objects.using(settings.SLAVE_DB_NAME).filter(topic_id=topic_id, is_deleted=False,
                                                                        is_online=True).values_list("url", flat=True))
            return len(query_list)
        except:
            logging.error("catch exception,err_msg:%s" % traceback.format_exc())
            return 0

    def topic_has_image(self):
        try:
            has_image = False

            query_list = TopicImage.objects.using(settings.SLAVE_DB_NAME).filter(topic_id=self.id, is_deleted=False,
                                                                                 is_online=True)
            if len(query_list) > 0:
                has_image = True

            return has_image
        except:
            logging.error("catch exception,err_msg:%s" % traceback.format_exc())
            return False

    def get_pick_id_info(self):
        try:
            pick_id_list = list()
            query_list = PickTopic.objects.using(settings.SLAVE_DB_NAME).filter(topic_id=self.id, is_deleted=False)
            for item in query_list:
                pick_id_list.append(item.pick_id)

            return tuple(pick_id_list)
        except:
            logging.error("catch exception,err_msg:%s" % traceback.format_exc())
            return ()

    def get_topic_tag_id_list(self):
        try:
            topic_tag_id_list = list()
            edit_tag_id_list = list()

            tag_id_list = TopicTag.objects.using(settings.SLAVE_DB_NAME).filter(topic_id=self.id).values_list("tag_id",
                                                                                                              flat=True)
            tag_query_results = Tag.objects.using(settings.SLAVE_DB_NAME).filter(id__in=tag_id_list)
            for tag_item in tag_query_results:
                is_online = tag_item.is_online
                is_deleted = tag_item.is_deleted
                collection = tag_item.collection

                if is_online and not is_deleted:
                    topic_tag_id_list.append(tag_item.id)
                    if collection:
                        edit_tag_id_list.append(tag_item.id)

            return (topic_tag_id_list, edit_tag_id_list)
        except:
            logging.error("catch exception,err_msg:%s" % traceback.format_exc())
            return ([], [])

    def get_topic_useful_tag_id_list(self):
        try:
            topic_useful_tag_id_list = list()
            tag_id_list = TopicTag.objects.using(settings.SLAVE_DB_NAME).filter(topic_id=self.id, is_online=True,
                                                                                is_body_esthetics=1)
            for tag_item in tag_id_list:
                topic_useful_tag_id_list.append(tag_item.tag_id)
            return topic_useful_tag_id_list
        except:
            logging.error("catch exception,err_msg:%s" % traceback.format_exc())
            return []

    def get_tag_name_list(self, tag_id_list):
        try:
            tag_name_list = list()
            for i in range(0, len(tag_id_list), 100):
                query_terms_list = list(
                    Tag.objects.using(settings.SLAVE_DB_NAME).filter(id__in=tag_id_list[i:i + 100], is_online=True,
                                                                     is_deleted=False).values_list("name", flat=True))
                tag_name_list.extend(query_terms_list)
            return tag_name_list
        except:
            logging.error("catch exception,err_msg:%s" % traceback.format_exc())
            return []

    def judge_if_excellent_topic(self, topic_id):
        try:
            excelllect_object = ExcellentTopic.objects.filter(topic_id=topic_id).first()
            if excelllect_object and excelllect_object.is_online and not excelllect_object.is_deleted:
                return True
            else:
                return False
        except:
            logging.error("catch exception,err_msg:%s" % traceback.format_exc())
            return False

    def get_topic_offline_score(self):
        try:
            offline_score = 0.0
            user_is_shadow = False

            # 是否官方推荐用户
            user_query_results = UserExtra.objects.using(settings.SLAVE_DB_NAME).filter(user_id=self.user_id)
            if user_query_results.count() > 0:
                if user_query_results[0].is_recommend:
                    offline_score += 2.0 * 10
                elif user_query_results[0].is_shadow:
                    user_is_shadow = True

            # 帖子等级
            if self.content_level == '5':
                offline_score += 100.0 * 3
            elif self.content_level == '4':
                offline_score += 60.0 * 3
            elif self.content_level == '6':
                offline_score += 200.0 * 3

            if self.language_type == 1:
                offline_score += 60.0

            """
                1:马甲账号是否对总分降权？
            """
            if user_is_shadow:
                offline_score = offline_score * 0.5

            offline_score -= self.drop_score
            return offline_score
        except:
            logging.error("catch exception,err_msg:%s" % traceback.format_exc())
            return 0.0

    def get_related_billboard(self):
        try:

            pictorial_ids = Topic.get_pictorial_id(self)

            related_billboard_list = list()

            query_results = TopicBillBoard.objects.filter(pictorial_id__in=pictorial_ids)

            for items in query_results:
                total_vote_cnt = int(items.virt_vote_cnt) + int(items.real_vote_cnt)
                related_billboard_list.append({"pictorial_id": items.pictorial_id, "real_vote_cnt": items.real_vote_cnt,
                                               "virt_vote_cnt": items.virt_vote_cnt, "total_vote_cnt": total_vote_cnt})

            logging.info("product_brand_info" % related_billboard_list)

            return related_billboard_list

        except:
            logging.error("catch exception,err_msg:%s" % traceback.format_exc())
            return []


class TopicComplaint(models.Model):
    """帖子举报"""

    class Meta:
        verbose_name = u'帖子举报'
        app_label = 'community'
        db_table = 'topic_complaint'

    id = models.IntegerField(verbose_name='日记图片ID', primary_key=True)
    user_id = models.BigIntegerField(verbose_name=u'用户ID', db_index=True)
    topic = models.ForeignKey(
        Topic, verbose_name=u"关联的帖子", null=True, blank=True, default=None, on_delete=models.CASCADE,
        related_name='complaints')

    is_online = models.BooleanField(verbose_name=u"是否有效", default=True)


class PictorialTopic(models.Model):
    """画报帖子关系"""

    class Meta:
        verbose_name = u'画报帖子关系'
        app_label = 'community'
        db_table = 'community_pictorial_topic'

    id = models.IntegerField(verbose_name=u'日记ID', primary_key=True)
    pictorial_id = models.BigIntegerField(verbose_name=u'画报ID')
    topic_id = models.BigIntegerField(verbose_name=u'帖子ID')
    is_online = models.BooleanField(verbose_name=u"是否有效", default=True)
    is_deleted = models.BooleanField(verbose_name=u'是否删除')


class TopicExtra(models.Model):
    """帖子相关额外信息"""

    class Meta:
        verbose_name = '帖子额外信息'
        app_label = 'community'
        db_table = 'topic_extra'

    id = models.IntegerField(verbose_name=u'ID', primary_key=True)
    topic_id = models.IntegerField(verbose_name=u"帖子ID", db_index=True)
    virtual_vote_num = models.IntegerField(verbose_name="帖子虚拟点赞")


class TopicHomeRecommend(models.Model):
    """运营位帖子"""

    class Meta:
        verbose_name = "运营位帖子"
        db_table = "topic_home_recommend"

    id = models.IntegerField(verbose_name=u"id", primary_key=True)
    topic_id = models.IntegerField(verbose_name=u"帖子ID")
    is_online = models.BooleanField(verbose_name=u'是否上线')
    is_deleted = models.BooleanField(verbose_name=u'是否删除')


class TopicBillBoard(models.Model):
    """榜单投票"""

    class Meta:
        verbose_name = "榜单投票"
        db_table = "topic_vote_cnt"

    topic_id = models.IntegerField(verbose_name=u"帖子ID")
    pictorial_id = models.IntegerField(verbose_name=u"榜单ID")
    real_vote_cnt = models.IntegerField(verbose_name=u"真实赞")
    virt_vote_cnt = models.IntegerField(verbose_name=u"虚拟赞")
