import json
from utils.base import APIView
from utils.logger import error_logger


class PictorialListView(APIView):
    def get(self, request):
        star_id = request.GET.get('star_id')
        user_id = request.GET.get('user_id')
        offset = int(request.GET.get('page', 1))
        limit = int(request.GET.get('limit', 10))
        filters = self.handle_filter(request.GET.get('filter', '{}'))
        if star_id:
            filters.update({'star_id': star_id})
        if user_id:
            filters.update({'user_id': user_id})
        try:
            data = self.rpc['venus/sun/pictorial/list'](filters=filters, offset=(offset-1)*limit, limit=limit).unwrap()
        except Exception as e:
            error_logger.error(u'获取画报列表失败%s' , e)
            raise
        return data

    def post(self, request):
        ids = json.loads(request.POST.get('ids', '[]'))
        updates = json.loads(request.POST.get('update', '{}'))
        try:
            self.rpc['venus/sun/pictorial/batch/update'](updates=updates, ids=ids).unwrap()
        except Exception as e:
            error_logger.error(u'批量更新失败%s', e)
            raise
        return {
            "message": "更新成功"
        }


class PictorialUpdateOrCreate(APIView):
    def get(self, request):
        id = request.GET.get('id').split(':')[0]
        try:
            data = self.rpc['venus/sun/pictorial/get'](id=id).unwrap()
        except Exception as e:
            error_logger.error(u'获取%s画报信息失败%s'%(id, e))
            raise
        return data

    def post(self, request):

        id = int(request.POST.get('id')) if request.POST.get('id') else None
        star_ids = json.loads(request.POST.get('star', '[]'))
        collection_tag_ids = json.loads(request.POST.get('collection_tag_ids', '[]'))
        pictorial_activity_id = request.POST.get('pictorial_activity_id')
        tab_config = request.POST.get("tab_config", "[]")
        data = {
            'name': request.POST.get('name', ''),
            'description': request.POST.get('description', ''),
            'creator_id': request.POST.get('creator_id', ''),
            'celebrity_ids': star_ids,
            'is_online': int(request.POST.get('is_online', 1)),
            'is_recommend': int(request.POST.get('is_recommend', 0)),
            'icon': request.POST.get('icon', ''),
            'collection_tag_ids': collection_tag_ids,
            'is_home_recommend': int(request.POST.get('is_home_recommend', 0)),
            'add_score': int(request.POST.get('add_score', 0)),
            'is_public': True if request.POST.get('is_public') == "true" else False,
            'is_finished': int(request.POST.get('is_finished', 1)),
            'is_feed': int(request.POST.get('is_feed', 0)),
            'pictorial_activity_ids': [int(pictorial_activity_id)] if pictorial_activity_id else [],
            'alias': request.POST.get('alias', ''),
            'comment_talk': request.POST.get('pictorial_comment_talk', ''),
            'tab_config': json.loads(tab_config),
        }
        try:
            data = self.rpc['venus/sun/pictorial/edit'](id=id, data=data).unwrap()
        except Exception as e:
            error_logger.error(u'编辑%s用户信息失败%s' % (id, e))
            raise
        if not data.get('id'):
            return {
                'message': '画报已存在'
            }
        return {
            'message': '更新成功',
            'data': data
        }

    
class PictorialTopics(APIView):
    """多个帖子关系"""

    def post(self, request):
        topic_ids = json.loads(request.POST.get('topic_ids', '[]'))
        pictorial_id = request.POST.get('pictorial_id')

        try:
            self.rpc['venus/sun/pictorial/add_topics'](topic_ids=topic_ids, pictorial_id=pictorial_id).unwrap()
        except Exception as e:
            error_logger.error(u'编辑%s用户信息失败%s' % (id, e))
            raise
        return {
            'message': '更新成功'
        }


class PictorialFeedlListView(APIView):
    def get(self, request):
        order_by = request.GET.get('order_by', "-id")
        offset = int(request.GET.get('offset', 1))
        limit = int(request.GET.get('limit', 10))

        try:
            data = self.rpc['venus/sun/pictorial/feed/list'](offset=(offset - 1) * limit, limit=limit,
                                                             order_by=order_by).unwrap()
        except Exception as e:
            error_logger.error(u'获取画报列表失败%s' , e)
            raise
        return data


class PictorialFeedlRank(APIView):
    """修改画报首页like流rank"""

    def post(self, request):
        id_ = request.POST.get('id')
        rank = int(request.POST.get('rank'))

        try:
            self.rpc['venus/sun/pictorial/feed/rank'](id_=id_, rank=rank).unwrap()
        except Exception as e:
            error_logger.error(u'编辑%信息失败%s' % (e))
            raise
        return {
            'message': '更新成功'
        }


class PictorialFeedDelete(APIView):
    """删除画报首页like流"""

    def post(self, request):
        ids = json.loads(request.POST.get('ids', '[]'))

        try:
            self.rpc['venus/sun/pictorial/feed/delete'](ids=ids).unwrap()
        except Exception as e:
            error_logger.error(u'删除信息失败%s' % (e))
            raise
        return {
            'message': '删除成功'
        }


class PictorialTopicList(APIView):
    """获取榜单帖子列表"""

    def get(self, request):
        id_ = request.GET.get('id')
        offset = int(request.GET.get('page', 1))
        limit = int(request.GET.get('count', 10))

        try:
            data = self.rpc['venus/sun/pictorial/topic/list'](id_=id_, offset=(offset-1)*limit, limit=limit).unwrap()
        except Exception as e:
            error_logger.error(u'获取信息失败%s' % (e))
            raise
        return data


class PictorialTopicSetRank(APIView):
    """设置榜单中帖子的优先级"""

    def post(self, request):
        pictorial_id = request.POST.get('pictorial_id')
        topic_id = request.POST.get('topic_id')
        rank = request.POST.get('rank')
        try:
            self.rpc['venus/sun/pictorial/topic/set_rank'](pictorial_id=pictorial_id, topic_id=topic_id, rank=rank).unwrap()
        except Exception as e:
            error_logger.error(u'操作失败%s' % (e))
            raise
        return {
            'message': '操作成功'
        }


class PictorialTopicDel(APIView):
    """删除榜单中的帖子"""

    def post(self, request):
        pictorial_id = request.POST.get('pictorial_id')
        topic_id = request.POST.get('topic_id')
        try:
            self.rpc['venus/sun/pictorial/topic/del'](pictorial_id=pictorial_id, topic_id=topic_id).unwrap()
        except Exception as e:
            error_logger.error(u'操作失败%s' % (e))
            raise
        return {
            'message': '删除成功'
        }


class PictorialFollowsDel(APIView):
    """删除榜单中的关注者"""

    def post(self, request):
        pictorial_id = int(request.POST.get('pictorial_id', 0))
        user_ids = json.loads(request.POST.get('user_ids', '[]'))
        user_ids = [int(item) for item in user_ids]
        try:
            self.rpc['venus/sun/pictorial/follows/del'](pictorial_id=pictorial_id, user_ids=user_ids).unwrap()
        except Exception as e:
            error_logger.error(u'操作失败%s' % (e))
            raise
        return {
            'message': '删除成功'
        }


class PictorialFollowsAdd(APIView):
    """新增榜单中的关注者"""

    def post(self, request):
        pictorial_id = int(request.POST.get('pictorial_id', 0))
        user_ids = json.loads(request.POST.get('user_ids', '[]'))
        user_ids = [int(item) for item in user_ids]
        try:
            self.rpc['venus/sun/pictorial/follows/add'](pictorial_id=pictorial_id, user_ids=user_ids).unwrap()
        except Exception as e:
            error_logger.error(u'操作失败%s' % (e))
            raise
        return {
            'message': '添加成功'
        }
