import copy
from typing import List

from gm_types.gaia import TAG_VERSION
from gm_types.mimas import TRACTATE_COVER_TYPE
from gm_upload.utils.image_utils import Picture

from card.settings import settings
from card.views.schemas import Tag
from card.views.utils.html import gm_decode_html
from card.views.utils.tag import feed_tag_v3_show_filter
from card.libs.protocol import gm_protocol
from card.views.utils.format import BaseFormat
from .schemas import Media, TractateList, TractateInfo


class TractateFormat(BaseFormat):

    @classmethod
    def convert_tags(cls, tags, tag_version):

        ts: List[Tag] = []
        for tag in tags:
            ts.append(Tag(**{
                "id": tag.get("id"),
                "name": tag.get("name"),
                "type": tag.get("tag_type"),
                "gm_url": gm_protocol.get_polymer_detail(tag_id=tag.get("id"), new_tag=tag_version),
            }))

        return ts

    @classmethod
    def get_media_info(cls, video_list, images_list, content_videos, content_images):
        """
        转换用户帖卡片媒体信息
        :param video_list:
        :param images_list:
        :param kwargs:
        :return:
        """

        def package_media_info(media_info):

            _image = cls.revert_image(media_info.get("image_url", ''))

            return {
                "image_url": _image,
                "video_url": media_info.get("video_url", ''),
                "width": media_info.get("width", 0),
                "height": media_info.get("height", 0),
                "video_cover_url": _image,
                "short_video_url": media_info.get("short_video_url", ""),
            }

        video_list = video_list + content_videos
        images_list = images_list + content_images

        video = {}
        if video_list:
            video_dic = video_list[0]
            video_dic["image_url"] = video_dic.pop("video_cover_url", "")
            video = package_media_info(video_dic)

        images = []
        if images_list:
            # 取前9张图片
            for image_info in images_list[: 9]:
                images.append(package_media_info(image_info))

        return video, images

    @classmethod
    def get_video_and_image(cls, info):
        video, images = cls.get_media_info(
            video_list=info.pop("videos", []),
            images_list=info.pop("images", []),
            content_videos=info.pop("content_videos", []),
            content_images=info.pop("content_images", []),
        )

        image = copy.deepcopy(video if video else images and images[0] or {})
        # 封面图逻辑：
        # 有视频但是封面是图片，将video下的video_cover_url, short_video_url 替换为封面图
        # 有视频且封面是短视频，逻辑不变
        # 没有视频，封面是图片，替换图片第一张的image_url为封面图
        # 客户端卡片展示取值逻辑：ios有视频取video 没有视频取images; Android都取images
        cover_url = info.get('cover_url', '')
        cover_type = info.get('cover_type', TRACTATE_COVER_TYPE.NO_COVER)

        if all([video, cover_type == TRACTATE_COVER_TYPE.IMAGE]):
            cover_url = Picture.get_aspectscale_path(cover_url)
            video['video_cover_url'] = cover_url
            video['short_video_url'] = cover_url
            image['video_cover_url'] = cover_url
            image['short_video_url'] = cover_url

        elif all([not video, cover_type == TRACTATE_COVER_TYPE.IMAGE]):
            image['image_url'] = Picture.get_aspectscale_path(cover_url)

        return video, image

    def format_tractate(self, info, tag_v3_display):
        """处理用户帖的基础数据"""

        tractate_data = {
            "id": info["tractate_id"],
            "title": info["title"],
            "gm_url": gm_protocol.get_tractate_detail(info["tractate_id"]),
            "content": gm_decode_html(info["title"] or info["content"]),
            "is_voted": info.get("is_voted", False),
            "vote_num": info.get("is_voted", 0),
            "reply_amount": info.get("reply_amount", 0),
            "user": info["user"],
        }

        tractate = TractateInfo(**tractate_data)

        cover_url = info.get('cover_url', '')
        cover_type = info.get('cover_type', TRACTATE_COVER_TYPE.NO_COVER)
        if cover_type and cover_url:
            tractate.set_cover = True

        video, images = self.get_video_and_image(info)
        if video:
            tractate.video = self.get_video(
                video_cover=video.get("video_cover_url", ''),
                video_url=video.get("video_url", ''),
                short_video_url=video.get("short_video_url", ''),
                width=video.get("width", '0'),
                height=video.get("height", '0'),
            )
        else:
            tractate.video = {}

        if images:
            tractate.images = Media(**images)

        tags_v2 = self.convert_tags(info.pop("new_tags", []), tag_version=TAG_VERSION.V2)
        tags_v3 = feed_tag_v3_show_filter(info.pop("tags_v3", []))
        tags_v3 = self.convert_tags(tags_v3, tag_version=TAG_VERSION.V3)

        if tag_v3_display and tags_v3:
            _tags = tags_v3
        else:
            _tags = tags_v2

        tractate.tags = _tags[:settings.TRACTATE_TAG_DISPLAY_NUM_CONTROL]

        return tractate

    def format_feed_card(self, tractate_list) -> TractateList:

        res: TractateList = []
        for item in tractate_list:
            res.append(self.format_tractate(info=item, tag_v3_display=True))

        return TractateList(tractates=res)
