# -*- coding: utf-8 -*-
import os
from typing import Dict

from pydantic import BaseModel

LOG_DIR = "./data/log/card/app/"


class LoggingSettings(BaseModel):
    version: int = 1
    disable_existing_loggers: bool = False
    formatters: Dict = {
        "verbose": {
            "format": "%(asctime)s %(levelname)s %(module)s.%(funcName)s Line:%(lineno)d  %(message)s"
        },
        "simple": {"format": "%(levelname)s %(message)s"},
        "profile": {"format": "%(asctime)s %(message)s"},
        "raw": {"format": "%(message)s"},
    }
    handlers: Dict = {
        "null": {"level": "DEBUG", "class": "logging.NullHandler", },
        "console": {
            "level": "DEBUG",
            "class": "logging.StreamHandler",
            "formatter": "verbose",
        },
        "info_handler": {
            "level": "INFO",
            "class": "logging.handlers.RotatingFileHandler",
            "filename": os.path.join(LOG_DIR, "info.log"),
            "formatter": "verbose",
        },
        "error_handler": {
            "level": "ERROR",
            "class": "logging.handlers.RotatingFileHandler",
            "filename": os.path.join(LOG_DIR, "error_logger.log"),
            "formatter": "verbose",
        },
        "profile_handler": {
            "level": "INFO",
            "class": "logging.handlers.RotatingFileHandler",
            "filename": os.path.join(LOG_DIR, "profile.log"),
            "formatter": "profile",
        },
        "exception_handler": {
            "level": "INFO",
            "class": "logging.handlers.RotatingFileHandler",
            "filename": os.path.join(LOG_DIR, "exception.log"),
            "formatter": "verbose",
        },
        "tracer_handler": {
            "level": "INFO",
            "class": "logging.handlers.RotatingFileHandler",
            "filename": os.path.join(LOG_DIR, "tracer.log"),
            "formatter": "raw",
        },
        "sql_handler": {
            "level": "INFO",
            "class": "logging.handlers.RotatingFileHandler",
            "filename": os.path.join(LOG_DIR, "sql.log"),
            "formatter": "verbose",
        },
    }

    loggers: Dict = {
        "fastapi.request": {
            "handlers": ["error_handler"],
            "level": "ERROR",
            "propagate": False,
        },
        "info_logger": {
            "handlers": ["info_handler", "console"],
            "level": "DEBUG",
            "propagate": False,
        },
        "error_logger": {
            "handlers": ["error_handler"],
            "level": "ERROR",
            "propagate": False,
        },
        "exception_logger": {
            "handlers": ["exception_handler"],
            "level": "ERROR",
            "propagate": False,
        },
        "sql_logger": {
            "handlers": ["sql_handler", "console"],
            "level": "DEBUG",
            "propagate": False,
        },
    }


LOGGING = LoggingSettings().dict()
