import socket
import http

import requests
from requests.exceptions import ConnectionError

class HTTPMethods(object):

    GET = "get"
    POST = "post"


class ZIPKIN_VERION(object):

    V1 = "v1"
    V2 = "v1"


class ReuqestError(Exception):
    pass


class Zipkin(object):
    """A client for zipkin's V2 API.

    https://zipkin.apache.org/zipkin-api/
    """

    def __init__(self, domain, version, *args, **kwargs):
        self.domain = domain
        self.version = version

    def _request(self, method, path, params=None, json=None, **kwargs):
        """send a request."""

        url = "".join([self.domain, path])

        if method == HTTPMethods.GET:
            try:
                res = requests.get(url=url, params=params, **kwargs)
            except (socket.gaierror, ConnectionError):
                raise ReuqestError("网络异常，无法连接")

        elif method == HTTPMethods.POST:
            try:
                res = requests.post(url=url, data=params, json=json, **kwargs)
            except (socket.gaierror, ConnectionError):
                raise ReuqestError("网络异常，无法连接")

        if res.status_code == http.HTTPStatus.OK:
            try:
                return res.json()
            except:
                return res.text

        raise ReuqestError("request error: {}".format(res.text))

    def services(self):
        """List services name."""

        path = "/zipkin/api/{version}/services".format(version=self.version)

        response = self._request(method=HTTPMethods.GET, path=path)

        return response

    def spans(self, service_name):
        """List service's spans."""

        path = "/zipkin/api/{version}/spans".format(version=self.version)

        spans = self._request(method=HTTPMethods.GET, path=path, params={"serviceName": service_name})

        return spans

    def traces(self, service_name, error=False, limit=100, lookback=30 * 1000):
        """List service's traces.

        params: error: boolean. 如果位 True 则仅仅显示错误的trace.
        params: lookback: 当前时间多久之前。如果是自定义时间模式，为custom，需要参数：startTs、endTs.
        """

        path = "/zipkin/api/{version}/traces".format(version=self.version)

        params = {
            "serviceName": service_name,
            "lookback": lookback,
            "sortOrder": "timestamp-desc",
            "limit": limit,
        }
        if error:
            params["annotationQuery"] = {"error": True}

        traces = self._request(method=HTTPMethods.GET, path=path, params=params)

        return traces
