# -*- coding: utf-8 -*-
import json
from typing import Optional

from django.conf import settings
import requests

from adapter.log import exception_logger
from hx.exceptions import HuanXinException
from hx.errors import HuanXinError
from common.caches import huanxin_cache


def handle_huanxin_response_error(response: requests.models.Response, client=None):
    # http://docs-im.easemob.com/im/other/errorcode/restapi#%E7%B4%A2%E5%BC%95_%E9%94%99%E8%AF%AF%E7%8A%B6%E6%80%81%E7%A0%81

    result = None
    try:
        result = json.loads(response.content.decode('utf-8', 'ignore'),
                            strict=False)
    except Exception as e:
        exception_logger.exception(str(e))
        raise HuanXinException(
            error=HuanXinError.HUAN_XIN_SERVER_ERROR.value,
            error_description=HuanXinError.HUAN_XIN_SERVER_ERROR.value,
            client=client,
            request=reqe.request,
            response=reqe.response
        )

    if 400 <= response.status_code <= 599:
        raise HuanXinException(
            error=result.get('error'),
            error_description=result.get('error_description'),
            client=client,
            request=reqe.request,
            response=reqe.response
        )
    return result


class HuanXinClient(object):
    BASE_URL = getattr(settings, 'HX_BASE_URL', None)
    ORG_NAME = getattr(settings, 'HX_ORG_NAME', None)
    APP_NAME = getattr(settings, 'HX_APP_NAME', None)
    CLIENT_ID = getattr(settings, 'HX_CLIENT_ID', None)
    CLIENT_SECRECT = getattr(settings, 'HX_CLIENT_SECRECT', None)

    TOKEN_CACHE_KEY = 'HX_TOEKN'

    def __init__(self):
        assert self.BASE_URL and self.ORG_NAME and \
        self.APP_NAME and self.CLIENT_ID and self.CLIENT_SECRECT, \
        'huanxin config error'
        self.session = requests.Session()

    def _get_auth_token(self):
        GET_TOKEN_URL = '{base_url}/{org_name}/{app_name}/token'.format(
            base_url=self.BASE_URL,
            org_name=self.ORG_NAME,
            app_name=self.APP_NAME)

        data = {
            'grant_type': 'client_credentials',
            'client_id': self.CLIENT_ID,
            'client_secret': self.CLIENT_SECRECT
        }

        response = self.session.post(GET_TOKEN_URL, json=data)
        return handle_huanxin_response_error(response)

    def check_then_set_token(self):
        cached_hx_token = huanxin_cache.get(self.TOKEN_CACHE_KEY)
        if not cached_hx_token:
            token_dict = self._get_auth_token()
            expires_in = token_res.get('expires_in', 5184000) - 5*60
            huanxin_cache.set(self.TOKEN_CACHE_KEY, json.dumps(token_res), expires_in)
        else:
            token_dict = json.loads(cached_hx_token)
        self.session.headers['Authorization'] = 'Bearer {}'.format(token_dict.get('access_token'))

    def register_single_user(self, hx_username:str, hx_password:str, nickname: Optional[str]=None):
        REGISTER_SINGLE_USER_URL = '{base_url}/{org_name}/{app_name}/users'.format(
            base_url=self.BASE_URL,
            org_name=self.ORG_NAME,
            app_name=self.APP_NAME)
        data = dict(
            username=hx_username,
            password=hx_password,
            nickname=nickname
        )
        response = self.session.post(REGISTER_SINGLE_USER_URL, json=data)
        return handle_huanxin_response_error(response)