# -*- coding: utf-8 -*-
from typing import List, Dict, Optional

from six import string_types
import dateutil
from django.db.models import Q
from gm_types.gaia import MESSAGE_ORDER_TYPE
from gm_types.msg import CONVERSATION_TYPE, CONVERSATION_ORDER

from rpc_framework.exceptions import RPCPermanentException
from api.models.message import ConversationUserStatus
from rpc import gaia_client
from rpc.gaia_client import get_single_doctor_basic_info,\
    get_user_info_by_user_id, \
    get_fans_relationship, is_kefu_user
from rpc_framework import exceptions
from rpc_framework.context import RPCViewContext
from rpc_framework.decorators import rpc_view, interceptor_classes
from rpc_framework.interceptors import CalleeParametersInterceptor, \
    SessionUserInterceptor
from search.utils import search_conversation_from_es
from services.unread.stat import UserUnread


@rpc_view('message/conversation/list_v2')
# @interceptor_classes([CalleeParametersInterceptor])
def batch_get_conversations_v2(context: RPCViewContext,
                               user_ids: List[int],
                               offset: int,
                               size: int,
                               order_by: MESSAGE_ORDER_TYPE=MESSAGE_ORDER_TYPE.READ_STATUS,
                               read_status: Optional[bool]=None,
                               is_star: Optional[bool]=None,
                               conversation_type: Optional[CONVERSATION_TYPE]=None,
                               search_content: Optional[str]=None,
                               with_fields: Optional[List[str]]=None)\
        -> Dict:
    """获取对话列表的ID， 注意和之前的v1版本不同的是， 这里只会返回conversation_ids
        权限放在上层处理

    :param user_ids: List[int] 用户ID列表
    :param offset: int 偏移量
    :param size: int 每页数量
    :param order_by: MESSAGE_ORDER_TYPE 排序规则 详细看枚举值
    :param read_status: Optional[bool] 读取状态
    :param is_star: Optional[bool] 是否标星
    :param conversation_type: CONVERSATION_TYPE 会话类型
    :param search_content str: 搜索关键词  主要针对 user_name 和 message_content进行过滤
    :return: Dict
    """
    if not isinstance(user_ids, list):
        raise RPCPermanentException
    if not isinstance(size, int):
        raise RPCPermanentException
    if size <= 0 or size >= 50:
        size = 50
    if with_fields is None or not isinstance(with_fields, list):
        with_fields = []

    # 如果没有查询关键字 走mysql过滤， 如果有查询关键字 则从es获取对应的conversation_ids
    # ES filters
    es_filters = {'user_ids': user_ids}
    # Mysql filters
    query_q = Q(user_id__in=user_ids)
    ordering_by = list()

    if read_status is not None and isinstance(read_status, bool):
        query_q &= Q(read_status=read_status)
        es_filters['multi_filter'] = {
            'is_unread': read_status,
            'user_ids': user_ids,
        }

    if is_star is not None and isinstance(is_star, bool):
        query_q &= Q(is_star=is_star)
        es_filters['is_star'] = is_star

    # 这块暂时不支持es的过滤,
    # conversation_type 暂时没有什么特别的需求
    if conversation_type is not None and isinstance(conversation_type, list):
        conversation_type = filter(lambda cy: cy in CONVERSATION_TYPE, conversation_type)
        query_q &= Q(conversation_type__in=conversation_type)

    es_sort_type = CONVERSATION_ORDER.UNREAD
    if order_by == MESSAGE_ORDER_TYPE.LAST_REPLY_TIME:
        ordering_by.append('-last_reply_time')
        es_sort_type = CONVERSATION_ORDER.LAST_REPLY_TIME
    elif order_by == MESSAGE_ORDER_TYPE.READ_STATUS:
        ordering_by.extend(['-read_status', '-last_reply_time'])

    if search_content is not None:
        es_query = {
            'content': search_content,
            'user_last_name': search_content,
        }
        assert isinstance(search_content, string_types)
        es_result = search_conversation_from_es(offset=offset, size=size, filters=es_filters,
                                                      query=es_query, sort_type=es_sort_type)
        total_count = es_result['total_count']
        _conversation_ids = es_result['conversation_ids']
        conversation_user_status = ConversationUserStatus.objects.filter(
            user_id__in=user_ids, conversation_id__in=_conversation_ids
        ).all()
        # 重排序
        conversation_id_to_conversation_user_status = {cus.conversation_id: cus for cus in conversation_user_status}
        conversation_user_status = []
        for conversation_id in _conversation_ids:
            if conversation_id_to_conversation_user_status.get(conversation_id):
                conversation_user_status.append(conversation_id_to_conversation_user_status[conversation_id])
    else:
        total_count = ConversationUserStatus.objects.filter(query_q).count()
        conversation_user_status = ConversationUserStatus.objects.prefetch_related('conversation').\
                                  filter(query_q).order_by(*ordering_by)[offset:offset + size]
    # conversation_info in database
    conversation_info_list = [cs.conversation.conversation_info_v2(cs) for cs in conversation_user_status]
    conversation_id_to_user_id = {cs.conversation_id: cs.user_id for cs in conversation_user_status}
    cs_ids = [cs.id for cs in conversation_user_status]
    conversation_ids = [cs.conversation_id for cs in conversation_user_status]

    # 此接口目前只用于客户端私信，返回unread_num
    for c in conversation_info_list:
        default = 1 if c['is_new'] else 0
        c['unread_num'] = UserUnread(conversation_id_to_user_id[c['id']]).get_conversation_unread(c['id'], default=default)

    if 'mark' in with_fields:
        conversation_id_and_target_user_id = ConversationUserStatus.objects.filter(
            conversation_id__in=conversation_ids
        ).exclude(id__in=cs_ids).values('conversation_id', 'user_id')
        conversation_id_to_target_user_id = {item['conversation_id']: item['user_id']
                                             for item in conversation_id_and_target_user_id}
        mark_user_dict_list = gaia_client.get_message_mark_user(user_id_list=user_ids, target_user_id_list=list(conversation_id_to_target_user_id.values()))

        _user_ids_to_comment = {(entry.get('user_id'), entry.get('target_user_id')): entry.get('comment') for entry in mark_user_dict_list}
        _conversation_id_to_user_ids = {conversation_id: (user_id, conversation_id_to_target_user_id[conversation_id])
                                        for conversation_id, user_id in conversation_id_to_user_id.items()
                                        if conversation_id_to_target_user_id.get(conversation_id)}
        for c in conversation_info_list:
            c['comment'] = _user_ids_to_comment.get(_conversation_id_to_user_ids[c['id']], u'')

    # filter out items with empty user_ids
    conversation_info_list = [c for c in conversation_info_list if len(c['user_ids']) > 1]

    return {
        'conversation_list': conversation_info_list,
        'total_count': total_count,
    }


@rpc_view('message/conversation/list_v3')
# @interceptor_classes([CalleeParametersInterceptor])
def message_conversation_list_v3(context: RPCViewContext,
                                 user_ids: List[int],
                                 offset: int,
                                 size: int,
                                 order_by: CONVERSATION_ORDER=CONVERSATION_ORDER.LAST_REPLY_TIME,
                                 last_reply_time_start: Optional[str]=None,
                                 last_reply_time_end: Optional[str]=None,
                                 reply_status: Optional[bool]=None,
                                 is_star: Optional[bool]=None,
                                 user_id: Optional[bool]=None,
                                 user_last_name: Optional[bool]=None,
                                 comment: Optional[bool]=None) -> Dict:
    """
    获取会话列表， 与v2不同的是全部走es获取conversation_ids
    :param user_ids: LIST[USER_ID]
    :param offset: INT 偏移量
    :param size: INT 每页数量
    :param order_by: 排序规则
    :param last_reply_time_start: str 最后回复开始时间筛选
    :param last_reply_time_end: str 最后回复结束时间筛选
    :param reply_status: bool 回复状态（新消息、旧消息）
    :param is_star: bool
    :param user_id: int 精确搜索用户ID
    :param user_last_name: int 搜索用户昵称
    :param comment: int 搜索备注
    :return:
    """
    if not isinstance(user_ids, list):
        raise RPCPermanentException
    if not isinstance(size, int):
        raise RPCPermanentException
    if size <= 0 or size >= 50:
        size = 50
    es_filters = {'user_ids': user_ids}
    if is_star is not None and isinstance(is_star, bool):
        es_filters['is_star'] = is_star
    if last_reply_time_start is not None:
        es_filters['last_reply_time_start_gte'] = dateutil.parser.parse('{}+0800'.format(last_reply_time_start)).isoformat()
    if last_reply_time_end is not None:
        es_filters['last_reply_time_end_lte'] = dateutil.parser.parse('{}+0800'.format(last_reply_time_end)).isoformat()
    es_query = dict()
    if user_id is not None:
        es_query['user_id'] = user_id
    if user_last_name:
        es_query['user_last_name'] = user_last_name
    if comment:
        es_query['comment'] = comment
    es_sort_type = order_by
    # 全部
    es_result_total = search_conversation_from_es(offset=offset, size=size, filters=es_filters,
                                              query=es_query, sort_type=es_sort_type)
    # 已回复
    es_filters['multi_filter_status'] = {
        'status': True,
        'user_ids': user_ids,
    }
    es_result_reply = search_conversation_from_es(offset=offset, size=size, filters=es_filters,
                                 query=es_query, sort_type=es_sort_type)
    # 未回复
    es_filters['multi_filter_status'] = {
        'status': False,
        'user_ids': user_ids,
    }
    es_result_not_reply = search_conversation_from_es(offset=offset, size=size, filters=es_filters,
                                      query=es_query, sort_type=es_sort_type)
    if reply_status == None:
        es_result = es_result_total
    else:
        if reply_status:
            es_result = es_result_reply
        else:
            es_result = es_result_not_reply
    _conversation_ids = es_result['conversation_ids']
    cus_ids = []
    for conversation_id in _conversation_ids:
        sub_ids = ConversationUserStatus.objects.filter(
            user_id__in=user_ids, conversation_id=conversation_id
        ).values_list('id', flat=True)
        cus_ids.extend(list(sub_ids))
    conversation_user_status = ConversationUserStatus.objects.filter(id__in=cus_ids).order_by('-last_reply_time')

    conversation_info_list = [cs.conversation.conversation_info_v3(cs, user_ids) for cs in conversation_user_status]
    conversation_id_to_user_id = {cs.conversation_id: cs.user_id for cs in conversation_user_status}
    for c in conversation_info_list:
        default = 1 if c['is_new'] else 0
        c['unread_num'] = UserUnread(conversation_id_to_user_id[c['id']]
                                     ).get_conversation_unread(c['id'], default=default)
    return {
        'conversation_list': conversation_info_list,
        'total_count': es_result_total['total_count'],
        'reply_total': es_result_reply['total_count'],
        'not_reply_total': es_result_not_reply['total_count']
    }


@rpc_view('message/conversation/can_send')
@interceptor_classes([CalleeParametersInterceptor, SessionUserInterceptor])
def check_can_send_message(context: RPCViewContext, target_uid: str) -> Dict:
    user = context.user
    doctor_info = get_single_doctor_basic_info(user.id)

    target_user_info = get_user_info_by_user_id(target_uid)
    if not target_user_info:
        raise exceptions.NotFoundException

    target_doctor_info = get_single_doctor_basic_info(target_uid)
    if doctor_info or target_doctor_info:
        return {'can_send': True}

    user_follow, target_follow = get_fans_relationship(user.id, target_uid)
    target_user_id = (doctor_info or {}).get('user_id')
    if user_follow and target_follow:
        return {'can_send': True}
    if is_kefu_user(target_user_id):
        return {'can_send': True}

    return {'can_send': False}