import contextlib
import threading

from django.conf import settings
from helios.rpc import create_default_invoker

_base_invoker = create_default_invoker(
    debug=settings.DEBUG
).with_config(
    dump_curl=True
)

class ContextManager(object):
    _active_context_local = threading.local()

    @classmethod
    @contextlib.contextmanager
    def with_active_context(cls, context):
        """
        :type context: Context
        """
        acl = cls._active_context_local
        previous = getattr(acl, 'context', None)
        acl.context = context
        try:
            yield
        finally:
            acl.context = previous

    @classmethod
    def get_active_context(cls):
        """
        :rtype: Context | None
        """
        return getattr(cls._active_context_local, 'context', None)


def get_rpc_remote_invoker():
    context = ContextManager.get_active_context()
    if context:
        return _base_invoker.with_config(
            session_key=context._context.session_id,
            client_info=context.request.client_info if context.request else None,
        )
    return _base_invoker