# -*- coding: utf-8 -*-
import threading
from typing import Type

from .config import config
from .client import ESClientManager, ESHostBackupClientManager


_esop_instance_lock = threading.Lock()
_esop_instance = None # singleton
_esop_migrate_instance = None


def create_esop_for_database(database_name) -> ESClientManager:
    database_config = config.get_database_config(database_name)
    return ESClientManager(database_config=database_config)


def _create_esop() -> Type[ESClientManager]:
    es_client_manager_list = [
        create_esop_for_database(database_name)
        for database_name in config.order
    ]
    if len(es_client_manager_list) == 1:
        return es_client_manager_list[0]
    elif len(es_client_manager_list) == 2:
        default, secondary = es_client_manager_list
        return ESHostBackupClientManager(default=default, secondary=secondary)
    else:
        raise Exception('impossible')


def get_esop() -> Type[ESClientManager]:
    '''be sure thread-safed singleton'''
    global _esop_instance
    if _esop_instance is None:
        with _esop_instance_lock:
            if _esop_instance is None:
                _esop_instance = _create_esop()
    return _esop_instance
