# -*- coding: utf-8 -*-
from abc import ABCMeta
from typing import  Optional, Union, Dict, List

from gm_types.error import ERROR
from pydantic import ValidationError, create_model
from pydantic.error_wrappers import ErrorWrapper

from rpc_framework.errors import Error


def _check_methods(C, *methods):
    mro = C.__mro__
    for method in methods:
        for B in mro:
            if method in B.__dict__:
                if B.__dict__[method] is None:
                    return NotImplemented
                break
        else:
            return NotImplemented
    return True


class RPCViewBaseException(Exception, metaclass=ABCMeta):
    code: int
    message: str
    data: Optional[Union[Dict, List]]

    __slots__ = ()

    @classmethod
    def __subclasshook__(cls, C):
        if cls is RPCViewBaseException:
            return _check_methods(C, "as_rpc_view_exception")
        return NotImplemented


RequestErrorModel = create_model("Request")

class CalleeTypeHintValidationError(RPCViewBaseException):
    code = Error.PARAMS_INVALID
    message = ''

    @classmethod
    def from_error_wrapper_list(cls, errors:List[ErrorWrapper]):
        validation_error = ValidationError(errors, RequestErrorModel)
        self = cls()
        self.message = validation_error.json()
        return self


class AuthenticationFailed(RPCViewBaseException):
    code = Error.AUTH_FAILED
    message = Error.getDesc(Error.AUTH_FAILED)


# all above are copy from gaia
class NotFoundException(RPCViewBaseException):
    code = 1404
    message = "Not Found"


class RPCPermanentException(RPCViewBaseException):
    code = 2
    message = "Permanent Error"


class RPCTemporaryError(RPCViewBaseException):
    code = 3
    message = "Temporary Error"


class RPCValidationError(RPCViewBaseException):
    code = 17
    message = "Params/Result Validation Error"


class RPCLoginRequiredException(RPCViewBaseException):
    code = 401
    message = "Login Required"


class RPCPermissionDeniedException(RPCViewBaseException):
    code = 1001
    message = "Permission Denied"


class RPCTagRelationCycleException(RPCPermanentException):
    code = ERROR.TAG_RELATION_CYCLE
    message = ERROR.getDesc(code)


class RPCStaffRequiredException(RPCViewBaseException):
    code = 1002
    message = "Staff Required"


class RPCNotFoundException(RPCViewBaseException):
    code = 1404
    message = "Not Found"


class RPCIntegrityError(RPCViewBaseException):
    code = 1601
    message = "Integrity Error"