# -*- coding: utf-8 -*-
import traceback

from django.db import connection, transaction

from rpc_framework import exceptions
from rpc_framework.context import RPCViewContext
from rpc_framework.generic.base import RPCAbstractView
from rpc_framework.settings import api_settings


def set_rollback():
    atomic_requests = connection.settings_dict.get('ATOMIC_REQUESTS', False)
    if atomic_requests and connection.in_atomic_block:
        transaction.set_rollback(True)


def exception_handler(exc, wrapped_context) -> None:
    set_rollback()
    if isinstance(exc, exceptions.RPCViewBaseException):
        raise exc
    elif issubclass(exc.__class__, exceptions.RPCViewBaseException):
        raise exc.as_rpc_view_exception()
    elif isinstance(exc, Exception):
        print(exc, exc.args[0] if exc.args else '', traceback.format_exc())


class RPCView(RPCAbstractView):
    # The following policies may be set at either globally, or per-view.
    # renderer_classes = api_settings.DEFAULT_RENDERER_CLASSES

    # parser_classes = api_settings.DEFAULT_PARSER_CLASSES
    # authentication_classes = api_settings.DEFAULT_AUTHENTICATION_CLASSES
    # throttle_classes = api_settings.DEFAULT_THROTTLE_CLASSES
    # permission_classes = api_settings.DEFAULT_PERMISSION_CLASSES

    parser_classes = []
    authentication_classes = []
    interceptor_classes = []
    throttle_classes = []
    permission_classes = []
    # content_negotiation_class = api_settings.DEFAULT_CONTENT_NEGOTIATION_CLASS
    content_negotiation_class = lambda self: None


    # content_negotiation_class = api_settings.DEFAULT_CONTENT_NEGOTIATION_CLASS
    # metadata_class = api_settings.DEFAULT_METADATA_CLASS
    # versioning_class = api_settings.DEFAULT_VERSIONING_CLASS

    # Allow dependency injection of other settings to make testing easier.
    settings = api_settings

    # schema = DefaultSchema()

    @classmethod
    def rpc_bind(cls, **initkwargs):
        view = super().rpc_bind(**initkwargs)
        view.cls = cls
        view.initkwargs = initkwargs
        return view

    # Note: Views are made CSRF exempt from within `as_view` as to prevent
    # accidental removal of this exemption in cases where `dispatch` needs to
    # be overridden.
    def dispatch(self, context, *args, **kwargs):
        """
        `.dispatch()` is pretty much the same as Django's regular dispatch,
        but with extra hooks for startup, finalize, and exception handling.
        """
        self.args = args
        self.kwargs = kwargs
        context = self.initialize_context(context, *args, **kwargs)
        self.context = context

        try:
            self.initial(context, *args, **kwargs)

            # Get the appropriate handler method
            handler = self.handler
            result = handler(context, *args, **kwargs)

        except Exception as exc:
            result = self.handle_exception(exc)

        self.result = self.finalize_response(context, result, *args, **kwargs)
        return self.result

    # def permission_denied(self, request, message=None):
    #     """
    #     If request is not permitted, determine what kind of exception to raise.
    #     """
    #     if request.authenticators and not request.successful_authenticator:
    #         raise exceptions.NotAuthenticated()
    #     raise exceptions.PermissionDenied(detail=message)
    #
    # def throttled(self, request, wait):
    #     """
    #     If request is throttled, determine what kind of exception to raise.
    #     """
    #     raise exceptions.Throttled(wait)
    #
    # def get_authenticate_header(self, request):
    #     """
    #     If a request is unauthenticated, determine the WWW-Authenticate
    #     header to use for 401 responses, if any.
    #     """
    #     authenticators = self.get_authenticators()
    #     if authenticators:
    #         return authenticators[0].authenticate_header(request)
    #
    # def get_parser_context(self, http_request):
    #     """
    #     Returns a dict that is passed through to Parser.parse(),
    #     as the `parser_context` keyword argument.
    #     """
    #     # Note: Additionally `request` and `encoding` will also be added
    #     #       to the context by the Request object.
    #     return {
    #         'view': self,
    #         'args': getattr(self, 'args', ()),
    #         'kwargs': getattr(self, 'kwargs', {})
    #     }
    #
    # def get_renderer_context(self):
    #     """
    #     Returns a dict that is passed through to Renderer.render(),
    #     as the `renderer_context` keyword argument.
    #     """
    #     # Note: Additionally 'response' will also be added to the context,
    #     #       by the Response object.
    #     return {
    #         'view': self,
    #         'args': getattr(self, 'args', ()),
    #         'kwargs': getattr(self, 'kwargs', {}),
    #         'request': getattr(self, 'request', None)
    #     }

    def get_exception_handler_context_wrapper(self):
        """
        Returns a dict that is passed through to EXCEPTION_HANDLER,
        as the `context` argument.
        """
        return {
            'view': self,
            'args': getattr(self, 'args', ()),
            'kwargs': getattr(self, 'kwargs', {}),
            'context': getattr(self, 'context', None)
        }

    # def get_view_name(self):
    #     """
    #     Return the view name, as used in OPTIONS responses and in the
    #     browsable API.
    #     """
    #     func = self.settings.VIEW_NAME_FUNCTION
    #     return func(self)
    #
    # def get_view_description(self, html=False):
    #     """
    #     Return some descriptive text for the view, as used in OPTIONS responses
    #     and in the browsable API.
    #     """
    #     func = self.settings.VIEW_DESCRIPTION_FUNCTION
    #     return func(self, html)
    #
    # # API policy instantiation methods
    #
    # def get_format_suffix(self, **kwargs):
    #     """get_permissions
    #     Determine if the request includes a '.json' style format suffix
    #     """
    #     if self.settings.FORMAT_SUFFIX_KWARG:
    #         return kwargs.get(self.settings.FORMAT_SUFFIX_KWARG)
    #
    # def get_renderers(self):
    #     """
    #     Instantiates and returns the list of renderers that this view can use.
    #     """
    #     return [renderer() for renderer in self.renderer_classes]
    #
    def get_authenticators(self):
        """
        Instantiates and returns the list of authenticators that this view can use.
        """
        return [auth() for auth in self.authentication_classes]
    
    def get_interceptors(self):
        return [interceptor() for interceptor in self.interceptor_classes]

    def get_permissions(self):
        """
        Instantiates and returns the list of permissions that this view requires.
        """
        return [permission() for permission in self.permission_classes]

    def get_throttles(self):
        """
        Instantiates and returns the list of throttles that this view uses.
        """
        return [throttle() for throttle in self.throttle_classes]

    def get_content_negotiator(self):
        """
        Instantiate and return the content negotiation class to use.
        """
        if not getattr(self, '_negotiator', None):
            self._negotiator = self.content_negotiation_class()
        return self._negotiator

    def get_exception_handler(self):
        """
        Returns the exception handler that this view uses.
        """
        return self.settings.EXCEPTION_HANDLER

    # API policy implementation methods

    # def perform_content_negotiation(self, request, force=False):
    #     """
    #     Determine which renderer and media type to use render the response.
    #     """
    #     renderers = self.get_renderers()
    #     conneg = self.get_content_negotiator()
    #
    #     try:
    #         return conneg.select_renderer(request, renderers, self.format_kwarg)
    #     except Exception:
    #         if force:
    #             return (renderers[0], renderers[0].media_type)
    #         raise
    #
    def perform_authentication(self, context):
        """
        Perform authentication on the incoming request.

        Note that if you override this and simply 'pass', then authentication
        will instead be performed lazily, the first time either
        `request.user` or `request.auth` is accessed.
        """
        # context.user
        return None

    def perform_interceptor(self, context):
        """
        Check if the request should be permitted.
        Raises an appropriate exception if the request is not permitted.
        """
        for interceptor in self.get_interceptors():
            interceptor.intercept(context, self)

    def check_permissions(self, context):
        """
        Check if the request should be permitted.
        Raises an appropriate exception if the request is not permitted.
        """
        for permission in self.get_permissions():
            if not permission.has_permission(context, self):
                self.permission_denied(
                    context, message=getattr(permission, 'message', None)
                )

    # def check_object_permissions(self, request, obj):
    #     """
    #     Check if the request should be permitted for a given object.
    #     Raises an appropriate exception if the request is not permitted.
    #     """
    #     for permission in self.get_permissions():
    #         if not permission.has_object_permission(request, self, obj):
    #             self.permission_denied(
    #                 request, message=getattr(permission, 'message', None)
    #             )
    #
    def check_throttles(self, context):
        """
        Check if request should be throttled.
        Raises an appropriate exception if the request is throttled.
        """
        throttle_durations = []
        for throttle in self.get_throttles():
            if not throttle.allow_request(context, self):
                throttle_durations.append(throttle.wait())

        if throttle_durations:
            # Filter out `None` values which may happen in case of config / rate
            # changes, see #1438
            durations = [
                duration for duration in throttle_durations
                if duration is not None
            ]

            duration = max(durations, default=None)
            self.throttled(context, duration)

    # def determine_version(self, request, *args, **kwargs):
    #     """
    #     If versioning is being used, then determine any API version for the
    #     incoming request. Returns a two-tuple of (version, versioning_scheme)
    #     """
    #     if self.versioning_class is None:
    #         return (None, None)
    #     scheme = self.versioning_class()
    #     return (scheme.determine_version(request, *args, **kwargs), scheme)

    # Dispatch methods

    def initialize_context(self, context, *args, **kwargs):
        """
        Returns the initial request object.
        """
        # parser_context = self.get_parser_context(context)

        return RPCViewContext(
            context,
            authenticators=self.get_authenticators(),
            interceptors=self.get_interceptors(),
            negotiator=self.get_content_negotiator()
        )

    def initial(self, context: RPCViewContext, *args, **kwargs):
        """
        Runs anything that needs to occur prior to calling the method handler.
        """
        # self.format_kwarg = self.get_format_suffix(**kwargs)

        # Perform content negotiation and store the accepted info on the request
        # neg = self.perform_content_negotiation(context)
        # request.accepted_renderer, request.accepted_media_type = neg

        # Determine the API version, if versioning is in use.
        # version, scheme = self.determine_version(request, *args, **kwargs)
        # request.version, request.versioning_scheme = version, scheme

        # Ensure that the incoming request is permitted
        self.perform_interceptor(context)
        self.perform_authentication(context)
        self.check_permissions(context)
        self.check_throttles(context)

    def finalize_response(self, context, result, *args, **kwargs):
        return result

    def handle_exception(self, exc):
        """
        Handle any exception that occurs, by returning an appropriate response,
        or re-raising the error.
        """
        exception_handler = self.get_exception_handler()

        wrapped_context = self.get_exception_handler_context_wrapper()
        result = exception_handler(exc, wrapped_context)

        # if not result:
        #     self.raise_uncaught_exception(exc)

        # result.exception = True
        return result

    # def raise_uncaught_exception(self, exc):
    #     if settings.DEBUG:
    #         request = self.request
    #         renderer_format = getattr(request.accepted_renderer, 'format')
    #         use_plaintext_traceback = renderer_format not in ('html', 'api', 'admin')
    #         request.force_plaintext_errors(use_plaintext_traceback)
    #     raise exc