# -*- coding: utf-8 -*-
import json

from django.views import View
from django.http.response import HttpResponse, JsonResponse

from gm_rpcd.internals.protocol.request import make_request_from_v1
from gm_rpcd.internals.protocol.response import response_to_v1_json_value
from gm_rpcd.version import VERSION

from . import DISPATCHER

try:
    from django.db import close_old_connections
except ImportError:
    close_old_connections = None

# The followings are Django class-based views for gm_rpcd

class PingView(View):
    def post(self, request, *args, **kwargs):
        return HttpResponse('pong')


class ServerVersionView(View):
    def get(self, request, *args, **kwargs):
        return JsonResponse({
            'name': 'gm-rpcd',
            'version': VERSION
        })


class SystemStatusesView(View):
    def get(self, request, *args, **kwargs):
        data = DISPATCHER.system_statuses(request)
        if close_old_connections:
            close_old_connections()
        return JsonResponse(data)


class V1OnceView(View):
    def post(self, request, *args, **kwargs):
        requests_dict = json.loads(request.POST.get('requests'))

        method = requests_dict.get('method')
        params = requests_dict.get('params')
        session_key = requests_dict.get('session_key')
        environment = requests_dict.get('environment')

        request = make_request_from_v1(
            method=method,
            params=params,
            session_key=session_key,
            environment=environment,
        )
        response = DISPATCHER.process_single_request(request)
        response_value = response_to_v1_json_value(response)
        if close_old_connections:
            close_old_connections()

        return HttpResponse(json.dumps(response_value), {'content_type': 'application/json'})


class V1BatchView(View):
    def post(self, request, *args, **kwargs):
        requests_json = request.POST.get('requests')
        requests_value = json.loads(requests_json)
        response_v1_list = []
        for request_v1_value in requests_value:
            request = make_request_from_v1(
                method=request_v1_value['method'],
                params=request_v1_value['params'],
                session_key=request_v1_value.get('session_key'),
                environment=request_v1_value.get('environment'),
            )
            response = DISPATCHER.process_single_request(request)
            response_v1_list.append(response_to_v1_json_value(response))

        if close_old_connections:
            close_old_connections()
        return HttpResponse(json.dumps(response_v1_list), {'content_type': 'application/json'})
