# -*- coding: utf-8 -*-
import sys
from contextlib import contextmanager
from typing import Optional, Tuple

from gm_logging import RequestInfo
from gm_rpcd.internals.context import Context
from gm_rpcd.internals.protocol.request import Request

from rpc_framework import exceptions
from rpc_framework.settings import api_settings


class WrappedAttributeError(Exception):
    pass


@contextmanager
def wrap_attributeerrors():
    """
    Used to re-raise AttributeErrors caught during authentication, preventing
    these errors from otherwise being handled by the attribute access protocol.
    """
    try:
        yield
    except AttributeError:
        info = sys.exc_info()
        exc = WrappedAttributeError(str(info[1]))
        raise exc.with_traceback(info[2])


class RPCViewContext(object):
    '''
    Wrapper allowing to enhance a `gm_rpcd.internals.context.Context` instance.

    Kwargs:
        - context(gm_rpcd.internals.context.Context). The original context instance.
        - parsers_classes(list/tuple). The parsers to use for parsing the
          request content.
        - authentication_classes(list/tuple). The authentications used to try
          authenticating the request's user.

    Usage Examples:
        inspect property of `gm_rpcd.internals.context.Context` instance:
            print(context.request.request_id)
            print(context.request.session_id)
            print(context.request.method)
            print(context.request.params)
            print(context.request.environment)
            print(context.request_info.user_id)

    '''
    def __init__(self,
                 context: Context,
                 authenticators: Optional[Tuple]=None,
                 interceptors: Optional[Tuple]=None,
                 negotiator: Optional[Tuple]=None):
        self._context = context
        self.authenticators = authenticators or ()
        self.interceptors = interceptors or ()
        # self.negotiator = negotiator or self._default_negotiator()

    @property
    def user(self):
        """
        Returns the user associated with the current request, as authenticated
        by the authentication classes provided to the request.
        """
        if not hasattr(self, '_user'):
            with wrap_attributeerrors():
                self._authenticate()
        return self._user

    @user.setter
    def user(self, value):
        """
        Sets the user on the current request. This is necessary to maintain
        compatibility with django.contrib.auth where the user property is
        set in the login and logout functions.

        Note that we also set the user on Django's underlying `HttpRequest`
        instance, ensuring that it is available to any middleware in the stack.
        """
        self._user = value
        self._request.user = value

    @property
    def request(self) -> Optional[Request]:
        return getattr(self._context, '_Context__request', None)

    @property
    def request_info(self) -> Optional[RequestInfo]:
        return getattr(self._context, '_Context__request_info', None)

    def _authenticate(self):
        """
        Attempt to authenticate the request using each authentication instance
        in turn.
        """
        for authenticator in self.authenticators:
            try:
                user_auth_tuple = authenticator.authenticate(self)
            except exceptions.RPCViewBaseException:
                self._not_authenticated()
                raise

            if user_auth_tuple is not None:
                self._authenticator = authenticator
                self.user, self.auth = user_auth_tuple
                return

        self._not_authenticated()

    def _not_authenticated(self):
        """
        Set authenticator, user & authtoken representing an unauthenticated request.

        Defaults are None, AnonymousUser & None.
        """
        self._authenticator = None

        if api_settings.UNAUTHENTICATED_USER:
            self.user = api_settings.UNAUTHENTICATED_USER()
        else:
            self.user = None

        if api_settings.UNAUTHENTICATED_TOKEN:
            self.auth = api_settings.UNAUTHENTICATED_TOKEN()
        else:
            self.auth = None

    def __getattr__(self, attr):
        """
        If an attribute does not exist on this instance, then we also attempt
        to proxy it to the underlying HttpRequest object.
        """
        try:
            return getattr(self._request, attr)
        except AttributeError:
            return self.__getattribute__(attr)


