# -*- coding: utf-8 -*-
from abc import ABCMeta, abstractmethod
from typing import Sequence

from gm_rpcd.internals.exceptions import RPCDFaultException
from pydantic import ValidationError, create_model
from pydantic.error_wrappers import ErrorList

from extension.types import Error


def _check_methods(C, *methods):
    mro = C.__mro__
    for method in methods:
        for B in mro:
            if method in B.__dict__:
                if B.__dict__[method] is None:
                    return NotImplemented
                break
        else:
            return NotImplemented
    return True


class RPCViewBaseException(metaclass=ABCMeta):
    code: int
    message: str

    __slots__ = ()

    @classmethod
    def __subclasshook__(cls, C):
        if cls is RPCViewBaseException:
            return _check_methods(C, "as_rpcd_fault")
        return NotImplemented

    @abstractmethod
    def as_rpcd_fault(self):
        return RPCDFaultException(error=self.error, message=self.message)


RequestErrorModel = create_model("Request")

class CalleeTypeHintValidationError(ValidationError):
    def __init__(self, errors: Sequence[ErrorList]) -> None:
        return super().__init__(errors, RequestErrorModel)

    def as_rpcd_fault(self):
        return RPCDFaultException(code=Error.PARAMS_INVALID, message=self.json())

