# -*- coding: utf-8 -*-
import itertools
import threading
import functools

import elasticsearch
from django.conf import settings

from .config import config, ESDatabaseConfig, ESTableSchema, ESVersion, \
    load_mapping
from .models import ESBulkAction


class ESClientManagerInterface(object):
    def search(self, table, **kwargs):
        raise NotImplementedError

    def index(self, table, body, **kwargs):
        raise NotImplementedError

    def mget(self, table, body, **kwargs):
        raise NotImplementedError

    def bulk_single(self, action):
        raise NotImplementedError

    def helper_bulk(self, action_iter, **kwargs):
        raise NotImplementedError

    def helper_scan(self, table, **kwargs):
        raise NotImplementedError

    def alter_table(self, table, drop_if_exists=False):
        raise NotImplementedError


def _create_es_client(hosts):
    if settings.DEBUG:
        es = elasticsearch.Elasticsearch(
            hosts=hosts,
            # no sniffing
            sniff_on_start=False,
            sniff_on_connection_fail=False
        )
    else:
        es = elasticsearch.Elasticsearch(
            hosts=hosts,
            # sniff before doing anything
            sniff_on_start=True,
            # refresh nodes after a node fails to respond
            sniff_on_connection_fail=True,
            # and also every 60 seconds
            sniffer_timeout=60,
            sniff_timeout=1
        )
    return es


class ESClientManager(ESClientManagerInterface):
    INDEX = 'index'
    DOC_TYPE = 'doc_type'

    def __init__(self, database_config: ESDatabaseConfig):
        # database_config = config.get_database_config(database_name)
        assert isinstance(database_config, ESDatabaseConfig)
        self.client = _create_es_client(hosts=database_config.hosts)
        self.database_config = database_config

    def get_table_index(self, table):
        return self.database_config[table].index

    def get_table_doc_type(self, table):
        return self.database_config[table].doc_type

    def params_add_table(self, table, params):
        assert isinstance(params, dict)
        if self.INDEX in params or self.DOC_TYPE in params:
            raise ValueError(
                'params contains {} or {}'.format(self.INDEX, self.DOC_TYPE))
        table_config = self.database_config[table]
        params[self.INDEX] = table_config.index
        params[self.DOC_TYPE] = table_config.doc_type
        return params

    def bulk_action_to_dict(self, bulk_action):
        assert isinstance(bulk_action, ESBulkAction)
        d = dict(bulk_action.params)
        if bulk_action.table:
            table_config = self.database_config[bulk_action.table]
            d[bulk_action.INDEX] = table_config.index
            d[bulk_action.DOC_TYPE] = table_config.doc_type
        return d

    def search(self, table, **kwargs):
        assert isinstance(table, ESTableSchema)
        return self.client.search(
            index=self.get_table_index(table),
            doc_type=self.get_table_doc_type(table),
            **kwargs
        )

    def index(self, table, body, **kwargs):
        assert isinstance(table, ESTableSchema)
        return self.client.index(
            index=self.get_table_index(table),
            doc_type=self.get_table_doc_type(table),
            body=body,
            **kwargs
        )

    def mget(self, table, body, **kwargs):
        assert isinstance(table, ESTableSchema)
        return self.client.mget(
            index=self.get_table_index(table),
            doc_type=self.get_table_doc_type(table),
            body=body,
            **kwargs
        )

    def bulk_single(self, action):
        assert isinstance(action, ESBulkAction)
        action_dict = self.bulk_action_to_dict(action)
        expanded = elasticsearch.helpers.expand_action(action_dict)
        if expanded[1] is None:
            bulk_actions = (expanded[0],)
        else:
            bulk_actions = expanded

        self.client.bulk(bulk_actions)

    def helper_scan(self, table, **kwargs):
        params = dict(kwargs)
        self.params_add_table(table=table, params=params)
        return elasticsearch.helpers.scan(
            client=self.client,
            **params
        )

    def helper_bulk(self, action_iter, **kwargs):
        return elasticsearch.helpers.bulk(
            client=self.client,
            actions=itertools.imap(self.bulk_action_to_dict, action_iter),
            **kwargs
        )

    def alter_table(self, table, drop_if_exists=False):
        assert isinstance(table, ESTableSchema)

        if self.database_config.es_version == ESVersion.V1:
            mapping_name = table.mapping_v1_name
        elif self.database_config.es_version == ESVersion.V2:
            mapping_name = table.mapping_v2_name
        else:
            raise Exception('invalid es_version: {}'.format(
                self.database_config.es_version))
        mapping = load_mapping(mapping_name)

        cl = self.client.indices
        index = self.get_table_index(table)
        doc_type = self.get_table_doc_type(table)

        if not cl.exists(index=index):
            cl.create(index=index)

        if cl.exists_type(index=index, doc_type=doc_type) and drop_if_exists:
            cl.delete_mapping(index=index, doc_type=doc_type)

        return cl.put_mapping(index=[index], doc_type=doc_type, body=mapping)


class ESOperationType(object):
    HOST_ONLY = 1
    EXTENSIVE = 2


def es_operation_type_seletor(optype):
    def decorator(f):
        name = f.__name__
        if optype == ESOperationType.HOST_ONLY:
            @functools.wraps(f)
            def wrapper(self, *args, **kwargs):
                return getattr(self.default_operator, name)(*args, **kwargs)

            return wrapper
        elif optype == ESOperationType.EXTENSIVE:
            @functools.wraps(f)
            def wrapper(self, *args, **kwargs):
                try:
                    return getattr(self.default_operator, name)(*args, **kwargs)
                finally:
                    return getattr(self.secondary_operator, name)(*args, **kwargs)

            return wrapper
        else:
            raise Exception('invalid operation type: {}'.format(optype))

    return decorator


class ESHostBackupClientManager(ESClientManager):
    def __init__(self, default, secondary):
        assert isinstance(default, ESClientManager)
        assert isinstance(secondary, ESClientManager)
        self.default_operator = default # for host
        self.secondary_operator = secondary # for backup

    @es_operation_type_seletor(ESOperationType.HOST_ONLY)
    def search(self, table, **kwargs):
        raise NotImplementedError

    @es_operation_type_seletor(ESOperationType.EXTENSIVE)
    def index(self, table, body, **kwargs):
        raise NotImplementedError

    @es_operation_type_seletor(ESOperationType.HOST_ONLY)
    def mget(self, table, body, **kwargs):
        raise NotImplementedError

    @es_operation_type_seletor(ESOperationType.EXTENSIVE)
    def bulk_single(self, action):
        raise NotImplementedError

    @es_operation_type_seletor(ESOperationType.HOST_ONLY)
    def helper_scan(self, table, **kwargs):
        raise NotImplementedError

    def alter_table(self, table, drop_if_exists=False):
        raise NotImplementedError('alter_table does not work for ESOperatorCombined')