# -*- coding: utf-8 -*-
import json
import os
import re

import jsonschema
import six
from django.conf import settings


def load_mapping(mapping_name):
    mapping_file_path = os.path.join(
        settings.BASE_DIR,
        'message', 'mapping', '%s.json' % (mapping_name,)
    )
    mapping = ''
    with open(mapping_file_path, 'r') as f:
        for line in f:
            # 去掉注释
            mapping += re.sub(r'//.*$', '', line)
    mapping = json.loads(mapping)
    return mapping


class ESVersion(object):
    V1 = 'v1'
    V2 = 'v2'

class ESTableSchema(object):
    def __init__(self, table_name, mapping_v1_name, mapping_v2_name):
        assert isinstance(table_name, six.string_types)
        assert isinstance(mapping_v1_name, six.string_types)
        self.table_name = table_name
        self.mapping_v1_name = mapping_v1_name
        self.mapping_v2_name = mapping_v2_name

    def __repr__(self):
        return '{}(table_name={})'.format(
            self.__class__.__name__,
            self.table_name,
        )

    def __reduce__(self):
        raise Exception('unserializable')

    def __reduce_ex__(self, *args, **kwargs):
        raise Exception('unserializable')


table_message = ESTableSchema(
    table_name='message',
    mapping_v1_name='message.v1',
    mapping_v2_name='message.v2',
)

table_conversation = ESTableSchema(
    table_name='conversation',
    mapping_v1_name='conversation.v1',
    mapping_v2_name='conversation.v2',
)

table_schema_map = {
    ts.table_name: ts
    for ts in [table_conversation, table_message]
}


_config_schema = {
    '$schema': 'http://json-schema.org/draft-04/schema#',
    'type': 'object',
    'properties': {
        'order': {
            'type': 'array',
            'items': {
                'type': 'string',
                'minItems': 1,
                'maxItems': 2,
                'uniqueItems': True,
            }
        },
        'database': {
            'type': 'array',
            'items': {
                'type': 'object',
                'properties': {
                    'name': {'type': 'string'},
                    'es_version': {'enum': [ESVersion.V1, ESVersion.V2]},
                    'hosts': {'type': 'array'},
                    'table': {
                        'type': 'array',
                        'items': {
                            'type': 'object',
                            'properties': {
                                'name': {
                                    'type': 'string',
                                    'enum': list(table_schema_map),
                                },
                                'index': {'type': 'string'},
                                'doc_type': {'type': 'string'},
                            },
                            'required': ['name', 'index', 'doc_type'],
                        }
                    }
                },
                 'required': ['name', 'es_version', 'hosts', 'table'],
            },
        },
    },
    'required': ['order', 'database'],
}


class ESTableConfig(object):
    def __init__(self, name, index, doc_type):
        assert isinstance(name, six.string_types)
        assert isinstance(index, six.string_types)
        assert isinstance(doc_type, six.string_types)
        self.name = name
        self.index = index
        self.doc_type = doc_type


class ESDatabaseConfig(object):
    def __init__(self, name, es_version, hosts, table_list):
        self.name = name
        self.es_version = es_version
        self.hosts = hosts
        self.table_list = table_list
        self.__table_map = {
            table.name: table
            for table in table_list
        }

    def __getitem__(self, item):
        assert isinstance(item, ESTableSchema)
        return self.__table_map[item.table_name]


class Config(object):
    def __init__(self, config_data):
        jsonschema.validate(config_data, _config_schema)
        self.config_data = config_data

        order = config_data['order']
        assert isinstance(order, list)
        self.order = order

        database_list = []
        for database_config in config_data['database']:
            table_list = []
            for table_config in database_config['table']:
                table_list.append(ESTableConfig(
                    name=table_config['name'],
                    index=table_config['index'],
                    doc_type=table_config['doc_type'],
                ))
            database_list.append(ESDatabaseConfig(
                name=database_config['name'],
                es_version=database_config['es_version'],
                hosts=database_config['hosts'],
                table_list=table_list,
            ))
        self.database_list = database_list
        self.__database_map = {
            database.name: database
            for database in database_list
        }

    def get_database_config(self, name):
        return self.__database_map[name]


config = Config(settings.ES_MSG)

