# -*- coding: utf-8 -*-
import json
import logging
import os
import traceback
import uuid

from django.conf import settings
from gm_logging.internal.instance import request_logging_guard_maker
from gm_rpcd.internals.common import now_str

from gm_rpcd.internals.configuration.model import Config, environ, \
    from_property
from gm_rpcd.internals.configuration.model_base import literal, xml_text, xml_text_list, \
    DefaultConfigPropertyWrapper, EnvironmentConfigProperty
from gm_rpcd.internals.context import Context
from gm_rpcd.internals.dispatcher import json_encode_with_length
from gm_rpcd.internals.dynamic_scopes import dynamic_scope, Keys
from gm_rpcd.internals.protocol.request import Request, \
    dump_request_to_json_value
from gm_rpcd.internals.protocol.response import FaultResponse, SuccessResponse, \
    response_to_v1_json_value, SystemErrorResponse
from gm_rpcd.internals.protocol.response_extras import \
    make_method_not_found_response
from gm_rpcd.internals.utils import strict_check_json_value
from gm_tracer.context import current_tracer

from rpc_framework.exceptions import RPCViewBaseException

DISPATCHER = None

GM_RPCD_APP_CONF_PATH_KEY = 'GM_RPCD_APP_CONF_PATH'
GM_RPCD_DEVELOP_CONF_PATH_KEY = 'GM_RPCD_DEVELOP_CONF_PATH'
GM_RPCD_DEPLOY_CONF_PATH_KEY = 'GM_RPCD_DEPLOY_CONF_PATH'
GM_RPCD_MODE_KEY = 'GM_RPCD_MODE'


class DjangoRpcdConfig(Config):
    '''因为 最终会从实例的wrapped属性解析出配置， 而wrapped实际上读取的是Config实例
    的`.__class__.__dict__.`,因此父类中的类成员无法继承, 所以父类中的类成员都要重写'''
    process_unique_id = literal(uuid.uuid4().hex)

    app_conf_path = DefaultConfigPropertyWrapper(gm_rpcd_app_path) \
        if (gm_rpcd_app_path := getattr(settings, GM_RPCD_APP_CONF_PATH_KEY, None)) \
        else environ('GM_RPCD_APP_CONF_PATH').default(os.path.abspath('app_conf.xml'))

    develop_conf_path = DefaultConfigPropertyWrapper(gm_develop_conf_path) \
            if (gm_develop_conf_path := getattr(settings, GM_RPCD_DEVELOP_CONF_PATH_KEY, None)) \
            else environ('GM_RPCD_DEVELOP_CONF_PATH').default( os.path.abspath('.gm_rpcd.develop_conf.xml'))

    deploy_conf_path = EnvironmentConfigProperty(gm_deploy_conf_path) \
            if (gm_deploy_conf_path := getattr(settings, GM_RPCD_DEPLOY_CONF_PATH_KEY, None)) \
            else environ('GM_RPCD_DEPLOY_CONF_PATH')

    # mode is instance of  `ConfigProperty('develop')` or `ConfigProperty('deploy')`
    mode = EnvironmentConfigProperty(gm_rpcd_mode) \
            if (gm_rpcd_mode := getattr(settings, GM_RPCD_MODE_KEY, None)) \
            else environ('GM_RPCD_MODE')
    is_develop_mode, is_deploy_mode  = mode.eq('develop'), mode.eq('deploy')

    application_name = xml_text('app', 'application_name')
    service_list = xml_text_list('app', 'service_list').default_func(list)
    initializer_list = xml_text_list('app', 'initializer_list').default_func(list)
    statuses_func = xml_text('app', 'statuses')

    # give app a chance for using customized request info extractor, if not
    # set, use default one GMRpcdRequestInfoExtractor
    request_info_extractor_cls = xml_text('app', 'request_info_extractor').default(None)

    log_dir_for_develop = xml_text('develop', 'log_dir').\
        default(os.path.abspath('.gm_rpcd.log'))
    log_dir_for_deploy = xml_text('deploy', 'log_dir')

    sentry_dsn_for_deploy = xml_text('deploy', 'sentry_dsn')

    log_encoding = literal('utf-8')

    @from_property
    @property
    def log_dir(self):
        if self.is_develop_mode:
            return self.log_dir_for_develop
        if self.is_deploy_mode:
            return self.log_dir_for_deploy
        raise AttributeError('invalid mode')

    @from_property
    @property
    def local_request_log_path(self):
        return os.path.join(self.log_dir, 'gm_rpcd_request.log')

    @from_property
    @property
    def local_request_error_log_path(self):
        return os.path.join(self.log_dir, 'gm_rpcd_request_error.log')


def process_single_request(self, request):
    assert isinstance(request, Request)
    tracer = current_tracer()
    method_function = self._method_table.get_method_function(request.method)
    if not method_function:
        return make_method_not_found_response(request)

    request_json_value = dump_request_to_json_value(request)
    request_json_str = json.dumps(request_json_value, ensure_ascii=False)

    now = now_str()
    request_info = self._request_info_extractor(request)

    log_id = request_info.log_id
    span_id = getattr(request_info, 'span_id', '')
    parent_span_id = getattr(request_info, 'parent_span_id', '')
    gm_request_id = getattr(request_info, 'gm_request_id', '')
    log_locating_info = 'log_id={} span_id={} parent_span_id={}'.format(
        log_id, span_id, parent_span_id
    )

    with request_logging_guard_maker(request_info) as guard:
        context = Context(
            request=request,
            request_info=request_info,
            logger=guard.logger,
            helios_base_invoker=self._helios_base_invoker,
        )
        with dynamic_scope.set(Keys.CONTEXT, context), dynamic_scope.set(Keys.REQUEST_INFO, request_info):
            try:
                result = method_function(**request.params)
                strict_check_json_value(result)
                response = SuccessResponse(
                    request_id=request.request_id,
                    result=result,
                )
            except RPCViewBaseException as e:
                guard.set_errno(e.code)
                response = FaultResponse(
                    request_id=request.request_id,
                    error={
                        # 最终相应为： {'error': '..','message': '..', 'data': ''}
                        'code': e.code,
                        'message': e.message,
                    }
                )
            except Exception as e:
                guard.set_errno_unexpected()

                if self._raven_client:
                    self._raven_client.captureException(
                        tags={
                            'log_id': log_id,
                            'gm_request_id': gm_request_id,
                            'span_id': span_id,
                            'parent_span_id': parent_span_id,
                        },
                        extra={
                            'log_id': log_id,
                            'span_id': span_id,
                            'parent_span_id': parent_span_id,
                            'request': request_json_value,
                            'gm_request_id': gm_request_id,
                        }
                    )


                format_exc = traceback.format_exc()
                log_message = json.dumps({
                    'timestamp': now,
                    'gm_request_id': gm_request_id,
                    'log_locating_info': log_locating_info,
                    'format_exc': format_exc,
                    'request_json_str': request_json_str,
                }, ensure_ascii=False)
                self._request_error_logger.error(log_message)

                from gm_rpcd.internals.configuration.model import config
                if config.is_develop_mode:
                    debug_message = '{}\nrequest = {}'.format(
                        format_exc,
                        request_json_str,
                    )
                else:
                    debug_message = None
                response = SystemErrorResponse(
                    message='Unexpected exception on server side. {}'.format(log_locating_info),
                    debug_message=debug_message,
                )
                if tracer:
                    # tracer.add_annotation({'error': time.time()})
                    tracer.add_binary_annotation({
                        'error': 'true',
                        'exc.info': format_exc,
                        'request_json_str': request_json_str,
                    })
    self._request_logger.info(json.dumps({
        'timestamp': now,
        'gm_request_id': gm_request_id,
        'log_locating_info': log_locating_info,
        'request_json_value': json_encode_with_length(request_json_value),
        'response_json_value': json_encode_with_length(response_to_v1_json_value(response)),
    }, ensure_ascii=False))
    return response


def setup_rpcd():
    logger = logging.getLogger('django.start')
    logger.info('****** setup rpcd ******')
    # call `initialize_app` of rpcd
    config_wrapper = DjangoRpcdConfig().wrapped
    config_wrapper.is_deploy_mode = True
    config_wrapper.freeze()

    ######################## PATCH #############################
    from gm_rpcd.internals.configuration import model
    model.__dict__['config'] = config_wrapper

    from gm_rpcd.internals import configuration
    configuration.__dict__['config'] = config_wrapper

    from gm_rpcd.internals.dispatcher import Dispatcher
    Dispatcher.process_single_request = process_single_request
    ####################### end PATCH ##########################

    from gm_rpcd.internals.initializations import initialize
    # expose to module scope to make dispatcher singleton
    global DISPATCHER
    if not DISPATCHER:
        DISPATCHER = initialize().dispatcher

    # avoid call `setup_rpcd` repeatly
    global setup_rpcd
    del setup_rpcd


setup_rpcd()


__all__ = [
    'DISPATCHER',
    'views'
]